﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/htclow/htclow_ResultPrivate.h>
#include <nn/htclow/detail/htclow_Log.h>

#include "htclow_Listener.h"

namespace nn { namespace htclow { namespace server {

Listener::Listener(Worker* pWorker) NN_NOEXCEPT
    : m_ConnectEvent(nn::os::EventClearMode_ManualClear, true)
    , m_DisconnectEvent(nn::os::EventClearMode_ManualClear, true)
    , m_pDriver(nullptr)
    , m_pWorker(pWorker)
    , m_ThreadRunning(false)
    , m_Cancelled(false)

{
    m_pListenThreadStack = AllocateThreadStack(ThreadStackSize);
    SetEvent(false);
}

Listener::~Listener() NN_NOEXCEPT
{
    FreeThreadStack(m_pListenThreadStack);
}

Result Listener::Start(driver::IDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_ThreadRunning);

    NN_RESULT_DO(nn::os::CreateThread(&m_ListenThread, ListenThreadEntry, this, m_pListenThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(htc, ListenThread)));
    nn::os::SetThreadNamePointer(&m_ListenThread, NN_SYSTEM_THREAD_NAME(htc, ListenThread));

    m_Cancelled = false;

    m_ThreadRunning = true;
    m_pDriver = pDriver;

    nn::os::StartThread(&m_ListenThread);

    NN_RESULT_SUCCESS;
}

void Listener::Cancel() NN_NOEXCEPT
{
    m_Cancelled = true;
    m_pWorker->Cancel();
}

void Listener::Wait() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_ThreadRunning);

    nn::os::WaitThread(&m_ListenThread);
    nn::os::DestroyThread(&m_ListenThread);

    m_pDriver = nullptr;
    m_ThreadRunning = false;
}

void Listener::SetEvent(bool connected) NN_NOEXCEPT
{
    if (connected)
    {
        m_DisconnectEvent.Clear();
        m_ConnectEvent.Signal();
    }
    else
    {
        m_ConnectEvent.Clear();
        m_DisconnectEvent.Signal();
    }
}

void Listener::ListenThreadEntry(void* pThis) NN_NOEXCEPT
{
    reinterpret_cast<Listener*>(pThis)->ListenThread();
}

void Listener::ListenThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pDriver);
    NN_SDK_ASSERT_NOT_NULL(m_pWorker);

    m_pWorker->Initialize(m_pDriver);

    nn::Result result;

    while (!m_Cancelled)
    {
        result = m_pDriver->Connect();
        if (result.IsFailure())
        {
            break;
        }

        SetEvent(true);

        result = m_pWorker->Start();
        if (result.IsFailure())
        {
            break;
        }

        m_pWorker->Wait();
        m_pDriver->Shutdown();

        SetEvent(false);
    }

    NN_DETAIL_HTCLOW_INFO("Listen thread finished with result (module:%d, description:%d).\n", result.GetModule(), result.GetDescription());
}

}}}
