﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/os.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_InternalTypes.h>
#include <nn/htclow/htclow_ResultPrivate.h>

#include "htclow_Allocator.h"
#include "htclow_Listener.h"
#include "htclow_Worker.h"

#include "driver/htclow_DriverManager.h"
#include "mux/htclow_Mux.h"

namespace nn { namespace htclow { namespace server {

// htclow サーバライブラリの主要なオブジェクトを保持するシングルトン
class HtclowImpl
{
    NN_DISALLOW_COPY(HtclowImpl);
    NN_DISALLOW_MOVE(HtclowImpl);

public:
    static HtclowImpl* GetInstance() NN_NOEXCEPT;

    driver::DriverManager* GetDriverManager() NN_NOEXCEPT
    {
        return &m_DriverManager;
    }

    nn::Result Open(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.Open(channel);
    }

    nn::Result Close(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        m_Mux.Close(channel);
        NN_RESULT_SUCCESS;
    }

    nn::Result Connect(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.Connect(channel);
    }

    nn::Result Shutdown(detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.Shutdown(channel);
    }

    nn::Result Send(int* pOutSize, const void* pBuffer, int bufferSize, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.Send(pOutSize, pBuffer, bufferSize, channel);
    }

    nn::Result Receive(int* pOutSize, void* pOutBuffer, int bufferSize, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.Receive(pOutSize, pOutBuffer, bufferSize, channel);
    }

    nn::Result GetClosedEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.GetClosedEventHandle(pOutHandle, channel);
    }

    nn::Result GetEstablishedEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.GetEstablishedEventHandle(pOutHandle, channel);
    }

    nn::Result GetReceiveReadyEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.GetReceiveReadyEventHandle(pOutHandle, channel);
    }

    nn::Result GetSendReadyEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.GetSendReadyEventHandle(pOutHandle, channel);
    }

    nn::Result GetSendCompleteEventHandle(nn::os::NativeHandle* pOutHandle, detail::ChannelInternalType channel) NN_NOEXCEPT
    {
        return m_Mux.GetSendCompleteEventHandle(pOutHandle, channel);
    }

    nn::os::SystemEvent* GetConnectEvent()
    {
        return m_Listener.GetConnectEvent();
    }

    nn::os::SystemEvent* GetDisconnectEvent()
    {
        return m_Listener.GetDisconnectEvent();
    }

    nn::Result OpenDriver(detail::DriverType driverType) NN_NOEXCEPT
    {
        NN_RESULT_DO(m_DriverManager.OpenDriver(driverType));
        m_Mux.SetRetransmitEnabled(driverType != detail::DriverType::Debug);

        m_Listener.Start(m_DriverManager.GetCurrentDriver());

        NN_RESULT_SUCCESS;
    }

    void CloseDriver() NN_NOEXCEPT
    {
        m_DriverManager.Cancel();
        m_Mux.CancelAllChannel();
        m_Listener.Cancel();

        m_Listener.Wait();

        m_DriverManager.CloseDriver();
    }

    size_t GetDefaultAllocatorUsage() NN_NOEXCEPT
    {
        return nn::htclow::server::GetDefaultAllocatorUsage();
    }

    void PrintDefaultAllocatorUsage() NN_NOEXCEPT
    {
        nn::htclow::server::PrintDefaultAllocatorUsage();
    }

private:
    HtclowImpl() NN_NOEXCEPT
        : m_Listener(&m_Worker)
        , m_Worker(&m_Mux)
    {
    }

    driver::DriverManager m_DriverManager;
    mux::Mux m_Mux;
    Listener m_Listener;
    Worker m_Worker;
};

}}}
