﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   GPIO ドライバの API インタフェース部分。
 * @details  割り込みスレッド用のスタックリソースの定義を含む。
 */

#include <nn/nn_DeviceCode.h>

#include <nn/gpio/detail/gpio_Log.h>
#include <nn/gpio/driver/gpio_Lib.h>
#include <nn/gpio/driver/gpio_PadAccessor.h>
#include <nn/gpio/driver/gpio_PadAccessorDev.h>
#include <nn/gpio/driver/gpio_InitialConfig.h>
#include <nn/gpio/gpio_PadMap.h>
#include <nn/gpio/gpio_Result.h>

#include "detail/gpio_Driver.h"

namespace {

nn::gpio::driver::detail::Driver s_GpioDriver;

}

namespace nn {
namespace gpio {
namespace driver {


void Initialize() NN_NOEXCEPT
{
    s_GpioDriver.Initialize();
}

void Finalize() NN_NOEXCEPT
{
    s_GpioDriver.Finalize();
}

nn::Result OpenSession(GpioPadSession* pOutSession, nn::DeviceCode pad) NN_NOEXCEPT
{
    NN_UNUSED(pOutSession);
    NN_UNUSED(pad);
    NN_DETAIL_GPIO_FATAL("Driver generation mismatch: Gen2 API called for Gen1 server.\n");
    NN_ABORT();
    return nn::gpio::ResultDeviceNotFound(); // NEVER REACHED HERE
}

void OpenSession(GpioPadSession* pOutSession, GpioPadName pad) NN_NOEXCEPT
{
    s_GpioDriver.OpenSession(pOutSession, pad);
}

void OpenSessionForDev(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT
{
    s_GpioDriver.OpenSessionDev(pOutSession, padNumber);
}

void CloseSession(GpioPadSession* pSession) NN_NOEXCEPT
{
    s_GpioDriver.CloseSession(pSession);
}

void SetDirection(GpioPadSession* pSession, Direction direction) NN_NOEXCEPT
{
    s_GpioDriver.SetDirection(*pSession, direction );
}

Direction GetDirection(GpioPadSession* pSession ) NN_NOEXCEPT
{
    return s_GpioDriver.GetDirection(*pSession);
}

void SetInterruptMode(GpioPadSession* pSession, InterruptMode mode ) NN_NOEXCEPT
{
    s_GpioDriver.SetInterruptMode(*pSession, mode);
}

InterruptMode GetInterruptMode(GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.GetInterruptMode(*pSession);
}

void SetInterruptEnable(GpioPadSession* pSession, bool enable ) NN_NOEXCEPT
{
    s_GpioDriver.SetInterruptEnable(pSession, enable);
}

bool GetInterruptEnable(GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.GetInterruptEnable(*pSession);
}

InterruptStatus GetInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.GetInterruptStatus(*pSession);
}

void ClearInterruptStatus(GpioPadSession* pSession) NN_NOEXCEPT
{
    s_GpioDriver.ClearInterruptStatus(*pSession);
}

GpioValue GetValue(GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.GetValue(*pSession);
}

void SetValue(GpioPadSession* pSession, GpioValue value ) NN_NOEXCEPT
{
    s_GpioDriver.SetValue(*pSession, value);
}

void SetValueForSleepState(GpioPadSession* pSession, GpioValue value ) NN_NOEXCEPT
{
    s_GpioDriver.SetValueForSleepState(*pSession, value);
}

void SetDebounceEnabled(GpioPadSession* pSession, bool isEnable) NN_NOEXCEPT
{
    s_GpioDriver.SetDebounceEnabled(pSession, isEnable);
}

bool GetDebounceEnabled(GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.GetDebounceEnabled(pSession);
}

void SetDebounceTime(GpioPadSession* pSession, int msecTime) NN_NOEXCEPT
{
    s_GpioDriver.SetDebounceTime(pSession, msecTime);
}

int GetDebounceTime(GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.GetDebounceTime(pSession);
}

Result BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT
{
    return s_GpioDriver.BindInterrupt(pEvent, pSession);
}

void UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT
{
    s_GpioDriver.UnbindInterrupt(pSession);
}

nn::Result IsWakeEventActive(bool* pOutIsActive, nn::DeviceCode name) NN_NOEXCEPT
{
    NN_UNUSED(pOutIsActive);
    NN_UNUSED(name);
    NN_DETAIL_GPIO_FATAL("Driver generation mismatch: Gen2 API called for Gen1 server.");
    NN_ABORT();
    return nn::gpio::ResultDeviceNotFound(); // NEVER REACHED HERE
}

bool IsWakeEventActive(GpioPadName name) NN_NOEXCEPT
{
    return s_GpioDriver.IsWakeEventActive(name);
}

void Suspend() NN_NOEXCEPT
{
    s_GpioDriver.Suspend();
}

void SuspendLow() NN_NOEXCEPT
{
    s_GpioDriver.SuspendLow();
}

void Resume() NN_NOEXCEPT
{
    s_GpioDriver.Resume();
}

void ResumeLow() NN_NOEXCEPT
{
    s_GpioDriver.ResumeLow();
}

void SetInitialGpioConfig() NN_NOEXCEPT
{
    s_GpioDriver.SetInitialGpioConfig();
}

void SetInitialWakePinConfig() NN_NOEXCEPT
{
    s_GpioDriver.SetInitialWakePinConfig();
}

WakeBitFlag GetWakeEventActiveFlagSet() NN_NOEXCEPT
{
    return s_GpioDriver.GetWakeEventActiveFlagSet();
}

nn::Result SetWakeEventActiveFlagSetForDebug(nn::DeviceCode name, bool isEnabled) NN_NOEXCEPT
{
    NN_UNUSED(name);
    NN_UNUSED(isEnabled);
    NN_DETAIL_GPIO_FATAL("Driver generation mismatch: Gen2 API called for Gen1 server.");
    NN_ABORT();
    return nn::gpio::ResultDeviceNotFound(); // NEVER REACHED HERE
}

void SetWakeEventActiveFlagSetForDebug(GpioPadName name, bool isEnabled) NN_NOEXCEPT
{
    s_GpioDriver.SetWakeEventActiveFlagSetForDebug(name, isEnabled);
}

void SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT
{
    s_GpioDriver.SetWakePinDebugMode(mode);
}

} // driver
} // gpio
} // nn
