﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   NXFP2 用 サポートパッド一覧
 * @detail  本ヘッダーファイルは NXFP2 向けにサポートしているパッドの一覧です。
 *          本ファイルと公開ヘッダーである、gpio_PadMap.NX.h へ対応したいパッドを追記することで
 *          GPIO ライブラリで対応するパッドの追加、および外部の定義と内部の番号の紐付の変更が可能です。
 */

#pragma once

#include <nn/nn_Abort.h>

#include <nn/gpio/gpio_PadMap.h>

#include <nne/wec/wec.h>

namespace nn{
namespace gpio{
namespace driver{
namespace detail{

/**
* @brief Tegra 向けの GPIO パッド番号計算用マクロ
*
* @details Tegra 向けの GPIO パッド番号計算用マクロです。
*/
#define NN_GPIO_GET_PAD_NUM(port, bit) ((port) * 8 + (bit))


/**
* @brief Tegra 向けの GPIO ポート一覧
*
* @details Tegra 向けの GPIO ポート一覧です。GPIO 番号計算用マクロを使用し、GPIO パッド番号を算出するために使用します。
*/
enum GpioPadPort
{
    GpioPadPort_PA = 0,
    GpioPadPort_PB,
    GpioPadPort_PC,
    GpioPadPort_PD,

    GpioPadPort_PE = 4,
    GpioPadPort_PF,
    GpioPadPort_PG,
    GpioPadPort_PH,

    GpioPadPort_PI = 8,
    GpioPadPort_PJ,
    GpioPadPort_PK,
    GpioPadPort_PL,

    GpioPadPort_PM = 12,
    GpioPadPort_PN,
    GpioPadPort_PO,
    GpioPadPort_PP,

    GpioPadPort_PQ = 16,
    GpioPadPort_PR,
    GpioPadPort_PS,
    GpioPadPort_PT,

    GpioPadPort_PU = 20,
    GpioPadPort_PV,
    GpioPadPort_PW,
    GpioPadPort_PX,

    GpioPadPort_PY = 24,
    GpioPadPort_PZ,
    GpioPadPort_PAA,
    GpioPadPort_PBB,

    GpioPadPort_PCC = 28,
    GpioPadPort_PDD,
    GpioPadPort_PEE,
    GpioPadPort_PFF,

    GpioPadPort_Num,
};

// サポートしている内部で使用するレジスタアクセス用 Gpio 番号の列挙体
// サポートするパッドが増える場合にはここにレジスタへのアクセス用の番号を追加してください。
enum InternalGpioPadNumber
{
    InternalGpioPadNumber_GPIO_PBB1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PBB, 1),
    InternalGpioPadNumber_GPIO_PBB2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PBB, 2),
    InternalGpioPadNumber_GPIO_PBB3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PBB, 3),
    InternalGpioPadNumber_GPIO_PBB4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PBB, 4),
    InternalGpioPadNumber_GPIO_PB4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PB, 4 ),
    InternalGpioPadNumber_GPIO_PB5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PB, 5 ),
    InternalGpioPadNumber_GPIO_PC0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PC, 0 ),
    InternalGpioPadNumber_GPIO_PC1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PC, 1 ),
    InternalGpioPadNumber_GPIO_PC2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PC, 2 ),
    InternalGpioPadNumber_GPIO_PD1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PD, 1 ),
    InternalGpioPadNumber_GPIO_PD4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PD, 4 ),
    InternalGpioPadNumber_GPIO_PE0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 0 ),
    InternalGpioPadNumber_GPIO_PE1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 1 ),
    InternalGpioPadNumber_GPIO_PE2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 2 ),
    InternalGpioPadNumber_GPIO_PE3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 3 ),
    InternalGpioPadNumber_GPIO_PE4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 4 ),
    InternalGpioPadNumber_GPIO_PE5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 5 ),
    InternalGpioPadNumber_GPIO_PE6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 6 ),
    InternalGpioPadNumber_GPIO_PE7 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PE, 7 ),
    InternalGpioPadNumber_GPIO_PG0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PG, 0 ),
    InternalGpioPadNumber_GPIO_PG3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PG, 3 ),
    InternalGpioPadNumber_GPIO_PS0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 0 ),
    InternalGpioPadNumber_GPIO_PS1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 1 ),
    InternalGpioPadNumber_GPIO_PS2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 2 ),
    InternalGpioPadNumber_GPIO_PS3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 3 ),
    InternalGpioPadNumber_GPIO_PS4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 4 ),
    InternalGpioPadNumber_GPIO_PS5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 5 ),
    InternalGpioPadNumber_GPIO_PS6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 6 ),
    InternalGpioPadNumber_GPIO_PS7 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PS, 7 ),
    InternalGpioPadNumber_GPIO_PT0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PT, 0 ),
    InternalGpioPadNumber_GPIO_PT1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PT, 1 ),
    InternalGpioPadNumber_GPIO_PA5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PA, 5 ),
    InternalGpioPadNumber_GPIO_PP0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PP, 0 ),
    InternalGpioPadNumber_GPIO_PP1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PP, 1 ),
    InternalGpioPadNumber_GPIO_PP5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PP, 5 ),
    InternalGpioPadNumber_GPIO_PP4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PP, 4 ),
    InternalGpioPadNumber_GPIO_PP3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PP, 3 ),
    InternalGpioPadNumber_GPIO_PP2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PP, 2 ),
    InternalGpioPadNumber_GPIO_PX4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 4 ),
    InternalGpioPadNumber_GPIO_PV6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PV, 6 ),
    InternalGpioPadNumber_GPIO_PX1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 1 ),
    InternalGpioPadNumber_GPIO_PX2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 2 ),
    InternalGpioPadNumber_GPIO_PX3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 3 ),
    InternalGpioPadNumber_GPIO_PX5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 5 ),
    InternalGpioPadNumber_GPIO_PX6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 6 ),
    InternalGpioPadNumber_GPIO_PX7 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PX, 7 ),
    InternalGpioPadNumber_GPIO_PY0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PY, 0 ),
    InternalGpioPadNumber_GPIO_PY1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PY, 1 ),
    InternalGpioPadNumber_GPIO_PV1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PV, 1 ),
    InternalGpioPadNumber_GPIO_PV2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PV, 2 ),
    InternalGpioPadNumber_GPIO_PV4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PV, 4 ),
    InternalGpioPadNumber_GPIO_PV5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PV, 5 ),
    InternalGpioPadNumber_GPIO_PZ0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PZ, 0 ),
    InternalGpioPadNumber_GPIO_PZ1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PZ, 1 ),
    InternalGpioPadNumber_GPIO_PZ2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PZ, 2 ),
    InternalGpioPadNumber_GPIO_PZ3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PZ, 3 ),
    InternalGpioPadNumber_GPIO_PZ4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PZ, 4 ),
    InternalGpioPadNumber_GPIO_PJ4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PJ, 4 ),
    InternalGpioPadNumber_GPIO_PJ5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PJ, 5 ),
    InternalGpioPadNumber_GPIO_PJ7 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PJ, 7 ),
    InternalGpioPadNumber_GPIO_PK0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 0 ),
    InternalGpioPadNumber_GPIO_PK1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 1 ),
    InternalGpioPadNumber_GPIO_PK2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 2 ),
    InternalGpioPadNumber_GPIO_PK3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 3 ),
    InternalGpioPadNumber_GPIO_PK4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 4 ),
    InternalGpioPadNumber_GPIO_PK5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 5 ),
    InternalGpioPadNumber_GPIO_PK6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 6 ),
    InternalGpioPadNumber_GPIO_PK7 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PK, 7 ),
    InternalGpioPadNumber_GPIO_PL0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PL, 0 ),
    InternalGpioPadNumber_GPIO_PH0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 0 ),
    InternalGpioPadNumber_GPIO_PH1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 1 ),
    InternalGpioPadNumber_GPIO_PH2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 2 ),
    InternalGpioPadNumber_GPIO_PH3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 3 ),
    InternalGpioPadNumber_GPIO_PH4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 4 ),
    InternalGpioPadNumber_GPIO_PH5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 5 ),
    InternalGpioPadNumber_GPIO_PH6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 6 ),
    InternalGpioPadNumber_GPIO_PH7 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PH, 7 ),
    InternalGpioPadNumber_GPIO_PI0 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PI, 0 ),
    InternalGpioPadNumber_GPIO_PI1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PI, 1 ),
    InternalGpioPadNumber_GPIO_PI2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PI, 2 ),
    InternalGpioPadNumber_GPIO_PCC1 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PCC, 1 ),
    InternalGpioPadNumber_GPIO_PCC2 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PCC, 2 ),
    InternalGpioPadNumber_GPIO_PCC3 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PCC, 3 ),
    InternalGpioPadNumber_GPIO_PCC4 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PCC, 4 ),
    InternalGpioPadNumber_GPIO_PCC5 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PCC, 5 ),
    InternalGpioPadNumber_GPIO_PCC6 = NN_GPIO_GET_PAD_NUM(GpioPadPort_PCC, 6 ),

    InternalGpioPadNumber_None       = -1,
};

// 外部の定義と内部のレジスタへのアクセス用番号の組み合わせの構造体
struct PadMapCombination
{
    GpioPadName              publicPadName;
    InternalGpioPadNumber    internalPadName;
    nne::wec::WakeEvent      wakeEvent;
};

const nne::wec::WakeEvent WakeEventNone = static_cast<nne::wec::WakeEvent>(-1);

// 外部の定義と内部のレジスタへのアクセス用番号への変換一覧
// サポートするパッドの物理的なピン番号が変更になった場合や新しいパッドをサポートすることになった場合は
// ここに追加してください。
const PadMapCombination PadMapCombinationList[] =
{
    // 公開ヘッダーの定義              内部のレジスタアクセス用番号
    { GpioPadName_RamCode3           , InternalGpioPadNumber_GPIO_PBB2,    WakeEventNone },
    { GpioPadName_GameCardReset      , InternalGpioPadNumber_GPIO_PBB3,    WakeEventNone },
    { GpioPadName_CodecAlert         , InternalGpioPadNumber_GPIO_PBB4,    WakeEventNone },
    { GpioPadName_PowSdEn            , InternalGpioPadNumber_GPIO_PE4,     WakeEventNone },
    { GpioPadName_PowGc              , InternalGpioPadNumber_GPIO_PE5,     WakeEventNone },
    { GpioPadName_DebugControllerDet , InternalGpioPadNumber_GPIO_PS0,     WakeEventNone },
    { GpioPadName_BattChgStatus      , InternalGpioPadNumber_GPIO_PS1,     WakeEventNone },
    { GpioPadName_BattChgEnableN     , InternalGpioPadNumber_GPIO_PS6,     WakeEventNone },
    { GpioPadName_FanTach            , InternalGpioPadNumber_GPIO_PS7,     WakeEventNone },
    { GpioPadName_Vdd50AEn           , InternalGpioPadNumber_GPIO_PA5,     WakeEventNone },
    { GpioPadName_SdevCoaxSel1       , InternalGpioPadNumber_GPIO_PP0,     WakeEventNone },
    { GpioPadName_GameCardCd         , InternalGpioPadNumber_GPIO_PS3,     nne::wec::WakeEvent_CAM_I2C_SDA },
    { GpioPadName_ProdType0          , InternalGpioPadNumber_GPIO_PP5,     WakeEventNone },
    { GpioPadName_ProdType1          , InternalGpioPadNumber_GPIO_PP4,     WakeEventNone },
    { GpioPadName_ProdType2          , InternalGpioPadNumber_GPIO_PP3,     WakeEventNone },
    { GpioPadName_ProdType3          , InternalGpioPadNumber_GPIO_PP2,     WakeEventNone },
    { GpioPadName_TempAlert          , InternalGpioPadNumber_GPIO_PX4,     WakeEventNone },
    { GpioPadName_CodecHpDetIrq      , InternalGpioPadNumber_GPIO_PV6,     WakeEventNone },
    { GpioPadName_TpIrq              , InternalGpioPadNumber_GPIO_PX1,     WakeEventNone },
    { GpioPadName_ButtonSleep2       , InternalGpioPadNumber_GPIO_PX5,     WakeEventNone },
    { GpioPadName_ButtonVolUp        , InternalGpioPadNumber_GPIO_PX6,     WakeEventNone },
    { GpioPadName_ButtonVolDn        , InternalGpioPadNumber_GPIO_PX7,     WakeEventNone },
    { GpioPadName_BattMgicIrq        , InternalGpioPadNumber_GPIO_PY0,     nne::wec::WakeEvent_BUTTON_SLIDE_SW },
    { GpioPadName_RecoveryKey        , InternalGpioPadNumber_GPIO_PY1,     WakeEventNone },
    { GpioPadName_PowLcdBlEn         , InternalGpioPadNumber_GPIO_PV1,     WakeEventNone },
    { GpioPadName_LcdReset           , InternalGpioPadNumber_GPIO_PV2,     WakeEventNone },
    { GpioPadName_PdRstN             , InternalGpioPadNumber_GPIO_PV5,     WakeEventNone },
    { GpioPadName_Bq24190Irq         , InternalGpioPadNumber_GPIO_PZ0,     nne::wec::WakeEvent_GPIO_PZ0 },
    { GpioPadName_SdevCoaxSel0       , InternalGpioPadNumber_GPIO_PZ2,     WakeEventNone },
    { GpioPadName_SdWp               , InternalGpioPadNumber_GPIO_PZ3,     WakeEventNone },
    { GpioPadName_CodecLdoEnTemp     , InternalGpioPadNumber_GPIO_PZ4,     WakeEventNone },
    { GpioPadName_TpReset            , InternalGpioPadNumber_GPIO_PJ7,     WakeEventNone },
    { GpioPadName_BtGpio2            , InternalGpioPadNumber_GPIO_PK0,     WakeEventNone },
    { GpioPadName_BtGpio3            , InternalGpioPadNumber_GPIO_PK1,     WakeEventNone },
    { GpioPadName_BtGpio4            , InternalGpioPadNumber_GPIO_PK2,     WakeEventNone },
    { GpioPadName_CradleIrq          , InternalGpioPadNumber_GPIO_PK4,     nne::wec::WakeEvent_GPIO_PK4 },
    { GpioPadName_PowVcpuInt         , InternalGpioPadNumber_GPIO_PK6,     WakeEventNone },
    { GpioPadName_Max77621GpuInt     , InternalGpioPadNumber_GPIO_PK7,     WakeEventNone },
    { GpioPadName_WifiRfDisable      , InternalGpioPadNumber_GPIO_PH0,     WakeEventNone },
    { GpioPadName_WifiReset          , InternalGpioPadNumber_GPIO_PH1,     WakeEventNone },
    { GpioPadName_ApWakeBt           , InternalGpioPadNumber_GPIO_PH3,     WakeEventNone },
    { GpioPadName_BtRst              , InternalGpioPadNumber_GPIO_PH4,     WakeEventNone },
    { GpioPadName_BtWakeAp           , InternalGpioPadNumber_GPIO_PH5,     nne::wec::WakeEvent_BT_WAKE_AP},
    { GpioPadName_BtGpio5            , InternalGpioPadNumber_GPIO_PH7,     WakeEventNone },
    { GpioPadName_PowLcdVddPEn       , InternalGpioPadNumber_GPIO_PI0,     WakeEventNone },
    { GpioPadName_PowLcdVddNEn       , InternalGpioPadNumber_GPIO_PI1,     WakeEventNone },
    { GpioPadName_RamCode2           , InternalGpioPadNumber_GPIO_PCC2,    WakeEventNone },
    { GpioPadName_Vdd50BEn           , InternalGpioPadNumber_GPIO_PCC4,    WakeEventNone },
    { GpioPadName_ExtconDetS         , InternalGpioPadNumber_GPIO_PE6,     nne::wec::WakeEvent_GPIO_PE6 }, // PD1 へ変更予定
    { GpioPadName_ExtconDetU         , InternalGpioPadNumber_GPIO_PH6,     nne::wec::WakeEvent_GPIO_PH6 }, // PG0 へ変更予定
    { GpioPadName_ExtconChgU         , InternalGpioPadNumber_GPIO_PK3,     WakeEventNone },
    { GpioPadName_ExtconChgS         , InternalGpioPadNumber_GPIO_PCC3,    WakeEventNone },
    { GpioPadName_MotionInt          , InternalGpioPadNumber_GPIO_PX2,     WakeEventNone },
    { GpioPadName_PdVconnEn          , InternalGpioPadNumber_GPIO_PK5,     WakeEventNone },
    { GpioPadName_WifiWakeHost       , InternalGpioPadNumber_GPIO_PH2,     nne::wec::WakeEvent_WIFI_WAKE_AP },
    { GpioPadName_SdCd               , InternalGpioPadNumber_GPIO_PZ1,     nne::wec::WakeEvent_GPIO_PZ1 },
    { GpioPadName_OtgFet1ForSdev     , InternalGpioPadNumber_GPIO_PJ5,     WakeEventNone },
    { GpioPadName_OtgFet2ForSdev     , InternalGpioPadNumber_GPIO_PL0,     WakeEventNone },
    { GpioPadName_ExtConWakeU        , InternalGpioPadNumber_GPIO_PH6,     WakeEventNone },
    { GpioPadName_ExtConWakeS        , InternalGpioPadNumber_GPIO_PE6,     WakeEventNone },
    { GpioPadName_PmuIrq             , InternalGpioPadNumber_None,         nne::wec::WakeEvent_PWR_INT_N },
    { GpioPadName_ExtUart2Cts        , InternalGpioPadNumber_GPIO_PG3,     nne::wec::WakeEvent_UART2_CTS },
    { GpioPadName_ExtUart3Cts        , InternalGpioPadNumber_GPIO_PD4,     nne::wec::WakeEvent_UART3_CTS },

    // TODO : Copper 向けの Pin は何かしらのチェック機構を用意したい。
    { GpioPadName_5VStepDownEn       , InternalGpioPadNumber_GPIO_PBB1,    WakeEventNone },
    { GpioPadName_UsbSwitchB2Oc      , InternalGpioPadNumber_GPIO_PB4,     WakeEventNone },
    { GpioPadName_5VStepDownPg       , InternalGpioPadNumber_GPIO_PB5,     WakeEventNone },
    { GpioPadName_UsbSwitchAEn       , InternalGpioPadNumber_GPIO_PE1,     WakeEventNone },
    { GpioPadName_UsbSwitchAFlag     , InternalGpioPadNumber_GPIO_PE7,     WakeEventNone },
    { GpioPadName_UsbSwitchB3Oc      , InternalGpioPadNumber_GPIO_PS2,     WakeEventNone },
    { GpioPadName_UsbSwitchB3En      , InternalGpioPadNumber_GPIO_PS5,     WakeEventNone },
    { GpioPadName_UsbSwitchB2En      , InternalGpioPadNumber_GPIO_PT0,     WakeEventNone },
    { GpioPadName_Hdmi5VEn           , InternalGpioPadNumber_GPIO_PC0,     WakeEventNone },
    { GpioPadName_UsbSwitchB1En      , InternalGpioPadNumber_GPIO_PC1,     WakeEventNone },
    { GpioPadName_HdmiPdTrEn         , InternalGpioPadNumber_GPIO_PC2,     WakeEventNone },
    { GpioPadName_FanEn              , InternalGpioPadNumber_GPIO_PI2,     WakeEventNone },
    { GpioPadName_UsbSwitchB1Oc      , InternalGpioPadNumber_GPIO_PCC6,    WakeEventNone },
    { GpioPadName_PwmFan             , InternalGpioPadNumber_GPIO_PV4,     WakeEventNone },
    { GpioPadName_HdmiHpd            , InternalGpioPadNumber_GPIO_PCC1,    WakeEventNone },

    // TODO : Mariko 向けの Pin は何かしらのチェック機構を用意したい。
    { GpioPadName_Max77812Irq        , InternalGpioPadNumber_GPIO_PK6,     WakeEventNone }, // PK6 はGpioPadName_PowVcpuInt と被っているので注意

    // Hoag 向けの Pin 設定
    // ToDo : 上記同様何かしらのチェックは入れた方がよい
    { GpioPadName_Debug0             , InternalGpioPadNumber_GPIO_PE0,     WakeEventNone },
    { GpioPadName_Debug1             , InternalGpioPadNumber_GPIO_PE1,     WakeEventNone }, // Copper の UsbSwitchAEn と被り
    { GpioPadName_Debug2             , InternalGpioPadNumber_GPIO_PE2,     WakeEventNone },
    { GpioPadName_Debug3             , InternalGpioPadNumber_GPIO_PE3,     WakeEventNone },
    { GpioPadName_NfcIrq             , InternalGpioPadNumber_GPIO_PJ4,     WakeEventNone },
    { GpioPadName_NfcReset           , InternalGpioPadNumber_GPIO_PK7,     WakeEventNone }, // PK7 は GpioPadName_Max77621GpuInt と被っているので注意
    { GpioPadName_McuInt             , InternalGpioPadNumber_GPIO_PS4,     WakeEventNone },
    { GpioPadName_McuBoot            , InternalGpioPadNumber_GPIO_PT0,     WakeEventNone }, // Copper の UsbSwitchB2En と被り
    { GpioPadName_McuReset           , InternalGpioPadNumber_GPIO_PT1,     WakeEventNone },
    { GpioPadName_Vdd5V3En           , InternalGpioPadNumber_GPIO_PX3,     WakeEventNone }, //Iowa でも使用
    { GpioPadName_McuPor             , InternalGpioPadNumber_GPIO_PCC5,    WakeEventNone },


    // For Compatibility
    { GpioPadName_AudioPowerEnable         , InternalGpioPadNumber_GPIO_PZ4,     WakeEventNone },
    { GpioPadName_EnablePowerToTheSdCard   , InternalGpioPadNumber_GPIO_PE4,     WakeEventNone },
    { GpioPadName_BluetoothPowerEnable     , InternalGpioPadNumber_GPIO_PH4,     WakeEventNone },

};

// 公開しているパッドの数
const int NumberOfSupportPublicPads = sizeof(PadMapCombinationList) / sizeof(PadMapCombinationList[0]);

} // detail
} // driver
} // gpio
} // nn
