﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nne/wec/wec.h>
#endif

#include <nn/gpio/driver/gpio_PadAccessorDev.h>

#include "gpio_RegAccessor-soc.tegra.h"
#include "gpio_RegAccessorPinmux-soc.tegra.h"
#include "gpio_EventHolder-soc.tegra.h"

//---------------------------------------------------------------------------
// 日本語以外の環境で表示される文字コードエンコーディングに関する警告の抑制
#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma warning( disable : 4566 )
#endif

namespace nn{
namespace gpio{
namespace driver{
namespace detail{


struct GpioPadSessionImpl
{
    explicit GpioPadSessionImpl(int number) NN_NOEXCEPT :
        padNumber(number),
        padFunction(GpioType),
        eventHolder(number)
    {}

    int                                padNumber;
    Bit32                              padFunction;
    mutable util::IntrusiveListNode    node;
    PadEventHolder                     eventHolder;
};

struct NN_ALIGNAS(GpioPadSessionAlign) GpioPadSessionImplPadded
{
    GpioPadSessionImpl impl;

    // 64bit ビルド時にサイズがかわるため、パディングを入れる。
    char           _padding[GpioPadSessionSize - sizeof(GpioPadSessionImpl)];
};


//---------------------------------------------------------------------------
// static_assert で TypedStorage のサイズとアラインを確認
//---------------------------------------------------------------------------

NN_STATIC_ASSERT(sizeof(GpioPadSessionImplPadded) == GpioPadSessionSize);
NN_STATIC_ASSERT(sizeof(GpioPadSessionImpl) <= GpioPadSessionSize);
NN_STATIC_ASSERT(NN_ALIGNOF(GpioPadSessionImplPadded) == GpioPadSessionAlign);

// 初期値設定用の構造体
struct GpioInitialConfig
{
    nn::gpio::GpioPadName     name;
    nn::gpio::Direction       direction;
    nn::gpio::GpioValue       value;
};

/**
 * @brief   GPIO のドライバークラス (Tegra K-1 向け)。
 * @detail  実質的なドライバのコア機能の実装部分。
 *          コピー・代入・移動禁止。
 */
class Driver
{
    NN_DISALLOW_COPY(Driver);
    NN_DISALLOW_MOVE(Driver);

    typedef nn::util::IntrusiveList<GpioPadSessionImpl, nn::util::IntrusiveListMemberNodeTraits<GpioPadSessionImpl, &GpioPadSessionImpl::node>>  OpenSessionList;

public:
    Driver() NN_NOEXCEPT :
        m_InitializeCount(0)
    {
        m_GpioVirtualAddress = nn::dd::QueryIoMappingAddress(GpioPhysicalAddress, GpioAddressSize);
        if (m_GpioVirtualAddress == 0)
        {
            // physicalAddress が指す I/O アドレスがマッピングされていない
            NN_ABORT("I/O registers for 0x%llx are not mapped. Make sure the capability setting is properly set for this process.\n", GpioPhysicalAddress);
        }

        m_WakeBitFlag.Reset();
        m_WakeBitFlagForDebug.Reset();
    }

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void OpenSession(GpioPadSession* pOutSession, GpioPadName pad) NN_NOEXCEPT;
    void OpenSessionDev(GpioPadSession* pOutSession, int padNumber) NN_NOEXCEPT;
    void CloseSession(GpioPadSession* pSession) NN_NOEXCEPT;

    void SetDirection(const GpioPadSession& session, Direction direction) NN_NOEXCEPT;
    Direction GetDirection(const GpioPadSession& session ) NN_NOEXCEPT;
    void SetInterruptMode(const GpioPadSession& session, InterruptMode mode ) NN_NOEXCEPT;
    InterruptMode GetInterruptMode(const GpioPadSession& session) NN_NOEXCEPT;
    void SetInterruptEnable(GpioPadSession* pSession, bool enable ) NN_NOEXCEPT;
    bool GetInterruptEnable(const GpioPadSession& session) NN_NOEXCEPT;
    InterruptStatus GetInterruptStatus(const GpioPadSession& session) NN_NOEXCEPT;
    void ClearInterruptStatus(const GpioPadSession& session) NN_NOEXCEPT;
    GpioValue GetValue(const GpioPadSession& session) NN_NOEXCEPT;
    void SetValue(const GpioPadSession& session, GpioValue value ) NN_NOEXCEPT;
    void SetValueForSleepState(const GpioPadSession& session, GpioValue value ) NN_NOEXCEPT;
    void SetDebounceEnabled(GpioPadSession* pSession, bool isEnable) NN_NOEXCEPT;
    bool GetDebounceEnabled(GpioPadSession* pSession) NN_NOEXCEPT;
    void SetDebounceTime(GpioPadSession* pSession, int msecTime) NN_NOEXCEPT;
    int GetDebounceTime(GpioPadSession* pSession) NN_NOEXCEPT;

    Result BindInterrupt(nn::os::SystemEventType* pEvent, GpioPadSession* pSession) NN_NOEXCEPT;
    void UnbindInterrupt(GpioPadSession* pSession) NN_NOEXCEPT;

    bool IsWakeEventActive(GpioPadName name) NN_NOEXCEPT;
    WakeBitFlag GetWakeEventActiveFlagSet() NN_NOEXCEPT;
    void SetWakeEventActiveFlagSetForDebug(GpioPadName name, bool isEnabled) NN_NOEXCEPT;
    void SetWakePinDebugMode(WakePinDebugMode mode) NN_NOEXCEPT;

    void SetAutoWakeFlagEnabled(bool enabled) NN_NOEXCEPT;
    void SetWakePinsDisableModeEnabled(bool enabled) NN_NOEXCEPT;

    void Suspend() NN_NOEXCEPT;
    void SuspendLow() NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;
    void ResumeLow() NN_NOEXCEPT;

    void SetInitialGpioConfig() NN_NOEXCEPT;
    void SetInitialWakePinConfig() NN_NOEXCEPT;

private:
    void OpenSessionImpl(GpioPadSession* pOutSession, InternalGpioPadNumber padNumber) NN_NOEXCEPT;
    void SetInterruptEnableToRegister(const GpioPadSession& session, bool enable )  NN_NOEXCEPT;

private:

    int m_InitializeCount;

    uintptr_t m_GpioVirtualAddress;
    uintptr_t m_ApbMiscVirtualAddress;

#ifndef NN_BUILD_CONFIG_HARDWARE_NX
    // TORIAEZU : Pad ごとに持って、Open 時に設定した方が良いが大きく構成を変えないため、一旦ここで持つ
    PinmuxRegAccessor m_PinmuxAccessor;
#endif

    // Open 済み Session を管理するリスト
    OpenSessionList                m_OpenSessionList;

    // Suspend 時にレジスタの値を保存する配列
    struct RegisterValues
    {
        nn::Bit16  conf;
        nn::Bit8   oe;
        nn::Bit8   out;
        nn::Bit8   intEnb;
        nn::Bit32  intLvl;
        nn::Bit8   dbCtrl;
        nn::Bit8   dbCnt;
    };
    RegisterValues m_EscapedRegisterValue[GpioPadPort_Num];

    // Suspend 時の GPIO 出力の設定値を保存しておく配列
    struct GpioValuesForSleepState
    {
        nn::Bit8   IsForceSet;
        nn::Bit8   out;
    };
    GpioValuesForSleepState m_GpioValuesForSleep[GpioPadPort_Num];

    // 起床要因を保存しておく BitFlag
    WakeBitFlag m_WakeBitFlag;
    WakeBitFlag m_WakeBitFlagForDebug;

    inline bool IsSessionOpened(const GpioPadSessionImpl& session) const
    {
        for (auto&& e : m_OpenSessionList)
        {
            if (&e == &session)
            {
                return true;
            }
        }
        return false;
    }
};



} // detail
} // driver
} // gpio
} // nn
