﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_Presence.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/detail/friends_TypesDetail.h>
#include <nn/friends/detail/friends_ImplPresence.h>
#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/friends_PresenceAccessor.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

#define GET_IMPL() NN_DETAIL_FRIENDS_GET_IMPL(UserPresence)
#define CONST_GET_IMPL() NN_DETAIL_FRIENDS_CONST_GET_IMPL(UserPresence)

#define GET_FRIEND_IMPL() NN_DETAIL_FRIENDS_GET_IMPL(FriendPresence)
#define CONST_GET_FRIEND_IMPL() NN_DETAIL_FRIENDS_CONST_GET_IMPL(FriendPresence)

namespace nn { namespace friends {

UserPresence::UserPresence() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_CLEAR_IMPL();
}

nn::Result UserPresence::Initialize(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::CheckUserStatus(uid));

    GET_IMPL()->uid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result UserPresence::Initialize() NN_NOEXCEPT
{
    GET_IMPL()->uid = detail::GetCurrentUser();

    NN_RESULT_SUCCESS;
}

void UserPresence::Clear() NN_NOEXCEPT
{
    std::memset(GET_IMPL()->appField, 0, sizeof (GET_IMPL()->appField));

    GET_IMPL()->onlinePlayDeclaration = detail::OnlinePlayDeclaration_None;
}

nn::Result UserPresence::SetDescription(const char* description) NN_NOEXCEPT
{
    return SetAppValueImpl("sys_description", description);
}

nn::Result UserPresence::SetAppValue(const char* key, const char* value) NN_NOEXCEPT
{
    if (nn::util::Strnicmp(key, "sys_", 4) == 0)
    {
        NN_RESULT_THROW(ResultReservedKey());
    }

    return SetAppValueImpl(key, value);
}

void UserPresence::DeclareOpenOnlinePlaySession() NN_NOEXCEPT
{
    GET_IMPL()->onlinePlayDeclaration = detail::OnlinePlayDeclaration_Open;
}

void UserPresence::DeclareCloseOnlinePlaySession() NN_NOEXCEPT
{
    GET_IMPL()->onlinePlayDeclaration = detail::OnlinePlayDeclaration_Close;
}

nn::Result UserPresence::Commit() NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(CONST_GET_IMPL()->uid));

    NN_RESULT_DO(session->UpdateUserPresence(
        CONST_GET_IMPL()->uid,
        *reinterpret_cast<detail::UserPresenceImpl*>(&m_Storage),
        0));

    NN_RESULT_SUCCESS;
}

nn::Result UserPresence::SetAppValueImpl(const char* key, const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(key);
    NN_SDK_REQUIRES_NOT_NULL(value);

    NN_RESULT_DO(detail::PresenceAccessor::AddKeyValue(key, value, GET_IMPL()->appField, sizeof (GET_IMPL()->appField)));

    NN_RESULT_SUCCESS;
}

FriendPresence::FriendPresence() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_CLEAR_IMPL();
}

PresenceStatus FriendPresence::GetStatus() const NN_NOEXCEPT
{
    return static_cast<PresenceStatus>(CONST_GET_FRIEND_IMPL()->status);
}

const ApplicationInfo& FriendPresence::GetLastPlayedApplication() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->lastPlayedApp;
}

const nn::time::PosixTime& FriendPresence::GetLastOnlineTime() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->lastUpdateTime;
}

const nn::time::PosixTime& FriendPresence::GetLastUpdateTime() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->lastUpdateTime;
}

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

const char* FriendPresence::GetDescription() const NN_NOEXCEPT
{
    return GetAppValue("sys_description");
}

NN_PRAGMA_POP_WARNINGS

const char* FriendPresence::GetAppValue(const char* key) const NN_NOEXCEPT
{
    const char* value = detail::PresenceAccessor::SearchKey(key,
        CONST_GET_FRIEND_IMPL()->appField, sizeof (CONST_GET_FRIEND_IMPL()->appField));

    if (!value)
    {
        return "";
    }

    return value;
}

bool FriendPresence::IsSamePresenceGroupApplication() const NN_NOEXCEPT
{
    return CONST_GET_FRIEND_IMPL()->isSamePresenceGroupApp;
}

}}
