﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/friends_ApiAdmin.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/detail/friends_ApiDetail.h>
#include <nn/friends/detail/friends_Async.h>
#include <nn/friends/detail/friends_ShimLibraryGlobal.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <cstdarg>

namespace nn { namespace friends {

void SetOption(int32_t option, ...) NN_NOEXCEPT
{
    std::va_list ap;

    va_start(ap, option);

    switch (option)
    {
    case OptionAdmin_CheckUserStatus:
        {
            detail::EnableCheckUserStatus(va_arg(ap, int) ? true : false);
        }
        break;
    default:
        break;
    }

    va_end(ap);
}

nn::Result GetFriendCount(int* outCount,
    const nn::account::Uid& uid, const FriendFilter& filter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    detail::ipc::SizedFriendFilter sizedFilter;
    sizedFilter.Box(filter);

    int32_t count = 0;

    NN_RESULT_DO(session->GetFriendCount(&count, uid, sizedFilter, 0));

    *outCount = count;

    NN_RESULT_SUCCESS;
}

nn::Result GetNewlyFriendCount(int* outCount,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    int32_t count = 0;

    NN_RESULT_DO(session->GetNewlyFriendCount(&count, uid));

    *outCount = count;

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendDetailedInfo(FriendDetailedInfo* outInfo,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outInfo);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->GetFriendDetailedInfo(
        nn::sf::Out<detail::FriendDetailedInfoImpl>(reinterpret_cast<detail::FriendDetailedInfoImpl*>(outInfo)),
        uid,
        accountId));

    NN_RESULT_SUCCESS;
}

nn::Result DropFriendNewlyFlags(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->DropFriendNewlyFlags(uid));

    NN_RESULT_SUCCESS;
}

nn::Result DeleteFriend(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
    }
    ap = {uid, accountId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::DeleteFriend",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->DeleteFriend(ap->uid, ap->accountId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result SyncFriendList(AsyncContext* outAsync,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
    }
    ap = {uid};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SyncFriendList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SyncFriendList(ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result RequestSyncFriendList(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->RequestSyncFriendList(uid));

    NN_RESULT_SUCCESS;
}

nn::Result GetReceivedFriendRequestCount(int* outUnreadCount, int* outReadCount,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outUnreadCount);
    NN_SDK_REQUIRES_NOT_NULL(outReadCount);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    int32_t unreadCount = 0;
    int32_t readCount = 0;

    NN_RESULT_DO(session->GetReceivedFriendRequestCount(&unreadCount, &readCount, uid));

    *outUnreadCount = unreadCount;
    *outReadCount = readCount;

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendRequestList(AsyncContext* outAsync, int* outCount, FriendRequest* outRequests,
    const nn::account::Uid& uid, int offset, int count, RequestListType listType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outRequests);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        int* outCount;
        detail::FriendRequestImpl* outRequests;
        nn::account::Uid uid;
        int offset;
        int count;
        RequestListType listType;
    }
    ap = {outCount, reinterpret_cast<detail::FriendRequestImpl*>(outRequests), uid, offset, count, listType};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetFriendRequestList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            // 対面フレンド申請リストはセーブデータからの読み込みなのでインターネット接続は要求しない。
            if (ap->listType != RequestListType_Faced)
            {
                NN_RESULT_DO(detail::CheckInternetRequestAccepted());
            }

            int32_t actualCount = 0;

            NN_RESULT_DO(session->GetFriendRequestList(
                &actualCount,
                nn::sf::OutArray<detail::FriendRequestImpl>(ap->outRequests, static_cast<size_t>(ap->count)),
                ap->uid,
                ap->offset,
                ap->listType));

            *ap->outCount = actualCount;

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        RequestType requestType;
    }
    ap = {uid, accountId, requestType};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SendFriendRequest",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SendFriendRequest(
                ap->uid,
                ap->accountId,
                ap->requestType));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        RequestType requestType;
        ApplicationInfo appInfo;
        InAppScreenName inAppScreenName;
        InAppScreenName myInAppScreenName;
    }
    ap = {uid, accountId, requestType, appInfo, inAppScreenName, myInAppScreenName};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SendFriendRequestWithApplicationInfo",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            InAppScreenName inAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&inAppScreenNameNormalized, ap->inAppScreenName));

            InAppScreenName myInAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&myInAppScreenNameNormalized, ap->myInAppScreenName));

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SendFriendRequestWithApplicationInfo(
                ap->uid,
                ap->accountId,
                ap->requestType,
                ap->appInfo,
                inAppScreenNameNormalized,
                myInAppScreenNameNormalized));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const ExternalApplicationCatalogId& catalogId, const InAppScreenName& inAppScreenName, const InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        RequestType requestType;
        ExternalApplicationCatalogId catalogId;
        InAppScreenName inAppScreenName;
        InAppScreenName myInAppScreenName;
    }
    ap = {uid, accountId, requestType, catalogId, inAppScreenName, myInAppScreenName};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SendFriendRequestWithExternalApplicationCatalogId",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            InAppScreenName inAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&inAppScreenNameNormalized, ap->inAppScreenName));

            InAppScreenName myInAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&myInAppScreenNameNormalized, ap->myInAppScreenName));

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SendFriendRequestWithExternalApplicationCatalogId(
                ap->uid,
                ap->accountId,
                ap->requestType,
                ap->catalogId,
                inAppScreenNameNormalized,
                myInAppScreenNameNormalized));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result SendFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, RequestType requestType,
    const MiiName& miiName, const MiiImageUrlParam& miiImageUrlParam, const MiiName& myMiiName, const MiiImageUrlParam& myMiiImageUrlParam) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        RequestType requestType;
        MiiName miiName;
        MiiImageUrlParam miiImageUrlParam;
        MiiName myMiiName;
        MiiImageUrlParam myMiiImageUrlParam;
    }
    ap = {uid, accountId, requestType, miiName, miiImageUrlParam, myMiiName, myMiiImageUrlParam};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SendFriendRequestWithNintendoNetworkIdInfo",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SendFriendRequestWithNintendoNetworkIdInfo(
                ap->uid,
                ap->accountId,
                ap->requestType,
                ap->miiName,
                ap->miiImageUrlParam,
                ap->myMiiName,
                ap->myMiiImageUrlParam));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result CancelFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        RequestId requestId;
    }
    ap = {uid, requestId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::CancelFriendRequest",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->CancelFriendRequest(
                ap->uid,
                ap->requestId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result AcceptFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        RequestId requestId;
    }
    ap = {uid, requestId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::AcceptFriendRequest",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->AcceptFriendRequest(
                ap->uid,
                ap->requestId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result RejectFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        RequestId requestId;
    }
    ap = {uid, requestId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::RejectFriendRequest",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->RejectFriendRequest(
                ap->uid,
                ap->requestId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result ReadFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, RequestId requestId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        RequestId requestId;
    }
    ap = {uid, requestId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ReadFriendRequest",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ReadFriendRequest(
                ap->uid,
                ap->requestId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result AddFacedFriendRequest(const nn::account::Uid& uid,
    const FacedFriendRequestRegistrationKey& key,
    const nn::account::Nickname& nickname, const void* image, size_t imageSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(image);
    NN_SDK_REQUIRES_MINMAX(imageSize, 1u, ProfileImageSizeMax);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->AddFacedFriendRequest(
        uid,
        key,
        nickname,
        nn::sf::InBuffer(static_cast<const char*>(image), imageSize)));

    NN_RESULT_SUCCESS;
}

nn::Result CancelFacedFriendRequest(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->CancelFacedFriendRequest(uid, accountId));

    NN_RESULT_SUCCESS;
}

nn::Result GetFacedFriendRequestProfileImage(size_t* outSize,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    int32_t actualSize = 0;

    NN_RESULT_DO(session->GetFacedFriendRequestProfileImage(
        &actualSize,
        uid,
        accountId,
        nn::sf::OutBuffer(static_cast<char*>(buffer), size)));

    *outSize = static_cast<size_t>(actualSize);

    NN_RESULT_SUCCESS;
}

nn::Result GetFacedFriendRequestProfileImage(size_t* outSize,
    const char* path, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSize);
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    int pathLength = nn::util::Strnlen(path, sizeof (Url));

    NN_RESULT_THROW_UNLESS(static_cast<size_t>(pathLength) < sizeof (Url),
        ResultInvalidArgument());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    int32_t actualSize = 0;

    NN_RESULT_DO(session->GetFacedFriendRequestProfileImageFromPath(
        &actualSize,
        nn::sf::InArray<char>(path, static_cast<size_t>(pathLength) + 1),
        nn::sf::OutBuffer(static_cast<char*>(buffer), size)));

    *outSize = static_cast<size_t>(actualSize);

    NN_RESULT_SUCCESS;
}

nn::Result ResendFacedFriendRequest(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
    }
    ap = {uid, accountId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::ResendFacedFriendRequest",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->ResendFacedFriendRequest(
                ap->uid,
                ap->accountId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetFriendCandidateList(AsyncContext* outAsync, int* outCount, FriendCandidate* outCandidates,
    const nn::account::Uid& uid, int count, FriendCandidateType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outCandidates);
    NN_SDK_REQUIRES(count > 0);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        int* outCount;
        detail::FriendCandidateImpl* outCandidates;
        nn::account::Uid uid;
        int count;
        FriendCandidateType type;
    }
    ap = {outCount, reinterpret_cast<detail::FriendCandidateImpl*>(outCandidates), uid, count, type};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetFriendCandidateList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            int32_t actualCount = 0;

            NN_RESULT_DO(session->GetFriendCandidateList(
                &actualCount,
                nn::sf::OutArray<detail::FriendCandidateImpl>(ap->outCandidates, static_cast<size_t>(ap->count)),
                ap->uid,
                ap->type));

            *ap->outCount = actualCount;

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetNintendoNetworkIdInfo(AsyncContext* outAsync, NintendoNetworkIdUserInfo* outUserInfo,
    int* outCount, NintendoNetworkIdFriend* outFriends,
    const nn::account::Uid& uid, int count, NintendoNetworkIdAccountType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outUserInfo);
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outFriends);
    NN_SDK_REQUIRES(count > 0);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        NintendoNetworkIdUserInfo* outUserInfo;
        int* outCount;
        detail::NintendoNetworkIdFriendImpl* outFriends;
        nn::account::Uid uid;
        int count;
        NintendoNetworkIdAccountType type;
    }
    ap = {outUserInfo, outCount, reinterpret_cast<detail::NintendoNetworkIdFriendImpl*>(outFriends), uid, count, type};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetNintendoNetworkIdInfo",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            int32_t actualCount = 0;

            NN_RESULT_DO(session->GetNintendoNetworkIdInfo(
                ap->outUserInfo,
                &actualCount,
                nn::sf::OutArray<detail::NintendoNetworkIdFriendImpl>(ap->outFriends, static_cast<size_t>(ap->count)),
                ap->uid,
                ap->type));

            *ap->outCount = actualCount;

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetSnsAccountLinkPageUrl(AsyncContext* outAsync, WebPageUrl* outUrl,
    const nn::account::Uid& uid, SnsAccountType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outUrl);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    struct AsyncParam
    {
        WebPageUrl* outUrl;
        nn::account::Uid uid;
        SnsAccountType type;
    }
    ap = {outUrl, uid, type};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetSnsAccountLinkPageUrl",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetSnsAccountLinkPageUrl(
                ap->outUrl,
                ap->uid,
                ap->type));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result UnlinkSnsAccount(AsyncContext* outAsync,
    const nn::account::Uid& uid, SnsAccountType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    struct AsyncParam
    {
        nn::account::Uid uid;
        SnsAccountType type;
    }
    ap = {uid, type};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::UnlinkSnsAccount",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->UnlinkSnsAccount(
                ap->uid,
                ap->type));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetSnsAccountLinkage(AsyncContext* outAsync, SnsAccountLinkage* outLinkage,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outLinkage);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    struct AsyncParam
    {
        SnsAccountLinkage* outLinkage;
        nn::account::Uid uid;
    }
    ap = {outLinkage, uid};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetSnsAccountLinkage",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetSnsAccountLinkage(
                ap->outLinkage,
                ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetSnsAccountProfile(AsyncContext* outAsync, SnsAccountProfile* outProfile,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, SnsAccountType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outProfile);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    struct AsyncParam
    {
        SnsAccountProfile* outProfile;
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        SnsAccountType type;
    }
    ap = {outProfile, uid, accountId, type};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetSnsAccountProfile",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetSnsAccountProfile(
                ap->outProfile,
                ap->uid,
                ap->accountId,
                ap->type));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetSnsAccountFriendList(AsyncContext* outAsync, int* outCount, SnsAccountFriend* outFriends,
    const nn::account::Uid& uid, int count, SnsAccountType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outFriends);
    NN_SDK_REQUIRES(count > 0);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    struct AsyncParam
    {
        int* outCount;
        detail::SnsAccountFriendImpl* outFriends;
        nn::account::Uid uid;
        int count;
        SnsAccountType type;
    }
    ap = {outCount, reinterpret_cast<detail::SnsAccountFriendImpl*>(outFriends), uid, count, type};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetSnsAccountFriendList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            int32_t actualCount = 0;

            NN_RESULT_DO(session->GetSnsAccountFriendList(
                &actualCount,
                nn::sf::OutArray<detail::SnsAccountFriendImpl>(ap->outFriends, static_cast<size_t>(ap->count)),
                ap->uid,
                ap->type));

            *ap->outCount = actualCount;

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetBlockedUserList(int* outCount, BlockedUser* outUsers,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outUsers);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    int32_t actualCount = 0;

    NN_RESULT_DO(session->GetBlockedUserList(
        &actualCount,
        nn::sf::OutArray<detail::BlockedUserImpl>(reinterpret_cast<detail::BlockedUserImpl*>(outUsers), static_cast<size_t>(count)),
        uid,
        static_cast<int32_t>(offset)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result BlockUser(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, BlockReason blockReason) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        BlockReason blockReason;
    }
    ap = {uid, accountId, blockReason};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::BlockUser",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->BlockUser(
                ap->uid,
                ap->accountId,
                ap->blockReason));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result BlockUser(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, BlockReason blockReason,
    const ApplicationInfo& appInfo, const InAppScreenName& inAppScreenName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
        BlockReason blockReason;
        ApplicationInfo appInfo;
        InAppScreenName inAppScreenName;
    }
    ap = {uid, accountId, blockReason, appInfo, inAppScreenName};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::BlockUserWithApplicationInfo",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            InAppScreenName inAppScreenNameNormalized;
            NN_RESULT_DO(detail::NormalizeInAppScreenName(&inAppScreenNameNormalized, ap->inAppScreenName));

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->BlockUserWithApplicationInfo(
                ap->uid,
                ap->accountId,
                ap->blockReason,
                ap->appInfo,
                inAppScreenNameNormalized));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result UnblockUser(AsyncContext* outAsync,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
    }
    ap = {uid, accountId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::UnblockUser",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->UnblockUser(
                ap->uid,
                ap->accountId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result SyncBlockedUserList(AsyncContext* outAsync,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
    }
    ap = {uid};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SyncBlockedUserList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SyncBlockedUserList(ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileExtraList(AsyncContext* outAsync, ProfileExtra* outProfiles,
    const nn::account::Uid& uid, const nn::account::NetworkServiceAccountId* accountIds, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outProfiles);
    NN_SDK_REQUIRES_NOT_NULL(accountIds);
    NN_SDK_REQUIRES_MINMAX(count, 1, GetProfileCountMax);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParamForShallowCopy
    {
        detail::ProfileExtraImpl* outProfiles;
        nn::account::Uid uid;
        const nn::account::NetworkServiceAccountId* accountIds;
        int count;
    }
    ap = {reinterpret_cast<detail::ProfileExtraImpl*>(outProfiles), uid, accountIds, count};

    struct AsyncParam
    {
        detail::ProfileExtraImpl* outProfiles;
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountIds[GetProfileCountMax];
        int count;
    };

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetProfileExtraList",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetProfileExtraList(
                nn::sf::OutArray<detail::ProfileExtraImpl>(ap->outProfiles, static_cast<size_t>(ap->count)),
                ap->uid,
                nn::sf::InArray<nn::account::NetworkServiceAccountId>(ap->accountIds, static_cast<size_t>(ap->count))));

            NN_RESULT_SUCCESS;
        },
        [](const void* source, void* dest, size_t size) -> void
        {
            NN_SDK_ASSERT(sizeof (AsyncParam) <= size);
            NN_UNUSED(size);

            const AsyncParamForShallowCopy* sp = static_cast<const AsyncParamForShallowCopy*>(source);
            AsyncParam* dp = static_cast<AsyncParam*>(dest);

            dp->outProfiles = sp->outProfiles;
            dp->uid = sp->uid;
            std::memcpy(dp->accountIds, sp->accountIds, sizeof (nn::account::NetworkServiceAccountId) * sp->count);
            dp->count = sp->count;
        },
        &ap));

    NN_RESULT_SUCCESS;
}

nn::Result GetRelationship(AsyncContext* outAsync, Relationship* outRelationship,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outRelationship);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        Relationship* outRelationship;
        nn::account::Uid uid;
        nn::account::NetworkServiceAccountId accountId;
    }
    ap = {outRelationship, uid, accountId};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetRelationship",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetRelationship(
                ap->outRelationship,
                ap->uid,
                ap->accountId));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetUserPresenceView(UserPresenceView* outPresence,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPresence);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->GetUserPresenceView(
        reinterpret_cast<detail::UserPresenceViewImpl*>(outPresence),
        uid));

    NN_RESULT_SUCCESS;
}

nn::Result SyncUserSetting(AsyncContext* outAsync,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        nn::account::Uid uid;
    }
    ap = {uid};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::SyncUserSetting",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->SyncUserSetting(ap->uid));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetProfileExtra(AsyncContext* outAsync, ProfileExtra* outProfile,
    const nn::account::Uid& uid, const FriendCode& friendCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outProfile);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    struct AsyncParam
    {
        detail::ProfileExtraImpl* outProfile;
        nn::account::Uid uid;
        FriendCode friendCode;
    }
    ap = {reinterpret_cast<detail::ProfileExtraImpl*>(outProfile), uid, friendCode};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetProfileExtraFromFriendCode",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetProfileExtraFromFriendCode(
                ap->outProfile,
                ap->uid,
                ap->friendCode));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetPlayHistoryList(int* outCount, PlayHistory* outHistories,
    const nn::account::Uid& uid, int offset, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outHistories);
    NN_SDK_REQUIRES(offset >= 0);
    NN_SDK_REQUIRES(count > 0);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    int32_t actualCount = 0;

    NN_RESULT_DO(session->GetPlayHistoryList(
        &actualCount,
        nn::sf::OutArray<detail::PlayHistoryImpl>(reinterpret_cast<detail::PlayHistoryImpl*>(outHistories), static_cast<size_t>(count)),
        uid,
        static_cast<int32_t>(offset)));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result GetPlayHistoryStatistics(PlayHistoryStatistics* outStatistics,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outStatistics);

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->GetPlayHistoryStatistics(outStatistics, uid));

    NN_RESULT_SUCCESS;
}

nn::Result DeletePlayHistory(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->DeletePlayHistory(uid));

    NN_RESULT_SUCCESS;
}

nn::Result RequestListSummaryOverlayNotification() NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(session->RequestListSummaryOverlayNotification());

    NN_RESULT_SUCCESS;
}

nn::Result SuspendDaemon(DaemonSuspension* outSuspension) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSuspension);

    detail::ipc::IDaemonSuspendSessionService* pService = nullptr;
    NN_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateDaemonSuspendSessionService(&pService));

    outSuspension->Attach(pService);

    NN_RESULT_SUCCESS;
}

nn::Result DeleteNetworkServiceAccountCache(const nn::account::Uid& uid) NN_NOEXCEPT
{
    auto session = detail::ShimLibraryGlobal::GetInstance().GetSyncSession();

    NN_RESULT_DO(detail::CheckUserStatus(uid));

    NN_RESULT_DO(session->DeleteNetworkServiceAccountCache(uid));

    NN_RESULT_SUCCESS;
}

nn::Result GetExternalApplicationCatalog(AsyncContext* outAsync, ExternalApplicationCatalog* outCatalog,
    const ExternalApplicationCatalogId& catalogId, const nn::settings::LanguageCode language) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAsync);
    NN_SDK_REQUIRES_NOT_NULL(outCatalog);

    detail::ShimLibraryGlobal::GetInstance().Initialize();

    struct AsyncParam
    {
        ExternalApplicationCatalog* outCatalog;
        ExternalApplicationCatalogId catalogId;
        nn::settings::LanguageCode language;
    }
    ap = {outCatalog, catalogId, language};

    NN_RESULT_DO(detail::CallAsync(outAsync,
        "nn::friends::GetExternalApplicationCatalog",
        [](detail::ipc::IFriendService* session, void* param) -> nn::Result
        {
            AsyncParam* ap = static_cast<AsyncParam*>(param);

            NN_RESULT_DO(detail::CheckInternetRequestAccepted());

            NN_RESULT_DO(session->GetExternalApplicationCatalog(
                ap->outCatalog,
                ap->catalogId,
                ap->language));

            NN_RESULT_SUCCESS;
        },
        &ap, sizeof (AsyncParam)));

    NN_RESULT_SUCCESS;
}

nn::Result GetMiiImageUrl(Url* outUrl, const MiiImageUrlParam& miiImageUrlParam) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::GetMiiImageUrl(outUrl, miiImageUrlParam));

    NN_RESULT_SUCCESS;
}

}}
