﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/friends_FriendService.h>
#include <nn/friends/detail/service/friends_Common.h>

#include <nn/friends/detail/service/core/friends_BlockedUserListManager.h>
#include <nn/friends/detail/service/core/friends_FriendListManager.h>
#include <nn/friends/detail/service/core/friends_FriendRequestCountManager.h>
#include <nn/friends/detail/service/core/friends_FacedFriendRequestManager.h>
#include <nn/friends/detail/service/core/friends_ImageDownloadQueueManager.h>
#include <nn/friends/detail/service/core/friends_ImageManager.h>
#include <nn/friends/detail/service/core/friends_UserPresenceManager.h>
#include <nn/friends/detail/service/core/friends_UserSettingManager.h>
#include <nn/friends/detail/service/core/friends_PlayHistoryManager.h>
#include <nn/friends/detail/service/core/friends_PlayLogManager.h>

#include <nn/friends/detail/service/core/friends_BackgroundTaskThread.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_ForegroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_UuidManager.h>

#include <nn/friends/detail/service/core/task/friends_TaskSyncFriendList.h>
#include <nn/friends/detail/service/core/task/friends_TaskDeleteFriend.h>
#include <nn/friends/detail/service/core/task/friends_TaskChangeFriendFavoriteFlag.h>
#include <nn/friends/detail/service/core/task/friends_TaskChangeFriendOnlineNotificationFlag.h>

#include <nn/friends/detail/service/core/task/friends_TaskGetFriendRequestList.h>
#include <nn/friends/detail/service/core/task/friends_TaskSendFacedFriendRequest.h>
#include <nn/friends/detail/service/core/task/friends_TaskSendFriendRequest.h>
#include <nn/friends/detail/service/core/task/friends_TaskChangeFriendRequestStatus.h>
#include <nn/friends/detail/service/core/task/friends_TaskReadFriendRequest.h>

#include <nn/friends/detail/service/core/task/friends_TaskGetFriendCandidateList.h>
#include <nn/friends/detail/service/core/task/friends_TaskGetNintendoNetworkIdInfo.h>

#include <nn/friends/detail/service/core/task/friends_TaskGetSnsAccountLinkPageUrl.h>
#include <nn/friends/detail/service/core/task/friends_TaskUnlinkSnsAccount.h>
#include <nn/friends/detail/service/core/task/friends_TaskGetSnsAccountFriendList.h>
#include <nn/friends/detail/service/core/task/friends_TaskGetSnsAccountLinkage.h>
#include <nn/friends/detail/service/core/task/friends_TaskGetSnsAccountProfile.h>

#include <nn/friends/detail/service/core/task/friends_TaskSyncBlockedUserList.h>
#include <nn/friends/detail/service/core/task/friends_TaskBlockUser.h>
#include <nn/friends/detail/service/core/task/friends_TaskUnblockUser.h>

#include <nn/friends/detail/service/core/task/friends_TaskSyncUserSetting.h>
#include <nn/friends/detail/service/core/task/friends_TaskChangePresencePermission.h>
#include <nn/friends/detail/service/core/task/friends_TaskChangeFriendRequestReception.h>
#include <nn/friends/detail/service/core/task/friends_TaskChangePlayLogPermission.h>
#include <nn/friends/detail/service/core/task/friends_TaskClearPlayLog.h>
#include <nn/friends/detail/service/core/task/friends_TaskIssueFriendCode.h>

#include <nn/friends/detail/service/core/task/friends_TaskGetProfileList.h>
#include <nn/friends/detail/service/core/task/friends_TaskGetRelationship.h>
#include <nn/friends/detail/service/core/task/friends_TaskGetExternalApplicationCatalog.h>

#include <nn/friends/detail/service/util/friends_ApplicationInfo.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

#define CHECK_UID(uid) \
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidArgument())

#define TASK_DO(TCLASS, uid, ...) \
    do                                                                                                                      \
    {                                                                                                                       \
        NN_STATIC_ASSERT(sizeof (TCLASS) <= sizeof (m_TaskBuffer));                                                         \
                                                                                                                            \
        NN_RESULT_THROW_UNLESS(!m_Task, ResultInvalidOperation());                                                          \
                                                                                                                            \
        TCLASS* task_ = new (m_TaskBuffer) TCLASS;                                                                          \
        task_->SetUid(uid);                                                                                                 \
        NN_RESULT_DO(task_->SetParameter(__VA_ARGS__));                                                                     \
                                                                                                                            \
        if (m_IsAlreadyCanceled)                                                                                            \
        {                                                                                                                   \
            task_->Cancel();                                                                                                \
        }                                                                                                                   \
                                                                                                                            \
        m_Task = task_;                                                                                                     \
                                                                                                                            \
        detail::service::core::ForegroundTaskManager::GetInstance().Add(m_Task);                                            \
        m_Task->Wait();                                                                                                     \
        detail::service::core::ForegroundTaskManager::GetInstance().Remove(m_Task);                                         \
                                                                                                                            \
        detail::service::core::BackgroundTaskManager::GetInstance().NotifyForegroundCommunicationDone(m_Task->GetResult()); \
                                                                                                                            \
        NN_RESULT_DO(m_Task->GetResult());                                                                                  \
    }                                                                                                                       \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace friends { namespace detail { namespace service {

FriendService::FriendService(const Capability& capability) NN_NOEXCEPT :
    m_Mutex(false),
    m_Capability(capability),
    m_IsCompletionEventCreated(false),
    m_Task(nullptr),
    m_IsAlreadyCanceled(false)
{
}

FriendService::~FriendService() NN_NOEXCEPT
{
    if (m_IsCompletionEventCreated)
    {
        m_CompletionEvent.Signal();
    }
    if (m_Task)
    {
        m_Task->~Task();
    }
}

nn::Result FriendService::GetCompletionEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_IsCompletionEventCreated)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateSystemEvent(m_CompletionEvent.GetBase(),
            nn::os::EventClearMode_ManualClear, true));

        m_IsCompletionEventCreated = true;
    }

    outHandle.Set(nn::sf::NativeHandle(m_CompletionEvent.DetachReadableHandle(), true));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::Cancel() NN_NOEXCEPT
{
    if (m_Task)
    {
        m_Task->Cancel();
    }
    else
    {
        m_IsAlreadyCanceled = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendListIds(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::account::NetworkServiceAccountId>& outAccountIds,
    const nn::account::Uid& uid, std::int32_t offset, const nn::friends::detail::ipc::SizedFriendFilter& filter, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    int actualCount = 0;

    ApplicationInfo appInfo = {};

    if (filter.isSameAppPlayedOnly || filter.isSameAppPresenceOnly)
    {
        NN_RESULT_DO(detail::service::util::GetApplicationInfo(&appInfo, processId));
    }

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetFriendList(&actualCount,
        outAccountIds.GetData(), uid, offset, static_cast<int>(outAccountIds.GetLength()), filter, appInfo));

    *outCount = actualCount;

    if (actualCount > 0)
    {
        detail::service::core::ImageManager::GetInstance().MoveToTopRecords(outAccountIds.GetData(), actualCount);
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::friends::detail::FriendImpl>& outFriends,
    const nn::account::Uid& uid, std::int32_t offset, const nn::friends::detail::ipc::SizedFriendFilter& filter, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    // View 権限がある場合、すべてのアプリ固有情報を見ることができる。
    bool checkAppFieldPermission = !m_Capability.IsPermitted(Capability::Flag_View);

    int actualCount = 0;

    ApplicationInfo appInfo = {};
    nn::Result result = detail::service::util::GetApplicationInfo(&appInfo, processId);

    if (result.IsFailure() && (filter.isSameAppPlayedOnly || filter.isSameAppPresenceOnly))
    {
        return result;
    }

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetFriendList(&actualCount,
        outFriends.GetData(), uid, offset, static_cast<int>(outFriends.GetLength()), filter, appInfo, checkAppFieldPermission));

    *outCount = actualCount;

    if (actualCount > 0)
    {
        detail::service::core::ImageManager::GetInstance().MoveToTopRecords(outFriends.GetData(), actualCount);
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::UpdateFriendInfo(const nn::sf::OutArray<nn::friends::detail::FriendImpl>& outFriends,
    const nn::account::Uid& uid, const nn::sf::InArray<nn::account::NetworkServiceAccountId>& accountIds, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(outFriends.GetLength() == accountIds.GetLength(), ResultInvalidArgument());

    // View 権限がある場合、すべてのアプリ固有情報を見ることができる。
    bool checkAppFieldPermission = !m_Capability.IsPermitted(Capability::Flag_View);

    ApplicationInfo appInfo = {};
    detail::service::util::GetApplicationInfo(&appInfo, processId);

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetFriendInfo(outFriends.GetData(),
        uid, accountIds.GetData(), static_cast<int>(outFriends.GetLength()), appInfo, checkAppFieldPermission));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendProfileImage(nn::sf::Out<std::int32_t> outSize,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    CHECK_UID(uid);

    size_t size;

    NN_RESULT_TRY(detail::service::core::ImageManager::GetInstance().LoadImage(&size,
        accountId, buffer.GetPointerUnsafe(), buffer.GetSize()))
        NN_RESULT_CATCH(ResultProfileImageCacheNotFound)
        {
            NN_DETAIL_FRIENDS_INFO("[friends] Profile image is not found. Request to download the profile image...\n");

            NN_RESULT_DO(detail::service::core::ImageDownloadQueueManager::GetInstance().Add(uid, accountId));
            NN_RESULT_DO(detail::service::core::ImageDownloadQueueManager::GetInstance().Save());
            detail::service::core::BackgroundTaskManager::GetInstance().RequestUpdateFriendProfileImageCache(uid);

            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    *outSize = static_cast<int32_t>(size);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SendFriendRequestForApplication(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId,
    const nn::friends::InAppScreenName& inAppScreenName, const nn::friends::InAppScreenName& myInAppScreenName, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(inAppScreenName), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(myInAppScreenName), ResultInvalidArgument());

    ApplicationInfo appInfo = {};
    NN_RESULT_DO(detail::service::util::GetApplicationInfo(&appInfo, processId));

    TASK_DO(detail::service::core::SendFriendRequestTask, uid,
        accountId, RequestType_InApp, appInfo, inAppScreenName, myInAppScreenName);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::AddFacedFriendRequestForApplication(const nn::account::Uid& uid,
    const nn::friends::FacedFriendRequestRegistrationKey& key,
    const nn::account::Nickname& nickname, const nn::sf::InBuffer& image,
    const nn::friends::InAppScreenName& inAppScreenName, const nn::friends::InAppScreenName& myInAppScreenName, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyString(nickname.name, sizeof (nickname.name)), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(inAppScreenName), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(myInAppScreenName), ResultInvalidArgument());

    ApplicationInfo appInfo = {};
    NN_RESULT_DO(detail::service::util::GetApplicationInfo(&appInfo, processId));

    NN_RESULT_DO(detail::service::core::FacedFriendRequestManager::GetInstance().AddFacedFriendRequest(uid,
        key, nickname, image.GetPointerUnsafe(), image.GetSize(), appInfo, inAppScreenName, myInAppScreenName));

    detail::service::core::BackgroundTaskManager::GetInstance().NotifyFacedFriendRequestAdded(uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetBlockedUserListIds(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::account::NetworkServiceAccountId>& outAccountIds,
    const nn::account::Uid& uid, std::int32_t offset) NN_NOEXCEPT
{
    CHECK_UID(uid);

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::BlockedUserListManager::GetInstance().GetBlockedUserList(&actualCount,
        outAccountIds.GetData(), uid, offset, static_cast<int>(outAccountIds.GetLength())));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetProfileList(const nn::sf::OutArray<nn::friends::detail::ProfileImpl>& outProfiles,
    const nn::account::Uid& uid, const nn::sf::InArray<nn::account::NetworkServiceAccountId>& accountIds) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(outProfiles.GetLength() == accountIds.GetLength(), ResultInvalidArgument());

    TASK_DO(detail::service::core::GetProfileListTask, uid,
        outProfiles.GetData(), accountIds.GetData(), static_cast<int>(accountIds.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DeclareOpenOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::UserPresenceManager::GetInstance().DeclareOpenOnlinePlaySession(uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DeclareCloseOnlinePlaySession(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::UserPresenceManager::GetInstance().DeclareCloseOnlinePlaySession(uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::UpdateUserPresence(const nn::account::Uid& uid, const nn::friends::detail::UserPresenceImpl& presence, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    ApplicationInfo appInfo = {};
    NN_RESULT_DO(detail::service::util::GetApplicationInfo(&appInfo, processId));

    NN_RESULT_DO(detail::service::core::UserPresenceManager::GetInstance().SetPresence(uid, presence, appInfo));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetPlayHistoryRegistrationKey(nn::sf::Out<nn::friends::PlayHistoryRegistrationKey> outKey,
    const nn::account::Uid& uid, bool isLocalPlay) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::PlayHistoryManager::GetInstance().GetPlayHistoryRegistrationKey(outKey.GetPointer(),
        uid, isLocalPlay));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetPlayHistoryRegistrationKeyWithNetworkServiceAccountId(nn::sf::Out<nn::friends::PlayHistoryRegistrationKey> outKey, nn::account::NetworkServiceAccountId accountId, bool isLocalPlay) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::service::core::PlayHistoryManager::GetInstance().GetPlayHistoryRegistrationKey(outKey.GetPointer(),
        accountId, isLocalPlay));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::AddPlayHistory(const nn::account::Uid& uid, const nn::friends::PlayHistoryRegistrationKey& key,
    const nn::friends::InAppScreenName& inAppScreenName, const nn::friends::InAppScreenName& myInAppScreenName, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(inAppScreenName), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(myInAppScreenName), ResultInvalidArgument());

    ApplicationInfo appInfo = {};
    NN_RESULT_DO(detail::service::util::GetApplicationInfo(&appInfo, processId));

    NN_RESULT_DO(detail::service::core::PlayHistoryManager::GetInstance().AddPlayHistory(uid, key,
        appInfo, inAppScreenName, myInAppScreenName));

    nn::account::NetworkServiceAccountId accountId;

    if (detail::service::core::PlayHistoryManager::GetInstance().GetAccountIdFromPlayHistoryRegistrationKey(&accountId, key).IsSuccess())
    {
        PlayRecord record = {};

        record.appInfo = appInfo;
        record.name = inAppScreenName;
        record.myName = myInAppScreenName;

        // 更新時刻は記録しない。
        // NN_RESULT_DO(nn::time::StandardUserSystemClock::GetCurrentTime(&record.time));

        NN_RESULT_TRY(detail::service::core::FriendListManager::GetInstance().UpdateLastPlayRecord(uid, accountId, record))
            NN_RESULT_CATCH(ResultNotFriend)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetProfileImageUrl(nn::sf::Out<nn::friends::Url> outUrl, const nn::friends::Url& url, std::int32_t imageSize) NN_NOEXCEPT
{
    if (url.value[0] == '\0')
    {
        *outUrl = url;
    }
    else if (imageSize == ImageSize_Standard)
    {
        *outUrl = url;
    }
    else
    {
        switch (imageSize)
        {
        case ImageSize_64x64:
            break;
        case ImageSize_128x128:
            break;
        case ImageSize_256x256:
            break;
        default:
            NN_RESULT_THROW(ResultInvalidArgument());
        };

        nn::util::SNPrintf(outUrl->value, sizeof (outUrl->value), "%s?width=%d&height=%d", url.value, imageSize, imageSize);
    }

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendCount(nn::sf::Out<std::int32_t> outCount,
    const nn::account::Uid& uid, const nn::friends::detail::ipc::SizedFriendFilter& filter, nn::Bit64 processId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    ApplicationInfo appInfo = {};

    if (filter.isSameAppPlayedOnly || filter.isSameAppPresenceOnly)
    {
        NN_RESULT_DO(detail::service::util::GetApplicationInfo(&appInfo, processId));
    }

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetFriendCount(&actualCount, uid, filter, appInfo));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetNewlyFriendCount(nn::sf::Out<std::int32_t> outCount,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetNewlyFriendCount(&actualCount, uid));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendDetailedInfo(nn::sf::Out<nn::friends::detail::FriendDetailedInfoImpl> outInfo,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetFriendDetailedInfo(outInfo.GetPointer(), uid, accountId));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SyncFriendList(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::SyncFriendListTask, uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::RequestSyncFriendList(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    detail::service::core::BackgroundTaskManager::GetInstance().RequestSyncFriendList(uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::LoadFriendSetting(nn::sf::Out<nn::friends::detail::FriendSettingImpl> outSetting,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().GetFriendSetting(outSetting.GetPointer(), uid, accountId));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetReceivedFriendRequestCount(nn::sf::Out<std::int32_t> outUnreadCount, nn::sf::Out<std::int32_t> outReadCount,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FriendRequestCountManager::GetInstance().
        GetCount(outUnreadCount.GetPointer(), outReadCount.GetPointer(), uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendRequestList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::friends::detail::FriendRequestImpl>& outRequests,
    const nn::account::Uid& uid, std::int32_t offset, std::int32_t listType) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    TASK_DO(detail::service::core::GetFriendRequestListTask, uid,
        &actualCount, outRequests.GetData(), offset, static_cast<int>(outRequests.GetLength()),
        static_cast<RequestListType>(listType));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFriendCandidateList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::friends::detail::FriendCandidateImpl>& outCandidates,
    const nn::account::Uid& uid, std::int32_t type) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    TASK_DO(detail::service::core::GetFriendCandidateListTask, uid,
        &actualCount, outCandidates.GetData(), static_cast<int>(outCandidates.GetLength()), static_cast<FriendCandidateType>(type));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetNintendoNetworkIdInfo(nn::sf::Out<nn::friends::NintendoNetworkIdUserInfo> outUserInfo,
    nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::friends::detail::NintendoNetworkIdFriendImpl>& outFriends, const nn::account::Uid& uid, std::int32_t type) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    TASK_DO(detail::service::core::GetNintendoNetworkIdInfoTask, uid,
        outUserInfo.GetPointer(),
        &actualCount, outFriends.GetData(), static_cast<int>(outFriends.GetLength()), static_cast<NintendoNetworkIdAccountType>(type));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetSnsAccountLinkage(nn::sf::Out<nn::friends::SnsAccountLinkage> outLinkage,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::GetSnsAccountLinkageTask, uid,
        outLinkage.GetPointer());

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetSnsAccountProfile(nn::sf::Out<nn::friends::SnsAccountProfile> outProfile,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t type) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::GetSnsAccountProfileTask, uid,
        outProfile.GetPointer(), accountId, static_cast<SnsAccountType>(type));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetSnsAccountFriendList(nn::sf::Out<std::int32_t> outCount,
    const nn::sf::OutArray<nn::friends::detail::SnsAccountFriendImpl>& outFriends, const nn::account::Uid& uid, std::int32_t type) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    TASK_DO(detail::service::core::GetSnsAccountFriendListTask, uid,
        &actualCount, outFriends.GetData(), static_cast<int>(outFriends.GetLength()), static_cast<SnsAccountType>(type));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetBlockedUserList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::friends::detail::BlockedUserImpl>& outUsers,
    const nn::account::Uid& uid, std::int32_t offset) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::BlockedUserListManager::GetInstance().GetBlockedUserList(&actualCount,
        outUsers.GetData(), uid, offset, static_cast<int>(outUsers.GetLength())));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SyncBlockedUserList(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::SyncBlockedUserListTask, uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetProfileExtraList(const nn::sf::OutArray<nn::friends::detail::ProfileExtraImpl>& outProfiles,
    const nn::account::Uid& uid, const nn::sf::InArray<nn::account::NetworkServiceAccountId>& accountIds) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(outProfiles.GetLength() == accountIds.GetLength(), ResultInvalidArgument());

    TASK_DO(detail::service::core::GetProfileListTask, uid,
        outProfiles.GetData(), accountIds.GetData(), static_cast<int>(accountIds.GetLength()));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetRelationship(nn::sf::Out<nn::friends::Relationship> outRelationship,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::GetRelationshipTask, uid,
        outRelationship.GetPointer(), accountId);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetUserPresenceView(nn::sf::Out<nn::friends::detail::UserPresenceViewImpl> outPresence,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::UserPresenceManager::GetInstance().GetPresence(outPresence.GetPointer(), uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetPlayHistoryList(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::friends::detail::PlayHistoryImpl>& outHistories,
    const nn::account::Uid& uid, std::int32_t offset) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    int actualCount = 0;

    NN_RESULT_DO(detail::service::core::PlayHistoryManager::GetInstance().GetPlayHistoryList(&actualCount, outHistories.GetData(),
        uid, offset, static_cast<int>(outHistories.GetLength())));

    *outCount = actualCount;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetPlayHistoryStatistics(nn::sf::Out<nn::friends::PlayHistoryStatistics> outStatistics,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::PlayHistoryManager::GetInstance().GetPlayHistoryStatistics(outStatistics.GetPointer(), uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::LoadUserSetting(nn::sf::Out<nn::friends::detail::UserSettingImpl> outSetting,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::UserSettingManager::GetInstance().Read(outSetting.GetPointer(), uid));

    PlayLog localPlayLogs[PlayLogCountMax] = {};
    int localPlayLogCount = 0;

    NN_RESULT_DO(detail::service::core::PlayLogManager::GetInstance().GetPlayLogList(&localPlayLogCount,
        localPlayLogs, uid, NN_ARRAY_SIZE(localPlayLogs)));

    PlayLog* settingsPlayLogs = outSetting.GetPointer()->data.profileExtra.data.playLog;
    int settingsPlayLogCount = 0;

    for (int i = 0; i < PlayLogCountMax; i++)
    {
        if (settingsPlayLogs[i].appInfo.appId == nn::ApplicationId::GetInvalidId())
        {
            break;
        }
        settingsPlayLogCount++;
    }

    detail::service::core::ParameterConverter::MergePlayLog(&settingsPlayLogCount, settingsPlayLogs,
        settingsPlayLogs, settingsPlayLogCount, localPlayLogs, localPlayLogCount, PlayLogCountMax);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SyncUserSetting(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::SyncUserSettingTask, uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::RequestListSummaryOverlayNotification() NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    detail::service::core::BackgroundTaskManager::GetInstance().RequestUpdateListSummary();

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetExternalApplicationCatalog(nn::sf::Out<nn::friends::ExternalApplicationCatalog> outCatalog,
    const nn::friends::ExternalApplicationCatalogId& catalogId, nn::settings::LanguageCode language) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_View);

    TASK_DO(detail::service::core::GetExternalApplicationCatalogTask, nn::account::InvalidUid,
        outCatalog.GetPointer(), catalogId, language);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DropFriendNewlyFlags(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().DropNewlyAll(uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DeleteFriend(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::DeleteFriendTask, uid,
        accountId);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DropFriendNewlyFlag(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FriendListManager::GetInstance().DropNewly(uid, accountId));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ChangeFriendFavoriteFlag(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, bool newValue) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangeFriendFavoriteFlagTask, uid,
        accountId, newValue);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ChangeFriendOnlineNotificationFlag(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, bool newValue) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangeFriendOnlineNotificationFlagTask, uid,
        accountId, newValue);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SendFriendRequest(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t requestType) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::SendFriendRequestTask, uid,
        accountId, static_cast<RequestType>(requestType));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SendFriendRequestWithApplicationInfo(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t requestType,
    const nn::friends::ApplicationInfo& appInfo, const nn::friends::InAppScreenName& inAppScreenName, const nn::friends::InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(inAppScreenName), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(myInAppScreenName), ResultInvalidArgument());

    TASK_DO(detail::service::core::SendFriendRequestTask, uid,
        accountId, static_cast<RequestType>(requestType), appInfo, inAppScreenName, myInAppScreenName);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SendFriendRequestWithExternalApplicationCatalogId(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t requestType,
    const nn::friends::ExternalApplicationCatalogId& catalogId, const nn::friends::InAppScreenName& inAppScreenName, const nn::friends::InAppScreenName& myInAppScreenName) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(inAppScreenName), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(myInAppScreenName), ResultInvalidArgument());

    TASK_DO(detail::service::core::SendFriendRequestTask, uid,
        accountId, static_cast<RequestType>(requestType), catalogId, inAppScreenName, myInAppScreenName);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::SendFriendRequestWithNintendoNetworkIdInfo(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t requestType,
    const nn::friends::MiiName& miiName, const nn::friends::MiiImageUrlParam& miiImageUrlParam, const nn::friends::MiiName& myMiiName, const nn::friends::MiiImageUrlParam& myMiiImageUrlParam) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyString(miiName.value,
        sizeof (miiName.value), MiiNameLengthMax), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyMiiImageUrlParam(miiImageUrlParam),
        ResultInvalidArgument());

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyString(myMiiName.value,
        sizeof (myMiiName.value), MiiNameLengthMax), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyMiiImageUrlParam(myMiiImageUrlParam),
        ResultInvalidArgument());

    TASK_DO(detail::service::core::SendFriendRequestTask, uid,
        accountId, static_cast<RequestType>(requestType), miiName, miiImageUrlParam, myMiiName, myMiiImageUrlParam);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::CancelFriendRequest(const nn::account::Uid& uid, nn::friends::RequestId requestId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangeFriendRequestStatusTask, uid,
        requestId, RequestStatus_Canceled);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::AcceptFriendRequest(const nn::account::Uid& uid, nn::friends::RequestId requestId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangeFriendRequestStatusTask, uid,
        requestId, RequestStatus_Authorized);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::RejectFriendRequest(const nn::account::Uid& uid, nn::friends::RequestId requestId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangeFriendRequestStatusTask, uid,
        requestId, RequestStatus_Rejected);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ReadFriendRequest(const nn::account::Uid& uid, nn::friends::RequestId requestId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ReadFriendRequestTask, uid,
        requestId);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFacedFriendRequestRegistrationKey(nn::sf::Out<nn::friends::FacedFriendRequestRegistrationKey> outKey,
    const nn::account::Uid& uid) NN_NOEXCEPT
{
    // ver.2.0.0
    // CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FacedFriendRequestManager::GetInstance().
        GetFacedFriendRequestRegistrationKey(outKey.GetPointer(), uid));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::AddFacedFriendRequest(const nn::account::Uid& uid,
    const nn::friends::FacedFriendRequestRegistrationKey& key,
    const nn::account::Nickname& nickname, const nn::sf::InBuffer& image) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyString(nickname.name, sizeof (nickname.name)), ResultInvalidArgument());

    NN_RESULT_DO(detail::service::core::FacedFriendRequestManager::GetInstance().AddFacedFriendRequest(uid,
        key, nickname, image.GetPointerUnsafe(), image.GetSize()));

    detail::service::core::BackgroundTaskManager::GetInstance().NotifyFacedFriendRequestAdded(uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::CancelFacedFriendRequest(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_DO(detail::service::core::FacedFriendRequestManager::GetInstance().CancelFacedFriendRequest(uid, accountId));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFacedFriendRequestProfileImage(nn::sf::Out<std::int32_t> outSize,
    const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    size_t size;
    NN_RESULT_DO(detail::service::core::FacedFriendRequestManager::GetInstance().
        GetProfileImage(&size, uid, accountId, buffer.GetPointerUnsafe(), buffer.GetSize()));

    *outSize = static_cast<int32_t>(size);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetFacedFriendRequestProfileImageFromPath(nn::sf::Out<std::int32_t> outSize,
    const nn::sf::InArray<char>& path, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyString(path.GetData(),
        nn::fs::EntryNameLengthMax + 1), ResultInvalidArgument());

    size_t size;
    NN_RESULT_DO(detail::service::core::FacedFriendRequestManager::GetInstance().
        GetProfileImage(&size, path.GetData(), buffer.GetPointerUnsafe(), buffer.GetSize()));

    *outSize = static_cast<int32_t>(size);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ResendFacedFriendRequest(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::SendFacedFriendRequestTask, uid,
        accountId, true);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetSnsAccountLinkPageUrl(nn::sf::Out<nn::friends::WebPageUrl> outUrl,
    const nn::account::Uid& uid, std::int32_t type) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::GetSnsAccountLinkPageUrlTask, uid,
        outUrl.GetPointer(), static_cast<SnsAccountType>(type));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::UnlinkSnsAccount(const nn::account::Uid& uid, std::int32_t type) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::UnlinkSnsAccountTask, uid,
        static_cast<SnsAccountType>(type));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::BlockUser(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t blockReason) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::BlockUserTask, uid,
        accountId, static_cast<BlockReason>(blockReason));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::BlockUserWithApplicationInfo(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId, std::int32_t blockReason,
    const nn::friends::ApplicationInfo& appInfo, const nn::friends::InAppScreenName& inAppScreenName) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_THROW_UNLESS(detail::service::core::DataVerifier::VerifyInAppScreenName(inAppScreenName), ResultInvalidArgument());

    TASK_DO(detail::service::core::BlockUserTask, uid,
        accountId, static_cast<BlockReason>(blockReason), appInfo, inAppScreenName);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::UnblockUser(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::UnblockUserTask, uid,
        accountId);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::GetProfileExtraFromFriendCode(nn::sf::Out<nn::friends::detail::ProfileExtraImpl> outProfile,
    const nn::account::Uid& uid, const nn::friends::FriendCode& friendCode) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::GetProfileListTask, uid,
        outProfile.GetPointer(), friendCode);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DeletePlayHistory(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    NN_RESULT_TRY(detail::service::core::PlayHistoryManager::GetInstance().Delete(uid))
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ChangePresencePermission(const nn::account::Uid& uid, std::int32_t newValue) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangePresencePermissionTask, uid,
        static_cast<PresencePermission>(newValue));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ChangeFriendRequestReception(const nn::account::Uid& uid, bool newValue) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangeFriendRequestReceptionTask, uid,
        newValue);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ChangePlayLogPermission(const nn::account::Uid& uid, std::int32_t newValue) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ChangePlayLogPermissionTask, uid,
        static_cast<ExtraPermission>(newValue));

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::IssueFriendCode(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::IssueFriendCodeTask, uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::ClearPlayLog(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_Management);
    CHECK_UID(uid);

    TASK_DO(detail::service::core::ClearPlayLogTask, uid);

    NN_RESULT_SUCCESS;
}

nn::Result FriendService::DeleteNetworkServiceAccountCache(const nn::account::Uid& uid) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);
    CHECK_UID(uid);

    bool isMounted = true;

    NN_RESULT_TRY(detail::service::core::AccountStorageManager::GetInstance().Mount(uid, false));
        NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
        {
            isMounted = false;
        }
    NN_RESULT_END_TRY;

    if (isMounted)
    {
        NN_UTIL_SCOPE_EXIT
        {
            detail::service::core::AccountStorageManager::GetInstance().Unmount();
        };

        NN_RESULT_DO(detail::service::core::AccountStorageManager::GetInstance().DeleteNetworkServiceAccountDirectory());
    }

    detail::service::core::UserSettingManager::GetInstance().Invalidate();
    detail::service::core::PlayLogManager::GetInstance().Invalidate();
    detail::service::core::FriendListManager::GetInstance().Invalidate();
    detail::service::core::FacedFriendRequestManager::GetInstance().Invalidate();
    detail::service::core::BlockedUserListManager::GetInstance().Invalidate();

    NN_RESULT_SUCCESS;
}

}}}}
