﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskSyncFriendListBase.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerFriendResourceList.h>
#include <nn/friends/detail/service/core/friends_FriendListManager.h>
#include <nn/friends/detail/service/core/friends_FriendPresenceManager.h>
#include <nn/friends/detail/service/core/friends_ImageDownloadQueueManager.h>
#include <nn/friends/detail/service/core/friends_ImageManager.h>
#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>
#include <nn/friends/detail/service/core/friends_WorkBuffer.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

nn::Result SyncFriendListBaseTask::StepSyncFriendList() NN_NOEXCEPT
{
    NN_RESULT_DO(StepUpdateFriendInfo());
    NN_RESULT_DO(StepSyncFriendListImpl());

    NN_RESULT_SUCCESS;
}

nn::Result SyncFriendListBaseTask::StepUpdateFriendInfo() NN_NOEXCEPT
{
    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    FriendListManager::Record* records = reinterpret_cast<FriendListManager::Record*>(WorkBuffer::GetInstance().Get());
    int count;

    NN_RESULT_DO(FriendListManager::GetInstance().GetLocalUpdatedRecord(&count, records, GetUid(), FriendCountMax));

    for (int i = 0; i < count; i++)
    {
        NN_DETAIL_FRIENDS_INFO("[friends] SyncFriendListTask::StepUpdateFriendInfo(%016llx, %016llx) ...\n",
            accountId.id, records[i].accountId.id);

        detail::service::json::JsonHttpInputStream stream;

        char url[160] = {};
        nn::util::SNPrintf(url, sizeof (url),
            "https://%s/2.0.0/users/%016llx/friends/%016llx", WebApi::GetFqdn(), accountId.id, records[i].accountId.id);

        char patch[1024] = {};
        NN_RESULT_DO(CreateUpdateFriendInfoPatch(records[i], patch, sizeof (patch)));

        NN_RESULT_DO(stream.Open("PATCH", url));
        NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
        NN_RESULT_DO(stream.SetPostField(patch, false));

        detail::service::json::JsonEventHandler handler;

        NN_RESULT_TRY(WebApi::Call(handler, stream, GetUid(), GetCancelable()))
            NN_RESULT_CATCH(ResultServerError404ResourceIsNotFound)
            {
            }
            NN_RESULT_CATCH(ResultNetworkServiceAccountNotExists)
            {
            }
        NN_RESULT_END_TRY;
    }

    NN_RESULT_SUCCESS;
}

nn::Result SyncFriendListBaseTask::CreateUpdateFriendInfoPatch(const FriendListManager::Record& record,
    char* buffer, size_t size) NN_NOEXCEPT
{
    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        if (record.isNewlyConfirmed)
        {
            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/isConfirmed");
            {
                writer.Bool(true);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
        }

        if (record.isLastPlayRecordUpdated)
        {
            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:appInfo:appId");
            {
                char idString[17];
                nn::util::SNPrintf(idString, sizeof (idString), "%016llx", record.lastPlayRecord.appInfo.appId.value);

                writer.String(idString);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:appInfo:presenceGroupId");
            {
                char idString[17];
                nn::util::SNPrintf(idString, sizeof (idString), "%016llx", record.lastPlayRecord.appInfo.presenceGroupId);

                writer.String(idString);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:name");
            {
                writer.String(record.lastPlayRecord.name.name);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:name:language");
            {
                writer.String(record.lastPlayRecord.name.language.string);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:myName");
            {
                writer.String(record.lastPlayRecord.myName.name);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:myName:language");
            {
                writer.String(record.lastPlayRecord.myName.language.string);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

            // 更新時刻は記録しない。
/*
            detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/extras/self/lastPlayRecord:playedAt");
            {
                writer.Int64(record.lastPlayRecord.time.value);
            }
            detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
*/
        }
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

nn::Result SyncFriendListBaseTask::StepSyncFriendListImpl() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] SyncFriendListBaseTask::StepSyncFriendList ...\n");

    NN_RESULT_THROW_UNLESS(WorkBuffer::GetInstance().Acquire(GetCancelable()), ResultCanceled());

    NN_UTIL_SCOPE_EXIT
    {
        WorkBuffer::GetInstance().Release();
    };

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/2.0.0/users/%016llx/friends?count=%d", WebApi::GetFqdn(), accountId.id, FriendCountMax);

    NN_RESULT_DO(stream.Open("GET", url));

    FriendResourceListHandler handler;
    NN_RESULT_DO(handler.Initialize(WorkBuffer::GetInstance().Get(), WorkBuffer::GetInstance().GetSize()));

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_DO(FriendListManager::GetInstance().UpdateFriendList(GetUid(),
        handler.GetResourceList(), handler.GetResourceCount()));

    FriendPresenceManager::GetInstance().UpdateFriendList(GetUid(), handler.GetResourceList(), handler.GetResourceCount());

    bool isAdded = false;

    int count = handler.GetResourceCount();

    for (int i = 0; i < count; i++)
    {
        const FriendResource& resource = handler.GetResourceList()[i];

        if (ImageManager::GetInstance().CanAddOrUpdate(resource.accountId, resource.profileImageUrl))
        {
            NN_RESULT_DO(ImageDownloadQueueManager::GetInstance().Add(GetUid(), resource.accountId));
            isAdded = true;
        }

        NN_RESULT_THROW_UNLESS(!GetCancelable().IsCanceled(), ResultCanceled());
    }

    if (isAdded)
    {
        NN_RESULT_DO(ImageDownloadQueueManager::GetInstance().Save());
        BackgroundTaskManager::GetInstance().RequestUpdateFriendProfileImageCache(GetUid());
    }

    NN_RESULT_SUCCESS;
}

}}}}}
