﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_TaskChangeFriendRequestStatus.h>
#include <nn/friends/detail/service/core/handler/friends_HandlerFriendRequestResource.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

ChangeFriendRequestStatusTask::ChangeFriendRequestStatusTask() NN_NOEXCEPT :
    m_IsParameterSet(false),
    m_IsSyncNeeded(false),
    m_IsGetCountNeeded(false)
{
}

nn::Result ChangeFriendRequestStatusTask::SetParameter(RequestId requestId, RequestStatus newStatus) NN_NOEXCEPT
{
    m_RequestId = requestId;
    m_Status = newStatus;

    m_IsParameterSet = true;

    NN_RESULT_SUCCESS;
}

nn::Result ChangeFriendRequestStatusTask::Main() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsParameterSet);

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepChange());

    if (m_IsSyncNeeded)
    {
        NN_RESULT_DO(StepSyncFriendList());
    }
    if (m_IsGetCountNeeded)
    {
        NN_RESULT_DO(StepGetReceivedFriendRequestCount());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ChangeFriendRequestStatusTask::StepChange() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] ChangeFriendRequestStatusTask::StepChange(%s) ...\n",
        ParameterConverter::ConvertRequestStatus(m_Status));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/2.0.0/friend_requests/%016llx", WebApi::GetFqdn(), m_RequestId.value);

    char patch[128] = {};
    NN_RESULT_DO(CreateChangePatch(patch, sizeof (patch)));

    NN_RESULT_DO(stream.Open("PATCH", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
    NN_RESULT_DO(stream.SetPostField(patch, false));

    FriendRequestResourceHandler handler;
    handler.Initialize(m_Status == RequestStatus_Canceled ? RequestListType_Sent : RequestListType_Received);

    NN_RESULT_TRY(WebApi::Call(handler, stream, GetUid(), GetCancelable()))
        NN_RESULT_CATCH(ResultServerError400InvalidOperation)
        {
            NN_RESULT_THROW(ResultRequestStateAlreadyChanged());
        }
        NN_RESULT_CATCH(ResultServerError400SenderFriendCapacityIsFull)
        {
            NN_RESULT_THROW(ResultFriendListLimitReached());
        }
        NN_RESULT_CATCH(ResultServerError400ReceiverFriendCapacityIsFull)
        {
            NN_RESULT_THROW(ResultMyFriendListLimitReached());
        }
        NN_RESULT_CATCH(ResultServerError404ResourceIsNotFound)
        {
            NN_RESULT_THROW(ResultRequestNotFound());
        }
    NN_RESULT_END_TRY;

    if (handler.GetResource().requestStatus == RequestStatus_Authorized)
    {
        m_IsSyncNeeded = true;
    }
    if (m_Status == RequestStatus_Authorized || m_Status == RequestStatus_Rejected)
    {
        m_IsGetCountNeeded = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ChangeFriendRequestStatusTask::CreateChangePatch(char* buffer, size_t size) NN_NOEXCEPT
{
    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        detail::service::json::JsonPatchGenerator::BeginWriteReplaceOperation(writer, "/state");
        {
            writer.String(ParameterConverter::ConvertRequestStatus(m_Status));
        }
        detail::service::json::JsonPatchGenerator::EndWriteReplaceOperation(writer);
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

}}}}}
