﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/task/friends_BgTaskUpdateUserPresence.h>
#include <nn/friends/detail/service/core/friends_UserPresenceManager.h>
#include <nn/friends/detail/friends_PresenceAccessor.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

UpdateUserPresenceBgTask::UpdateUserPresenceBgTask() NN_NOEXCEPT
{
}

nn::Result UpdateUserPresenceBgTask::Main() NN_NOEXCEPT
{
    NN_RESULT_TRY(UserPresenceManager::GetInstance().GetPresence(&m_Presence, GetUid()))
        NN_RESULT_CATCH_ALL
        {
            // 更新しようとしたユーザーは既にいない。
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(Account::EnsureNetworkServiceAccessToken(GetUid(), GetCancelable()));

    NN_RESULT_DO(StepUpdate());

    NN_RESULT_SUCCESS;
}

nn::Result UpdateUserPresenceBgTask::StepUpdate() NN_NOEXCEPT
{
    NN_DETAIL_FRIENDS_INFO("[friends] UpdateUserPresenceBgTask::StepUpdate ...\n");

    nn::account::NetworkServiceAccountId accountId = {};
    NN_RESULT_DO(Account::GetNetworkServiceAccountId(&accountId, GetUid()));

    uint64_t deviceAccountId = 0;
    NN_RESULT_DO(Account::GetDeviceAccountId(&deviceAccountId, GetUid()));

    detail::service::json::JsonHttpInputStream stream;

    char url[160] = {};
    nn::util::SNPrintf(url, sizeof (url),
        "https://%s/1.0.0/users/%016llx/device_accounts/%016llx", WebApi::GetFqdn(), accountId.id, deviceAccountId);

    char patch[1024] = {};
    NN_RESULT_DO(CreateUpdatePatch(patch, sizeof (patch)));

    NN_RESULT_DO(stream.Open("PATCH", url));
    NN_RESULT_DO(stream.AddRequestHeader("Content-Type: application/json-patch+json"));
    NN_RESULT_DO(stream.SetPostField(patch, false));

    detail::service::json::JsonEventHandler handler;

    NN_RESULT_DO(WebApi::Call(handler, stream, GetUid(), GetCancelable()));

    NN_RESULT_SUCCESS;
}

nn::Result UpdateUserPresenceBgTask::CreateUpdatePatch(char* buffer, size_t size) NN_NOEXCEPT
{
    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    detail::service::json::JsonPatchGenerator::BeginWrite(writer);
    {
        detail::service::json::JsonPatchGenerator::BeginWriteReplaceOperation(writer, "/presence/state");
        {
            writer.String(ParameterConverter::
                ConvertPresenceStatus(static_cast<PresenceStatus>(m_Presence.data.status)));
        }
        detail::service::json::JsonPatchGenerator::EndWriteReplaceOperation(writer);

        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/presence/extras/friends/appField");
        {
            // キー名 2 バイト＋バリュー 1 バイト = 3
            NN_STATIC_ASSERT(PresenceAppFieldSize / 3 == 64);

            detail::PresenceAccessor::KeyValue keyValues[64];
            int count;

            NN_RESULT_DO(detail::PresenceAccessor::GetKeyValueList(&count, keyValues, 64,
                m_Presence.data.appField, sizeof (m_Presence.data.appField)));

            // キー 1 文字、バリュー 0 文字が一番バッファを消費する。
            // その時の消費バッファは「"k":"",」で 1 つあたり 7 バイトとなる。(7 * 64 = 448 バイトが最低限必要）
            char appFieldString[512] = {};

            int written = nn::util::SNPrintf(appFieldString, sizeof (appFieldString), "{");

            for (int i = 0; i < count; i++)
            {
                // キーにエスケープすべき文字は入らない。
                written += nn::util::SNPrintf(&appFieldString[written], sizeof (appFieldString) - written,
                    "\"%s\":", keyValues[i].key);

                written += EscapeString(&appFieldString[written], sizeof (appFieldString) - written, keyValues[i].value);

                if (i != count - 1)
                {
                    written += nn::util::SNPrintf(&appFieldString[written], sizeof (appFieldString) - written, ",");
                }
            }

            nn::util::SNPrintf(&appFieldString[written], sizeof (appFieldString) - written, "}");

            writer.String(appFieldString);
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/presence/extras/friends/appInfo:appId");
        {
            char idString[17];
            nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_Presence.data.lastPlayedApp.appId.value);

            writer.String(idString);
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);

        detail::service::json::JsonPatchGenerator::BeginWriteAddOperation(writer, "/presence/extras/friends/appInfo:presenceGroupId");
        {
            char idString[17];
            nn::util::SNPrintf(idString, sizeof (idString), "%016llx", m_Presence.data.lastPlayedApp.presenceGroupId);

            writer.String(idString);
        }
        detail::service::json::JsonPatchGenerator::EndWriteAddOperation(writer);
    }
    detail::service::json::JsonPatchGenerator::EndWrite(writer);

    stream.PutEnd();
    stream.Close();

    NN_RESULT_SUCCESS;
}

int UpdateUserPresenceBgTask::EscapeString(char* buffer, size_t size, const char* value) NN_NOEXCEPT
{
    detail::service::json::FixedSizeAllocator allocator;
    detail::service::json::JsonMemoryOutputStream stream;

    stream.Open(buffer, size);
    stream.PutBegin();

    detail::service::json::JsonWriter writer(stream, &allocator, 16);

    writer.String(value);

    stream.PutEnd();
    stream.Close();

    return nn::util::Strnlen(buffer, static_cast<int>(size));
}

}}}}}
