﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_ParameterConverter.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

PresenceStatus ParameterConverter::ConvertPresenceStatus(const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);

    if (nn::util::Strncmp(value, "OFFLINE", sizeof ("OFFLINE")) == 0)
    {
        return PresenceStatus_Offline;
    }
    if (nn::util::Strncmp(value, "ONLINE", sizeof ("ONLINE")) == 0)
    {
        return PresenceStatus_Online;
    }
    if (nn::util::Strncmp(value, "PLAYING", sizeof ("PLAYING")) == 0)
    {
        return PresenceStatus_OnlinePlay;
    }
    if (nn::util::Strncmp(value, "INACTIVE", sizeof ("INACTIVE")) == 0)
    {
        // INACTIVE はオフライン状態に見せる。
        return PresenceStatus_Offline;
    }

    return PresenceStatus_Offline;
}

const char* ParameterConverter::ConvertPresenceStatus(PresenceStatus value) NN_NOEXCEPT
{
    switch (value)
    {
    case PresenceStatus_Offline:
        // ユーザープレゼンスがオフライン状態になった時、サーバーには INACTIVE を送信する。
        return "INACTIVE";
    case PresenceStatus_Online:
        return "ONLINE";
    case PresenceStatus_OnlinePlay:
        return "PLAYING";
    default:
        return "-";
    }
}

RequestType ParameterConverter::ConvertRequestType(const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);

    if (nn::util::Strncmp(value, "NX_FACED", sizeof ("NX_FACED")) == 0)
    {
        return RequestType_Faced;
    }
    if (nn::util::Strncmp(value, "FRIEND_CODE", sizeof ("FRIEND_CODE")) == 0)
    {
        return RequestType_FriendCode;
    }
    if (nn::util::Strncmp(value, "IN_APP", sizeof ("IN_APP")) == 0)
    {
        return RequestType_InApp;
    }
    if (nn::util::Strncmp(value, "NINTENDO_ACCOUNT", sizeof ("NINTENDO_ACCOUNT")) == 0)
    {
        return RequestType_FriendCandidate;
    }
    if (nn::util::Strncmp(value, "3DS", sizeof ("3DS")) == 0)
    {
        return RequestType_3Ds;
    }
    if (nn::util::Strncmp(value, "NNID", sizeof ("NNID")) == 0)
    {
        return RequestType_Nnid;
    }
    if (nn::util::Strncmp(value, "FACEBOOK", sizeof ("FACEBOOK")) == 0)
    {
        return RequestType_Facebook;
    }
    if (nn::util::Strncmp(value, "TWITTER", sizeof ("TWITTER")) == 0)
    {
        return RequestType_Twitter;
    }

    return RequestType_Unknown;
}

const char* ParameterConverter::ConvertRequestType(RequestType value) NN_NOEXCEPT
{
    switch (value)
    {
    case RequestType_Faced:
        return "NX_FACED";
    case RequestType_FriendCode:
        return "FRIEND_CODE";
    case RequestType_InApp:
        return "IN_APP";
    case RequestType_FriendCandidate:
        return "NINTENDO_ACCOUNT";
    case RequestType_3Ds:
        return "3DS";
    case RequestType_Nnid:
        return "NNID";
    case RequestType_Facebook:
        return "FACEBOOK";
    case RequestType_Twitter:
        return "TWITTER";
    default:
        return "-";
    }
}

const char* ParameterConverter::ConvertRequestStatus(RequestStatus value) NN_NOEXCEPT
{
    switch (value)
    {
    case RequestStatus_Pending:
        return "PENDING";
    case RequestStatus_Canceled:
        return "CANCELED";
    case RequestStatus_Authorized:
        return "AUTHORIZED";
    case RequestStatus_Rejected:
        return "REJECTED";
    case RequestStatus_Expired:
        return "EXPIRED";
    default:
        return "-";
    }
}

RequestStatus ParameterConverter::ConvertRequestStatus(const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);

    if (nn::util::Strncmp(value, "PENDING", sizeof ("PENDING")) == 0)
    {
        return RequestStatus_Pending;
    }
    if (nn::util::Strncmp(value, "CANCELED", sizeof ("CANCELED")) == 0)
    {
        return RequestStatus_Canceled;
    }
    if (nn::util::Strncmp(value, "AUTHORIZED", sizeof ("AUTHORIZED")) == 0)
    {
        return RequestStatus_Authorized;
    }
    if (nn::util::Strncmp(value, "REJECTED", sizeof ("REJECTED")) == 0)
    {
        return RequestStatus_Rejected;
    }
    if (nn::util::Strnicmp(value, "EXPIRED", sizeof ("EXPIRED")) == 0)
    {
        return RequestStatus_Expired;
    }

    return RequestStatus_Unknown;
}

const char* ParameterConverter::ConvertPresencePermission(PresencePermission value) NN_NOEXCEPT
{
    switch (value)
    {
    case PresencePermission_Self:
        return "SELF";
    case PresencePermission_Friends:
        return "FRIENDS";
    case PresencePermission_FavoriteFriends:
        return "FAVORITE_FRIENDS";
    default:
        return "-";
    }
}

PresencePermission ParameterConverter::ConvertPresencePermission(const char* value) NN_NOEXCEPT
{
    if (nn::util::Strncmp(value, "SELF", sizeof ("SELF")) == 0)
    {
        return PresencePermission_Self;
    }
    if (nn::util::Strncmp(value, "FRIENDS", sizeof ("FRIENDS")) == 0)
    {
        return PresencePermission_Friends;
    }
    if (nn::util::Strncmp(value, "FAVORITE_FRIENDS", sizeof ("FAVORITE_FRIENDS")) == 0)
    {
        return PresencePermission_FavoriteFriends;
    }

    return PresencePermission_Self;
}

const char* ParameterConverter::ConvertExtraPermission(ExtraPermission value) NN_NOEXCEPT
{
    switch (value)
    {
    case ExtraPermission_Self:
        return "self";
    case ExtraPermission_FavoriteFriends:
        return "favoriteFriends";
    case ExtraPermission_Friends:
        return "friends";
    case ExtraPermission_FriendsOfFriend:
        return "foaf";
    case ExtraPermission_Everyone:
        return "everyone";
    default:
        return "self";
    }
}

BlockReason ParameterConverter::ConvertBlockReason(const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);

    if (nn::util::Strncmp(value, "BAD_FRIEND_REQUEST", sizeof ("BAD_FRIEND_REQUEST")) == 0)
    {
        return BlockReason_BadFriendRequest;
    }
    if (nn::util::Strncmp(value, "BAD_FRIEND", sizeof ("BAD_FRIEND")) == 0)
    {
        return BlockReason_BadFriend;
    }
    if (nn::util::Strncmp(value, "IN_APP", sizeof ("IN_APP")) == 0)
    {
        return BlockReason_InApp;
    }

    return BlockReason_Unknown;
}

const char* ParameterConverter::ConvertBlockReason(BlockReason value) NN_NOEXCEPT
{
    switch (value)
    {
    case BlockReason_BadFriendRequest:
        return "BAD_FRIEND_REQUEST";
    case BlockReason_BadFriend:
        return "BAD_FRIEND";
    case BlockReason_InApp:
        return "IN_APP";
    default:
        return "";
    }
}

bool ParameterConverter::ConvertExternalApplicationCatalogId(ExternalApplicationCatalogId* out, const char* value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);
    NN_SDK_REQUIRES_NOT_NULL(value);

    unsigned long long v[2];

    if (std::sscanf(value, "%016llx%016llx", &v[0], &v[1]) != 2)
    {
        return false;
    }

    out->value[0] = static_cast<Bit64>(v[0]);
    out->value[1] = static_cast<Bit64>(v[1]);

    return true;
}

char* ParameterConverter::ConvertPlayLog(char* buffer, size_t size, const PlayLog* playLogs, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    size_t written = 0;

    written += nn::util::SNPrintf(&buffer[written], size - written, "[");
    NN_ABORT_UNLESS(written < size);

    for (int i = 0; i < count; i++)
    {
        written += nn::util::SNPrintf(&buffer[written], size - written, "{");
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written,
            "\"appInfo:appId\":\"%016llx\",", playLogs[i].appInfo.appId.value);
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written,
            "\"appInfo:presenceGroupId\":\"%016llx\",", playLogs[i].appInfo.presenceGroupId);
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written,
            "\"totalPlayCount\":%d,", playLogs[i].totalPlayCount);
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written,
            "\"totalPlayTime\":%d,", playLogs[i].totalPlayTime);
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written,
            "\"firstPlayedAt\":%lld,", playLogs[i].firstPlayedTime.value);
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written,
            "\"lastPlayedAt\":%lld", playLogs[i].lastPlayedTime.value);
        NN_ABORT_UNLESS(written < size);

        written += nn::util::SNPrintf(&buffer[written], size - written, "}");
        NN_ABORT_UNLESS(written < size);

        if (i != count - 1)
        {
            written += nn::util::SNPrintf(&buffer[written], size - written, ",");
            NN_ABORT_UNLESS(written < size);
        }
    }

    written += nn::util::SNPrintf(&buffer[written], size - written, "]");
    NN_ABORT_UNLESS(written < size);

    return buffer;
}

void ParameterConverter::MergePlayLog(int* outCount, PlayLog* outMergedPlayLogs,
    const PlayLog* basePlayLogs, int baseCount, const PlayLog* addPlayLogs, int addCount, int mergedCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outMergedPlayLogs);
    NN_SDK_REQUIRES(mergedCount > 0);

    if (baseCount > PlayLogCountMax)
    {
        baseCount = PlayLogCountMax;
    }

    PlayLog playLogs[PlayLogCountMax] = {};
    int count = 0;

    for (int i = 0; i < baseCount; i++)
    {
        playLogs[i] = basePlayLogs[i];
    }
    count = baseCount;

    for (int a = addCount - 1; a >= 0; a--)
    {
        int index = -1;

        for (int i = 0; i < count; i++)
        {
            if (addPlayLogs[a].appInfo.appId == playLogs[i].appInfo.appId)
            {
                index = i;
                break;
            }
        }

        if (index != -1)
        {
            PlayLog mergedPlayLog = playLogs[index];

            // 総プレイ回数は、より多い方。
            if (mergedPlayLog.totalPlayCount < addPlayLogs[a].totalPlayCount)
            {
                mergedPlayLog.totalPlayCount = addPlayLogs[a].totalPlayCount;
            }
            // 総プレイ時間は、より長い方。
            if (mergedPlayLog.totalPlayTime < addPlayLogs[a].totalPlayTime)
            {
                mergedPlayLog.totalPlayTime = addPlayLogs[a].totalPlayTime;
            }
            // 最初に遊んだ時間は、より早い方。
            if (mergedPlayLog.firstPlayedTime.value > addPlayLogs[a].firstPlayedTime.value)
            {
                mergedPlayLog.firstPlayedTime = addPlayLogs[a].firstPlayedTime;
            }
            // 最後に遊んだ時間は、より遅い方。
            if (mergedPlayLog.lastPlayedTime.value < addPlayLogs[a].lastPlayedTime.value)
            {
                mergedPlayLog.lastPlayedTime = addPlayLogs[a].lastPlayedTime;
            }

            count = detail::service::util::ArrayAccessor::RemoveEntry(playLogs, count, index);
            count = detail::service::util::ArrayAccessor::InsertToTopEntry(playLogs, count, PlayLogCountMax, mergedPlayLog);
        }
        else
        {
            count = detail::service::util::ArrayAccessor::InsertToTopEntry(playLogs, count, PlayLogCountMax, addPlayLogs[a]);
        }
    }

    int copyCount = std::min(count, mergedCount);

    for (int i = 0; i < copyCount; i++)
    {
        outMergedPlayLogs[i] = playLogs[i];
    }

    *outCount = copyCount;
}

}}}}}
