﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_ImageDownloader.h>
#include <nn/friends/friends_Result.h>
#include <nn/friends/friends_ResultPrivate.h>
#include <nn/friends/detail/service/util/friends_HttpErrorHandler.h>
#include <nn/ssl/ssl_Context.h>

#define CURL_EASY_DO(exp) \
    do                                                              \
    {                                                               \
        CURLcode code_ = (exp);                                     \
                                                                    \
        if (code_ != CURLE_OK)                                      \
        {                                                           \
            return detail::service::util::HandleHttpError(code_);   \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

ImageDownloader::ImageDownloader() NN_NOEXCEPT :
    m_Cancelable(nullptr),
    m_DownloadedSize(0)
{
}

void ImageDownloader::SetCancelable(const detail::service::util::Cancelable* cancelable) NN_NOEXCEPT
{
    m_Cancelable = cancelable;
}

nn::Result ImageDownloader::Perform(const char* url, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(url);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size > 0);

    HttpFunctionParam param = {};

    param.buffer = reinterpret_cast<Bit8*>(buffer);
    param.bufferSize = size;

    CURL* curl = curl_easy_init();

    NN_RESULT_THROW_UNLESS(curl, ResultHttpErrorInterfaceFailed());

    NN_UTIL_SCOPE_EXIT
    {
        if (curl)
        {
            curl_easy_cleanup(curl);
        }
    };

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_URL, url));

    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, HttpHeaderFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_HEADERDATA, &param));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_WRITEDATA, &param));

    // CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_VERBOSE, 1));

    // 最大リダイレクト回数を制限する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_MAXREDIRS, 1));
    // 1 B/s 以下の転送速度の通信が 30 秒以上続いたら切断する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_LIMIT, 1));
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_LOW_SPEED_TIME, 30));
    // 接続タイムアウト（CONNECT 完了までの時間）を 30 秒にする。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_CONNECTTIMEOUT, 30));
    // 証明書の時刻検証を無効化する。
    CURL_EASY_DO(curl_easy_setopt(curl, CURLOPT_SSL_VERIFYDATE, 0));

    CURLM* multi = curl_multi_init();

    NN_RESULT_THROW_UNLESS(multi, ResultHttpErrorOutOfMemory());

    NN_UTIL_SCOPE_EXIT
    {
        if (multi)
        {
            curl_multi_cleanup(multi);
        }
    };

    CURLMcode code = curl_multi_add_handle(multi, curl);

    if (code != CURLM_OK)
    {
        if (code == CURLM_OUT_OF_MEMORY)
        {
            NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
        }
        else
        {
            NN_RESULT_THROW(ResultHttpErrorFailedInit());
        }
    }

    NN_UTIL_SCOPE_EXIT
    {
        curl_multi_remove_handle(multi, curl);
    };

    NN_RESULT_DO(PerformImpl(multi));

    NN_RESULT_THROW_UNLESS(param.is200Ok && param.isJpeg, ResultIvalidResponse());

    m_DownloadedSize = param.downloadedSize;

    NN_RESULT_SUCCESS;
}

size_t ImageDownloader::GetDownloadedSize() const NN_NOEXCEPT
{
    return m_DownloadedSize;
}

nn::Result ImageDownloader::PerformImpl(CURLM* multi) NN_NOEXCEPT
{
    int stillRunning = 0;

    do
    {
        int rval = 0;
        CURLMcode code = curl_multi_wait(multi, nullptr, 0, 10, &rval);

        if (code != CURLM_OK)
        {
            if (code == CURLM_OUT_OF_MEMORY)
            {
                NN_RESULT_THROW(ResultHttpErrorOutOfMemory());
            }
            else
            {
                NN_RESULT_THROW(ResultHttpErrorBadFunctionArgument());
            }
        }

        NN_RESULT_THROW_UNLESS(rval != -1, ResultHttpErrorRecvError());

        curl_multi_perform(multi, &stillRunning);

        if (m_Cancelable && m_Cancelable->IsCanceled())
        {
            NN_RESULT_THROW(ResultCanceled());
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    while (stillRunning > 0);

    CURLMsg* msg = nullptr;

    do
    {
        int n = 0;
        msg = curl_multi_info_read(multi, &n);

        if (msg && (msg->msg == CURLMSG_DONE))
        {
            return detail::service::util::HandleHttpError(msg->data.result);
        }
    }
    while (msg);

    NN_RESULT_SUCCESS;
}

CURLcode ImageDownloader::SslCtxFunction(CURL* curl, void* ssl, void* param) NN_NOEXCEPT
{
    NN_UNUSED(param);

    nn::ssl::Context* context = reinterpret_cast<nn::ssl::Context*>(ssl);

    if (context->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;
}

size_t ImageDownloader::HttpHeaderFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    HttpFunctionParam* httpParam = static_cast<HttpFunctionParam*>(param);

    size_t bufferSize = size * count;

    if (nn::util::Strncmp(buffer, "HTTP/", 5) == 0)
    {
        int majorVersion = 0;
        int minorVersion = 0;
        int statusCode = 0;

        char space = '\0';

        int result = std::sscanf(buffer, "HTTP/%1d.%1d %03d%c", &majorVersion, &minorVersion, &statusCode, &space);

        if (result == 4 && space == ' ')
        {
            if (statusCode == 200)
            {
                httpParam->is200Ok = true;
            }
        }

        return bufferSize;
    }

    char* label = buffer;
    char* colon = strchr(label, ':');

    if (!colon)
    {
        return bufferSize;
    }

    char* value = colon + 1;

    while (*value == ' ')
    {
        value++;
    }

    char* cr = strchr(value, '\r');

    if (!cr)
    {
        return bufferSize;
    }

    *colon = '\0';
    *cr = '\0';

    if (nn::util::Strncmp(label, "Content-Type", sizeof ("Content-Type")) == 0)
    {
        if (nn::util::Strncmp(value, "image/jpeg", sizeof ("image/jpeg")) == 0)
        {
            httpParam->isJpeg = true;
        }
    }

    *colon = ':';
    *cr = '\r';

    return bufferSize;
}

size_t ImageDownloader::HttpWriteFunction(char* buffer, size_t size, size_t count, void* param) NN_NOEXCEPT
{
    HttpFunctionParam* httpParam = static_cast<HttpFunctionParam*>(param);

    size_t bufferSize = size * count;

    if (httpParam->downloadedSize + bufferSize > httpParam->bufferSize)
    {
        httpParam->isJpeg = false;
    }

    if (httpParam->isJpeg)
    {
        std::memcpy(&httpParam->buffer[httpParam->downloadedSize], buffer, bufferSize);
        httpParam->downloadedSize += bufferSize;
    }

    return bufferSize;
}

}}}}}
