﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/friends/detail/service/core/friends_ImageDownloadQueueManager.h>

namespace nn { namespace friends { namespace detail { namespace service { namespace core {

namespace
{
    const char* FileName = "image.queue";
}

ImageDownloadQueueManager::ImageDownloadQueueManager() NN_NOEXCEPT :
    m_Mutex(true),
    m_Count(0),
    m_IsDirty(false)
{
}

nn::Result ImageDownloadQueueManager::Add(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    for (int i = 0; i < m_Count; i++)
    {
        if (accountId == m_Records[i].accountId)
        {
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW_UNLESS(m_Count < FriendCountMax, ResultOutOfResource());

    m_Records[m_Count++].accountId = accountId;
    m_IsDirty = true;

    NN_RESULT_SUCCESS;
}

nn::Result ImageDownloadQueueManager::Remove(const nn::account::Uid& uid, nn::account::NetworkServiceAccountId accountId) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    int count = detail::service::util::ArrayAccessor::RemoveEntry(m_Records, m_Count, accountId,
        [](const Record& lhs, const nn::account::NetworkServiceAccountId& rhs)
        {
            return lhs.accountId == rhs;
        });

    if (count < m_Count)
    {
        m_Count = count;
        m_IsDirty = true;
    }

    NN_RESULT_SUCCESS;
}

nn::Result ImageDownloadQueueManager::Pop(nn::account::NetworkServiceAccountId* outAccountId, const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAccountId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(Load(uid));

    // TODO: ResultNotFound
    NN_RESULT_THROW_UNLESS(m_Count > 0, ResultNetworkServiceAccountNotExistsGeneral());

    *outAccountId = m_Records[0].accountId;

    NN_RESULT_SUCCESS;
}

nn::Result ImageDownloadQueueManager::Save() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_CurrentUid || !m_IsDirty)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(m_CurrentUid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    if (m_Count > 0)
    {
        NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (Record) * m_Count));

        nn::fs::FileHandle handle = {};
        NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, m_Records, sizeof (Record) * m_Count,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_RESULT_DO(AccountStorageManager::GetInstance().Commit());

    m_IsDirty = false;

    NN_RESULT_SUCCESS;
}

nn::Result ImageDownloadQueueManager::Load(const nn::account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_TRY(LoadImpl(uid))
        NN_RESULT_CATCH_ALL
        {
            m_Count = 0;
        }
    NN_RESULT_END_TRY;

    m_CurrentUid = uid;

    NN_RESULT_SUCCESS;
}

nn::Result ImageDownloadQueueManager::LoadImpl(const nn::account::Uid& uid) NN_NOEXCEPT
{
    if (uid == m_CurrentUid)
    {
        NN_RESULT_SUCCESS;
    }

    NN_DETAIL_FRIENDS_ACCOUNT_STORAGE_SCOPED_LOCK(uid);

    char path[64];
    NN_RESULT_DO(AccountStorageManager::GetInstance().
        MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), FileName));

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t size;
    NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    if ((size % sizeof (Record)) != 0)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. ((size % sizeof (Record)) != 0)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    int count = static_cast<int>(size / sizeof (Record));

    if (count > FriendCountMax)
    {
        NN_DETAIL_FRIENDS_WARN("[friends] %s is corrupted. (count > FriendCountMax)\n", FileName);
        NN_RESULT_THROW(ResultSaveDataCorrupted());
    }

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, m_Records, sizeof (Record) * count));

    m_Count = count;

    NN_RESULT_SUCCESS;
}

}}}}}
