﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fgm/fgm_Types.h>

#include "fgm_RequestManager.h"
#include "fgm_Request.h"

namespace nn     {
namespace fgm    {
namespace server {

nn::Result RequestManager::UpdateOperatingRange()
NN_NOEXCEPT
{
    nn::Result result;
    Setting    min = m_Min;
    Setting    max = m_Max;

    for (auto itr  = m_Requests.cbegin();
              itr != m_Requests.cend();
              itr++)
    {
        if ((result = OperatingRange(min, max, itr->GetMin(), itr->GetMax(), &min, &max)).IsFailure())
        {
            return result;
        }
        if (min == max)
        {
            break;
        }
    }

    m_CurrentMin = min;
    m_CurrentMax = max;

    return ResultSuccess();
}

nn::Result RequestManager::Add(Request* pRequestIn)
NN_NOEXCEPT
{
    for (auto itr  = m_Requests.crbegin();
              itr != m_Requests.crend();
              itr++)
    {
        // requests are priority ordered in the list
        // higher priority requests are placed on front
        // newer requests are placed on front on each priority
        if (itr->GetPriority() < pRequestIn->GetPriority())
        {
            m_Requests.insert(itr.base(), *pRequestIn);
            return ResultSuccess();
        }
    }

    m_Requests.push_front(*pRequestIn);

    return ResultSuccess();
}

nn::Result RequestManager::Delete(Request* pRequestIn)
NN_NOEXCEPT
{
    nn::Result result = nn::fgm::ResultRequestNotFound();

    for (auto itr  = m_Requests.cbegin();
              itr != m_Requests.cend();
              itr++)
    {
        if (&(*itr) == pRequestIn)
        {
            m_Requests.erase(itr);
            result = UpdateOperatingRange();
            break;
        }
    }

    return result;
}

nn::Result RequestManager::Update(Request* pRequestIn, Setting min, Setting max)
NN_NOEXCEPT
{
    nn::Result result = nn::fgm::ResultRequestNotFound();

    for (auto itr  = m_Requests.cbegin();
              itr != m_Requests.cend();
              itr++)
    {
        if (&(*itr) == pRequestIn)
        {
            m_Requests.erase(itr);
            pRequestIn->Update(min, max);
            if ((result = Add(pRequestIn)).IsSuccess())
            {
                result = UpdateOperatingRange();
            }
            break;
        }
    }

    return result;
}

nn::Result RequestManager::Signal()
NN_NOEXCEPT
{
    for (Request & request : m_Requests)
    {
        request.Signal();
    }
    return ResultSuccess();
}

void RequestManager::SetRange(Setting min, Setting max) NN_NOEXCEPT
{
    m_Min = min;
    m_Max = max;
    m_CurrentMin = min;
    m_CurrentMax = max;
}

RequestManager::RequestManager()
NN_NOEXCEPT :
    m_Min(Setting_Min),
    m_Max(Setting_Max),
    m_CurrentMin(Setting_Min),
    m_CurrentMax(Setting_Max)
{

}

RequestManager::~RequestManager()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Requests.empty());
}

}}}
