﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/bpc/bpc_PowerButton.h>
#include <nn/fatal/detail/fatal_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Tick.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_SystemProcess.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/util/util_ScopeExit.h>

#include "fatalsrv_Environment.h"
#include "fatalsrv_PowerControl.h"

namespace nn { namespace fatalsrv {
    namespace {

        bool ShutdownBoardPower(IPowerController& powerController) NN_NOEXCEPT
        {
            NN_DETAIL_FATAL_TRACE("Battery voltage is low.\n");
            auto start = nn::os::GetSystemTick();
            bool cancel = false;
            while (NN_STATIC_CONDITION(true))
            {
                auto end = nn::os::GetSystemTick();
                if ((end - start).ToTimeSpan().GetSeconds() > powerController.GetMaxShutdownWaitSec())
                {
                    break;
                }

                auto state = powerController.GetBatteryVoltageState();
                NN_DETAIL_FATAL_TRACE("BatteryVoltageState: %d\n", state);
                if (state == nn::psm::BatteryVoltageState_ShutdownRequired)
                {
                    break;
                }

                if (state == nn::psm::BatteryVoltageState_Good)
                {
                    cancel = true;
                    break;
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            if (!cancel)
            {
                powerController.Shutdown();
                return true;
            }
            else
            {
                return false;
            }
        }

        class DefaultPowerController : public IPowerController
        {
            const int MaxShutdownWaitSec = 30;
            virtual void Shutdown() NN_NOEXCEPT
            {
                nn::bpc::InitializeBoardPowerControl();
                nn::bpc::ShutdownSystem();
            }
            virtual void Reboot() NN_NOEXCEPT
            {
                nn::bpc::InitializeBoardPowerControl();
                nn::bpc::RebootSystem();
            }
            virtual nn::psm::BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT { return nn::psm::GetBatteryVoltageState(); }
            virtual nn::bpc::SleepButtonState GetSleepButtonState() NN_NOEXCEPT
            {
                nn::bpc::SleepButtonState state;
                auto result = nn::bpc::GetSleepButtonState(&state);
                if (result.IsFailure())
                {
                    NN_DETAIL_FATAL_ERROR("bpc::GetSleepButtonState failed: %08x\n", result.GetInnerValueForDebug());
                    return nn::bpc::SleepButtonState::SleepButtonState_Released;
                }
                return state;
            }
            virtual int GetMaxShutdownWaitSec() const NN_NOEXCEPT { return MaxShutdownWaitSec; }
        };


        class StateTransitionStopTask : public ITask
        {
        public:
            virtual Result Run() NN_NOEXCEPT;
            virtual const char* GetTaskName() const NN_NOEXCEPT
            {
                return "StateTransitionStopTask";
            }
        };

        StateTransitionStopTask g_StateTransitionStopTask;
        DefaultPowerController g_DefaultPowerController;

        PowerButtonObserveTask g_PowerButtonTask;
        PowerControlTask g_PowerControlTask;

        class RebootTimingObserver
        {
        public:
            RebootTimingObserver(bool questFlag, int32_t interval) NN_NOEXCEPT : m_BeginTick(nn::os::GetSystemTick()), m_QuestFlag(questFlag), m_Interval(interval)
            {
            }
            bool IsRebootTiming() NN_NOEXCEPT
            {
                auto currentTick = nn::os::GetSystemTick();
                return m_QuestFlag
                    && (currentTick - m_BeginTick).ToTimeSpan().GetSeconds() >= m_Interval;
            }
        private:
            nn::os::Tick m_BeginTick;
            bool m_QuestFlag;
            int32_t m_Interval;
        };

    } // namespace

    void PowerControlTask::Initialize(nn::os::Event* errorReportWrittenEvent, nn::os::Event* batteryCheckedEvent) NN_NOEXCEPT
    {
        Initialize(errorReportWrittenEvent, batteryCheckedEvent, &g_DefaultPowerController);
    }

    void PowerControlTask::Initialize(nn::os::Event* errorReportWrittenEvent, nn::os::Event* batteryCheckedEvent, IPowerController* powerController) NN_NOEXCEPT
    {
        m_ErrorReportWrittenEvent = errorReportWrittenEvent;
        m_BatteryCheckedEvent = batteryCheckedEvent;
        m_pPowerController = powerController;
    }

    Result PowerControlTask::Run() NN_NOEXCEPT
    {
        ObserveBatteryPower();
        NN_RESULT_SUCCESS;
    }

    const char* PowerControlTask::GetTaskName() const NN_NOEXCEPT
    {
        return "PowerControlTask";
    }

    void PowerControlTask::ObserveBatteryPower() NN_NOEXCEPT
    {
        nn::psm::Initialize();
        NN_UTIL_SCOPE_EXIT{ nn::psm::Finalize(); };

        // 要シャットダウンの場合はエラーレポートの記録を1秒だけ待ってすぐにシャットダウンする
        auto state = m_pPowerController->GetBatteryVoltageState();
        if (state == nn::psm::BatteryVoltageState_ShutdownRequired)
        {
            m_ErrorReportWrittenEvent->TimedWait(nn::TimeSpan::FromSeconds(1));
            ShutdownBoardPower(*m_pPowerController);
            return;
        }

        // 要シャットダウンでなかったことを通知
        m_BatteryCheckedEvent->Signal();

        while (NN_STATIC_CONDITION(true))
        {
            auto state = m_pPowerController->GetBatteryVoltageState();
            switch (state)
            {
                case nn::psm::BatteryVoltageState_ShutdownRequired:
                case nn::psm::BatteryVoltageState_SleepRequired:
                {
                    auto succeeded = ShutdownBoardPower(*m_pPowerController);
                    if (succeeded)
                    {
                        return;
                    }
                }
                break;
            default:
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
        }
    }

    void PowerButtonObserveTask::Initialize(nn::os::Event* errorReportWrittenEvent) NN_NOEXCEPT
    {
        Initialize(errorReportWrittenEvent, &g_DefaultPowerController);
    }

    void PowerButtonObserveTask::Initialize(nn::os::Event* errorReportWrittenEvent, IPowerController* powerController) NN_NOEXCEPT
    {
        m_ErrorReportWrittenEvent = errorReportWrittenEvent;
        m_pPowerController = powerController;
    }

    Result PowerButtonObserveTask::Run() NN_NOEXCEPT
    {
        ObservePowerButton();
        NN_RESULT_SUCCESS;
    }

    const char* PowerButtonObserveTask::GetTaskName() const NN_NOEXCEPT
    {
        return "PowerButtonObserveTask";
    }

    void PowerButtonObserveTask::ObservePowerButton() NN_NOEXCEPT
    {
        nn::bpc::InitializePowerButton();
        NN_UTIL_SCOPE_EXIT{ nn::bpc::FinalizePowerButton(); };

        m_ErrorReportWrittenEvent->TimedWait(nn::TimeSpan::FromSeconds(1));

        auto& env = GetEnvironmentInfo();

        RebootTimingObserver rebootTimingObserver(
            env.questFlag,
            env.questRebootIntervalSecond);

        while (NN_STATIC_CONDITION(true))
        {
            auto state = m_pPowerController->GetSleepButtonState();
            if (state == bpc::SleepButtonState::SleepButtonState_Pushed
                || rebootTimingObserver.IsRebootTiming())
            {
                m_pPowerController->Reboot();
                return;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }

    Result StateTransitionStopTask::Run() NN_NOEXCEPT
    {
        spsm::Initialize();
        spsm::PutErrorState();
        spsm::Finalize();
        NN_RESULT_SUCCESS;
    }

    ITask* InitializeAndGetStateTransitionStopTask() NN_NOEXCEPT
    {
        return &g_StateTransitionStopTask;
    }

    ITask* InitializeAndGetPowerButtonTask(nn::os::Event* errorReportWrittenEvent) NN_NOEXCEPT
    {
        g_PowerButtonTask.Initialize(errorReportWrittenEvent);
        return &g_PowerButtonTask;
    }

    ITask* InitializeAndGetPowerControlTask(nn::os::Event* errorReportWrittenEvent, nn::os::Event* batteryCheckedEvent) NN_NOEXCEPT
    {
        g_PowerControlTask.Initialize(errorReportWrittenEvent, batteryCheckedEvent);
        return &g_PowerControlTask;
    }
}} // namespace nn::fatalsrv
