﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fatal/fatal_Api.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/fatalsrv/fatalsrv_Service.h>
#include <nn/fatalsrv/fatalsrv_ServiceName.h>
#include <nn/fatalsrv/fatalsrv_FatalContext.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_UserExceptionHandlerTypes.h>

#include "detail/fatal_CpuContext.h"

#define NN_FATAL_RETURN_IF_FAILED(result) \
    do \
    { \
        auto __result = (result); \
        if(__result.IsFailure()) \
        {\
            NN_SDK_LOG("[fatal] Warning: '%s' failed. Result = 0x%08x\n", #result, __result.GetInnerValueForDebug()); \
            return; \
        }\
    }\
    while(NN_STATIC_CONDITION(false))

namespace nn { namespace fatal {
    namespace {
        typedef sf::ShimLibraryObjectHolder<nn::fatalsrv::IService> ServiceObjectHolder;
        typedef sf::ShimLibraryObjectHolder<nn::fatalsrv::IPrivateService> PrivateServiceObjectHolder;

        sf::SimpleAllInOneHipcClientManager<2> g_ServiceManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
        ServiceObjectHolder g_ServiceObjectHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

        sf::SimpleAllInOneHipcClientManager<2> g_PrivateServiceManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
        PrivateServiceObjectHolder g_PrivateServiceObjectHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN)
        nn::sf::UnmanagedServiceObject<nn::fatalsrv::IService, nn::fatalsrv::Service> g_ServiceObject;
        nn::sf::UnmanagedServiceObject<nn::fatalsrv::IPrivateService, nn::fatalsrv::Service> g_PrivateServiceObject;

        Result InitializeServiceHolder()
        {
            g_ServiceObjectHolder.InitializeHolderDirectly(g_ServiceObject.GetShared());
            NN_RESULT_SUCCESS;
        }

        Result InitializePrivateServiceHolder()
        {
            g_PrivateServiceObjectHolder.InitializeHolderDirectly(g_PrivateServiceObject.GetShared());
            NN_RESULT_SUCCESS;
        }
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
        Result InitializeServiceHolder()
        {
            return g_ServiceManager.InitializeShimLibraryHolder(&g_ServiceObjectHolder, nn::fatalsrv::ServicePortName);
        }

        Result InitializePrivateServiceHolder()
        {
            return g_PrivateServiceManager.InitializeShimLibraryHolder(&g_PrivateServiceObjectHolder, nn::fatalsrv::PrivateServicePortName);
        }

#endif

        Result Initialize() NN_NOEXCEPT
        {
            NN_RESULT_DO(InitializeServiceHolder());
            NN_RESULT_SUCCESS;
        }

        Result InitializeForPrivate() NN_NOEXCEPT
        {
            NN_RESULT_DO(InitializePrivateServiceHolder());
            NN_RESULT_SUCCESS;
        }

        void Finalize() NN_NOEXCEPT
        {
            g_ServiceObjectHolder.FinalizeHolder();
        }

        void FinalizeForPrivate() NN_NOEXCEPT
        {
            g_PrivateServiceObjectHolder.FinalizeHolder();
        }

        const nn::Bit64 PseudoProcessId = 0;

        nn::fatal::FatalPolicy g_FatalPolicy = nn::fatal::FatalPolicy_Default;
        nn::fatalsrv::CpuContext g_CpuContext = {};

        void ThrowFatalImpl(const fatalsrv::FatalContext& context, fatal::FatalPolicy policy, const nn::os::UserExceptionInfo* exceptionInfo) NN_NOEXCEPT
        {
            NN_FATAL_RETURN_IF_FAILED(Initialize());
            NN_UTIL_SCOPE_EXIT{ Finalize(); };

            if (exceptionInfo == nullptr)
            {
                detail::FillCpuContext(&g_CpuContext);
            }
            else
            {
                detail::FillCpuContextFromUserExceptionInfo(&g_CpuContext, exceptionInfo);
            }

            NN_FATAL_RETURN_IF_FAILED(
                g_ServiceObjectHolder.GetObject()->ThrowFatalWithCpuContext(context, PseudoProcessId, policy, g_CpuContext));
        }
    }

    void ThrowFatal(const Result& lastResult) NN_NOEXCEPT
    {
        nn::fatalsrv::FatalContext context = {};
        context.lastResult = lastResult;
        ThrowFatalImpl(context, g_FatalPolicy, nullptr);
    }

    void ThrowFatalWithoutErrorReport(const Result& lastResult) NN_NOEXCEPT
    {
        nn::fatalsrv::FatalContext context = {};
        context.lastResult = lastResult;
        ThrowFatalImpl(context, fatal::FatalPolicy_WithoutErrorReport, nullptr);
    }

    void ThrowFatal(const Result& lastResult, const nn::os::UserExceptionInfo* exceptionInfo) NN_NOEXCEPT
    {
        nn::fatalsrv::FatalContext context = {};
        context.lastResult = lastResult;
        ThrowFatalImpl(context, g_FatalPolicy, exceptionInfo);
    }

    void SetFatalPolicy(FatalPolicy fatalPolicy) NN_NOEXCEPT
    {
        g_FatalPolicy = fatalPolicy;
    }

    void GetFatalEvent(nn::os::SystemEvent* out) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(out);

        NN_ABORT_UNLESS_RESULT_SUCCESS(InitializeForPrivate());
        NN_UTIL_SCOPE_EXIT{ FinalizeForPrivate(); };

        nn::sf::NativeHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_PrivateServiceObjectHolder.GetObject()->GetFatalEvent(&handle));

        out->AttachReadableHandle(handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode::EventClearMode_AutoClear);
        handle.Detach();
    }
}} // namespace nn::fatal
