﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/fan/impl/fan.h>

#include <nn/fan/detail/fan_IController.h>
#include <nn/fan/detail/fan_IManager.h>
#include <nn/fan/fan_ManagerImpl.h>

#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace fan {

// IController 実装クラス
class ManagerImpl::ControllerImpl
{
public:
    // コンストラクタで親の SharedPointer を設定
    ControllerImpl(ManagerImpl* pParent, int name) NN_NOEXCEPT
        : m_Parent(pParent, true)
    {
        nn::fan::impl::OpenController(&m_Controller, static_cast<nn::fan::FanName>(name));
    }

    // Session のデストラクタで Session のクローズを行う
    ~ControllerImpl()
    {
        nn::fan::impl::CloseController(&m_Controller);
    }

    // FAN ドライバライブラリの API
    nn::Result SetRotationSpeedLevel(nn::fan::RotationSpeedLevel level) NN_NOEXCEPT;
    nn::Result SetRotationSpeedLevelByDiff(nn::sf::Out<nn::fan::RotationSpeedLevel> outNewLevel, nn::fan::RotationSpeedLevel levelDiff) NN_NOEXCEPT;
    nn::Result GetRotationSpeedLevel(nn::sf::Out<nn::fan::RotationSpeedLevel> outLevel) NN_NOEXCEPT;
    nn::Result GetActualRotationSpeedLevel(nn::sf::Out<nn::fan::RotationSpeedLevel> outLevel) NN_NOEXCEPT;
    nn::Result SetRotationRate(int rotationRate) NN_NOEXCEPT;
    nn::Result GetRotationRate(nn::sf::Out<int> outRotationRate) NN_NOEXCEPT;
    nn::Result EnableFan() NN_NOEXCEPT;
    nn::Result DisableFan() NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl>  m_Parent;

    // FAN ドライバライブラリ向けのセッション構造体
    nn::fan::impl::Controller           m_Controller;
};

// ここから各 FAN API の内部実装
nn::Result ManagerImpl::ControllerImpl::SetRotationSpeedLevel(nn::fan::RotationSpeedLevel level) NN_NOEXCEPT
{
    nn::fan::impl::SetRotationSpeedLevel(&m_Controller, level);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::ControllerImpl::SetRotationSpeedLevelByDiff(nn::sf::Out<nn::fan::RotationSpeedLevel> outNewLevel, nn::fan::RotationSpeedLevel levelDiff) NN_NOEXCEPT
{
    *outNewLevel = nn::fan::impl::SetRotationSpeedLevelByDiff(&m_Controller, levelDiff);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::ControllerImpl::GetRotationSpeedLevel(nn::sf::Out<nn::fan::RotationSpeedLevel> outLevel) NN_NOEXCEPT
{
    *outLevel = nn::fan::impl::GetRotationSpeedLevel(&m_Controller);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::ControllerImpl::GetActualRotationSpeedLevel(nn::sf::Out<nn::fan::RotationSpeedLevel> outLevel) NN_NOEXCEPT
{
    *outLevel = nn::fan::impl::GetActualRotationSpeedLevel(&m_Controller);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::ControllerImpl::SetRotationRate(int rotationRate) NN_NOEXCEPT
{
    nn::fan::impl::SetRotationRate(&m_Controller, rotationRate);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::ControllerImpl::GetRotationRate(nn::sf::Out<int> outRotationRate) NN_NOEXCEPT
{
    *outRotationRate = nn::fan::impl::GetRotationRate(&m_Controller);
    return nn::ResultSuccess();
}

nn::Result ManagerImpl::ControllerImpl::EnableFan() NN_NOEXCEPT
{
    nn::fan::impl::EnableFan(&m_Controller);
    NN_RESULT_SUCCESS;
}

nn::Result ManagerImpl::ControllerImpl::DisableFan() NN_NOEXCEPT
{
    nn::fan::impl::DisableFan(&m_Controller);
    NN_RESULT_SUCCESS;
}

// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);

    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

nn::Result ManagerImpl::OpenController(nn::sf::Out<nn::sf::SharedPointer<nn::fan::detail::IController>> outSession, std::int32_t name) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<nn::fan::detail::IController, ManagerImpl::ControllerImpl>(&m_Allocator, this, static_cast<int>(name));

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}
