﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>

#include <nn/sf/sf_Types.h>                  // for nn::sf::SharedPointer
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ShimLibraryUtility.h>

#include <nn/fan/fan_Lib.h>
#include <nn/fan/fan_Control.h>
#include <nn/fan/fan_ControlDev.h>
#include <nn/fan/fan_ServiceName.h>

#include <nn/fan/detail/fan_IManager.h> // for IManager

namespace nn { namespace fan {

namespace {

const int NumberOfManagers = 1;
const int NumberOfSessions = 1;
const int NumberOfClients = NumberOfManagers + NumberOfSessions;

sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
sf::ShimLibraryObjectHolder<detail::IManager> g_Holder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

} // namespace

void Initialize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::fan::ServiceName));
}

void InitializeWith(sf::SharedPointer<detail::IManager>&& manager) NN_NOEXCEPT
{
    g_Holder.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<detail::IManager> GetInternalManager() NN_NOEXCEPT
{
    return g_Holder.GetObject();
}

void Finalize() NN_NOEXCEPT
{
    g_Holder.FinalizeHolder();
}

void OpenController(Controller* pOutController, FanName name) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fan::detail::IController> controller;
    auto result = g_Holder.GetObject()->OpenController(&controller, static_cast<std::int32_t>(name));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SharedPointer から所有権を切り離して、pOutHandle として返す
    pOutController->_controllerHandle = controller.Detach();
}

void CloseController(Controller* pController) NN_NOEXCEPT
{
    nn::sf::ReleaseSharedObject(static_cast<nn::fan::detail::IController*>(pController->_controllerHandle));
}

void SetRotationSpeedLevel(Controller* pController, RotationSpeedLevel level) NN_NOEXCEPT
{
    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->SetRotationSpeedLevel(level);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

RotationSpeedLevel SetRotationSpeedLevelByDiff(Controller* pController, int levelDiff) NN_NOEXCEPT
{
    // 出力用の変数
    RotationSpeedLevel outValue;

    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->SetRotationSpeedLevelByDiff(&outValue, levelDiff);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}

RotationSpeedLevel GetRotationSpeedLevel(Controller* pController) NN_NOEXCEPT
{
    // 出力用の変数
    RotationSpeedLevel outValue;

    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->GetRotationSpeedLevel(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}

RotationSpeedLevel GetActualRotationSpeedLevel(Controller* pController) NN_NOEXCEPT
{
    // 出力用の変数
    RotationSpeedLevel outValue;

    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->GetRotationSpeedLevel(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}

void SetRotationRate(Controller* pController, int rotationRate) NN_NOEXCEPT
{
    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->SetRotationRate(rotationRate);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

int GetRotationRate(Controller* pController) NN_NOEXCEPT
{
    // 出力用の変数
    int outValue;

    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->GetRotationRate(&outValue);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return outValue;
}

void EnableFan(Controller* pController) NN_NOEXCEPT
{
    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->EnableFan();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void DisableFan(Controller* pController) NN_NOEXCEPT
{
    auto result = static_cast<nn::fan::detail::IController*>(pController->_controllerHandle)->DisableFan();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

}} // namespace nn::fan
