﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nn;
using nn.sf;
using std;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;

namespace nn.es
{
    public interface IActiveRightsContext : IServiceObject
    {
        /// <summary>
        /// 権利コンテキストにアカウント限定権利の利用者リストを登録する
        /// </summary>
        /// <param name="userIdList">ElicenseUserId のリスト</param>
        /// <param name="anyAccount">デバイス共有アカウントか否か</param>
        /// <returns></returns>
        [MethodId(1)]
        Result SetUsersOfAccountRestrictedRights(
                [BufferTransferMode(BufferTransferMode.MapAlias)]
                InArray<ELicenseUserId> userIdList,
                bool anyAccount
        );

        /// <summary>
        /// 権利コンテキストに使用する RightsId(64bit) を登録する
        /// </summary>
        /// <param name="rightsIdList">RightsId のリスト</param>
        /// <returns></returns>
        [MethodId(2)]
        Result RegisterRightsIdList(
                [BufferTransferMode(BufferTransferMode.MapAlias)]
                InArray<RightsId> rightsIdList
        );


        /// <summary>
        /// 権利リスト上の RightsId(64bit) に対応する eLicense を検索する
        /// </summary>
        /// <returns></returns>
        [MethodId(3)]
        Result CheckRightsIdListValidity(
        );

        /// <summary>
        /// 対応する eLicense がない rightsId を権利リスト上から削除する
        /// </summary>
        /// <returns></returns>
        [MethodId(4)]
        Result RemoveUnavailableRightsIds(
        );


        /// <summary>
        /// 権利コンテキストから、RightsId 権利情報をリストで取得
        /// </summary>
        /// <param name="outCount">実際に返す要素数</param>
        /// <param name="outList">権利情報のリスト</param>
        /// <param name="isTemporaryOnly">有効期限ありのものだけを対象にするか否か</param>
        /// <returns></returns>
        [MethodId(6)]
        Result ListUsingRightsIds(
                Out<int> outCount,
                [BufferTransferMode(BufferTransferMode.MapAlias)]
                OutArray<RightsId> outList,
                bool isTemporaryOnly
        );

        /// <summary>
        /// 権利コンテキストから、使用中の eLicenseId のリストを取得
        /// </summary>
        /// <param name="outCount">実際に返す要素数</param>
        /// <param name="outList">格納先配列</param>
        /// <param name="isTemporaryOnly">有効期限ありのものだけを対象にするか否か</param>
        /// <param name="ownerId">権利オーナーの NAID</param>
        /// <returns></returns>
        /// <remarks>
        /// NAID を指定しないときは ownerId = InvalidELicenseOwnerId を指定する
        /// </remarks>
        [MethodId(7)]
        Result ListUsingELicenseIds(
                Out<int> outCount,
                [BufferTransferMode(BufferTransferMode.MapAlias)]
                OutArray<ELicenseId> outList,
                bool isTemporaryOnly,
                ELicenseOwnerId ownerId
        );

        /// <summary>
        /// 権利コンテキストから、アカウント限定権利の利用者リストを取得
        /// </summary>
        /// <param name="outCount">実際に返す要素数</param>
        /// <param name="outList">権利情報のリスト</param>
        /// <returns></returns>
        [MethodId(8)]
        Result GetUsersOfAccountRestrictedRights(
                Out<int> outCount,
                [BufferTransferMode(BufferTransferMode.MapAlias)]
                OutArray<ELicenseUserId> outList
        );

        /// <summary>
        /// 権利コンテキストから、使用中の eLicenseId の所有者リストを取得
        /// </summary>
        /// <param name="outCount">実際に返す要素数</param>
        /// <param name="outList">格納先配列</param>
        /// <returns></returns>
        [MethodId(9)]
        Result ListUsingELicenseOwnerIds(
                Out<int> outCount,
                [BufferTransferMode(BufferTransferMode.MapAlias)]
                OutArray<ELicenseOwnerId> outList
        );

        /// <summary>
        /// 権利コンテキストの使用開始
        /// </summary>
        /// <returns></returns>
        [MethodId(100)]
        Result BeginUsingActiveRightsContext(
        );

        /// <summary>
        /// 権利コンテキストの使用中止
        /// </summary>
        /// <returns></returns>
        [MethodId(101)]
        Result EndUsingActiveRightsContext(
        );

        /// <summary>
        /// 権利コンテキストの強制使用開始
        /// </summary>
        /// <returns></returns>
        [MethodId(102)]
        Result ForceActivateRightsContextForExit(
        );



        /// <summary>
        /// 権利コンテキストの権利リスト全体の総合的な状態を返す
        /// </summary>
        /// <param name="outCount">総合的な状態</param>
        /// <returns></returns>
        [MethodId(200)]
        Result GetActiveRightsContextStatus(
            Out<int> outStatus
        );

        /// <summary>
        /// 権利コンテキストの権利リスト全体で最も近い未来の有効期限を返す
        /// </summary>
        /// <param name="outExpiredTime">有効期限を示す os::Tick の即値</param>
        /// <returns></returns>
        [MethodId(201)]
        Result GetActiveRightsContextExpiredTime(
            Out<int64_t> outExpiredTime
        );

        /// <summary>
        /// 権利コンテキストの権利リスト全体での権利有無や有効期限変更を通知するシステムイベントの取得
        /// </summary>
        /// <param name="pOut">システムイベントの ReadableHandle</param>
        /// <returns></returns>
        [MethodId(202)]
        Result GetActiveRightsContextExpiredTimeChangedEvent(
                Out<NativeHandle> pOut
        );
    }
}
