﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <cmath>
#include <cstring>
#include <nn/es/es_Configuration.h>
#include <nn/es/es_MaxTicketSize.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/spl/spl_Api.h>
#include "es_Certificate.h"
#include "es_DevicePublicKey.h"
#include "es_EncryptionKey.h"
#include "es_EncryptionUtil.h"
#include "es_TicketDatabase.h"

namespace nn { namespace es {

const TicketRecord TicketRecord::NullRecord = {};

const TicketRecord TicketRecord::TerminatorRecord =
{
    {{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }},
    0xffffffffffffffff,
    0xffffffff,
};

const TicketMetaRecord TicketMetaRecord::NullRecord = {};

const TicketMetaRecord TicketMetaRecord::TerminatorRecord =
{
    {{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff }},
    0xffffffffffffffff,
    0xffffffff,
};

TicketMetaRecord GetTicketMetaRecordFromPrepurchaseRecord(PrepurchaseRecord record) NN_NOEXCEPT
{
    TicketMetaRecord metaRecord = {};

    metaRecord.rightsId = record.rightsId;
    metaRecord.ticketId = record.ticketId;
    metaRecord.accountId = record.accountId;
    metaRecord.type = TicketMetaRecordType_Prepurchase;
    metaRecord.prepurchase.deliveryScheduledTime = record.deliveryScheduledTime;

    return metaRecord;
}

PrepurchaseRecord GetPrepurchaseRecordFromTicketMetaRecord(TicketMetaRecord metaRecord) NN_NOEXCEPT
{
    NN_SDK_ASSERT(metaRecord.type == TicketMetaRecordType_Prepurchase);

    PrepurchaseRecord record = {};

    record.rightsId = metaRecord.rightsId;
    record.ticketId = metaRecord.ticketId;
    record.accountId = metaRecord.accountId;
    record.deliveryScheduledTime = metaRecord.prepurchase.deliveryScheduledTime;

    return record;
}

void ETicketServiceDatabase::SetVersion(int version) NN_NOEXCEPT
{
    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, m_VersionFilePath.Get(), nn::fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, 0, &version, sizeof(int), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
}

int ETicketServiceDatabase::GetVersion() const NN_NOEXCEPT
{
    int version;

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, m_VersionFilePath.Get(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, &version, sizeof(int)));

    return version;
}

void ETicketServiceDatabase::Initialize() NN_NOEXCEPT
{
    // バージョン管理用のファイルを作成する
    Result result = nn::fs::CreateFile(m_VersionFilePath.Get(), sizeof(int));
    NN_ABORT_UNLESS(result.IsSuccess() || fs::ResultPathAlreadyExists::Includes(result));
}

void CertificateDatabase::Initialize() NN_NOEXCEPT
{
    ETicketServiceDatabase::Initialize();

    // 証明書のディレクトリがなければ生成する
    Result result = fs::CreateDirectory(m_CertificatePath.Get());
    NN_ABORT_UNLESS(result.IsSuccess() || fs::ResultPathAlreadyExists::Includes(result));
}

void CertificateDatabase::ImportCertificate(const void* certificate, size_t certificateSize, const char* subjectName) NN_NOEXCEPT
{
    Path certificateFilePath;
    certificateFilePath.AssignFormat("%s/%s", m_CertificatePath.Get(), subjectName);

    Result result = fs::CreateFile(certificateFilePath.Get(), certificateSize);
    NN_ABORT_UNLESS(result.IsSuccess() || fs::ResultPathAlreadyExists::Includes(result));

    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, certificateFilePath.Get(), fs::OpenMode_Read | fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    // 証明書がまだ保存されていなかった場合は、インポートされた証明書を保存する
    if (result.IsSuccess())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, 0, certificate, certificateSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    // 証明書がすでに保存されている場合は、すでに保存されている証明書と比較し、等しい場合は何もしない
    // 等しくない場合、保存されている証明書を削除して、インポートされた証明書を保存する
    else if (fs::ResultPathAlreadyExists::Includes(result))
    {
        int64_t fileSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, fileHandle));

        std::unique_ptr<char[]> savedCertificate(new char[static_cast<size_t>(fileSize)]);
        NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(fileHandle, 0, savedCertificate.get(), static_cast<size_t>(fileSize)));

        // インポートされた証明書と保存されている証明書を比較する
        if (certificateSize == static_cast<size_t>(fileSize)  &&  memcmp(certificate, savedCertificate.get(), certificateSize) != 0)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::SetFileSize(fileHandle, certificateSize));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, 0, certificate, certificateSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }
    }
}

int CertificateDatabase::GetCertificateSize(const char* subjectName) NN_NOEXCEPT
{
    Path certificateFilePath;
    certificateFilePath.AssignFormat("%s/%s", m_CertificatePath.Get(), subjectName);

    fs::FileHandle fileHandle;
    NN_RESULT_TRY(fs::OpenFile(&fileHandle, certificateFilePath.Get(), fs::OpenMode_Read))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultCertificateNotFound());
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, fileHandle));

    return static_cast<int>(fileSize);
}

Result CertificateDatabase::GetCertificate(int* outSize, void* outBuffer, size_t outBufferSize, const char* subjectName) NN_NOEXCEPT
{
    Path certificateFilePath;
    certificateFilePath.AssignFormat("%s/%s", m_CertificatePath.Get(), subjectName);

    fs::FileHandle fileHandle;
    NN_RESULT_TRY(fs::OpenFile(&fileHandle, certificateFilePath.Get(), fs::OpenMode_Read))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultCertificateNotFound());
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::GetFileSize(&fileSize, fileHandle));
    NN_SDK_ASSERT(outBufferSize >= static_cast<size_t>(fileSize));

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(fileHandle, 0, outBuffer, static_cast<size_t>(fileSize)));
    *outSize = static_cast<int>(fileSize);

    NN_RESULT_SUCCESS;
}

void TicketDatabase::CreateTicketInfo(TicketInfo* outTicketInfo, const void* ticket, size_t ticketSize) NN_NOEXCEPT
{
    MemoryInputStream ticketReader(ticket, static_cast<unsigned int>(ticketSize));

    nn::escore::ETicket eTicket;
    eTicket.Set(ticketReader, nullptr, 0, false);

    *outTicketInfo = {};
    eTicket.GetAccountId(&outTicketInfo->accountId);
    eTicket.GetDeviceId(reinterpret_cast<nn::escore::ESDeviceId*>(&outTicketInfo->deviceId));
    eTicket.GetRightsId(reinterpret_cast<nn::escore::ESRightsId*>(&outTicketInfo->rightsId));
    eTicket.GetTicketId(reinterpret_cast<nn::escore::ESTicketId*>(&outTicketInfo->ticketId));
    uint32_t tmpTicketSize;
    eTicket.GetSize(&tmpTicketSize);
    outTicketInfo->ticketSize = static_cast<int64_t>(tmpTicketSize);
    eTicket.GetTicketVersion(&outTicketInfo->ticketVersion);
    eTicket.GetPropertyMask(&outTicketInfo->propertyMask);
}

void TicketDatabase::Write(int index, const void* ticket, size_t ticketSize)  NN_NOEXCEPT
{
    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, m_TicketFilePath.Get(), fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };

    uint8_t zeroBuffer[MaxTicketSize] = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, index * MaxTicketSize, ticket, ticketSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::WriteFile(fileHandle, index * MaxTicketSize + ticketSize, zeroBuffer, MaxTicketSize - ticketSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
}

void TicketDatabase::Read(void* outTicket, size_t ticketSize, int index) const NN_NOEXCEPT
{
    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, m_TicketFilePath.Get(), fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ fs::CloseFile(fileHandle); };

    // チケットデータ取得
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::ReadFile(fileHandle, index * MaxTicketSize, outTicket, ticketSize));
}

void TicketDatabase::Initialize() NN_NOEXCEPT
{
    TicketDatabaseBase::Initialize();
    ETicketServiceDatabase::Initialize();

    Result result = fs::CreateFile(m_TicketFilePath.Get(), m_TicketList.MaxRecordCount() * MaxTicketSize);
    NN_ABORT_UNLESS(result.IsSuccess() || fs::ResultPathAlreadyExists::Includes(result));
}

Result TicketDatabase::ImportTicket(const void* ticket, size_t ticketSize) NN_NOEXCEPT
{
    // MaxTicketSize よりもサイズが大きいチケットはインポートしない
    NN_RESULT_THROW_UNLESS(ticketSize <= MaxTicketSize, ResultTicketInvalid());

    TicketInfo ticketInfo;
    CreateTicketInfo(&ticketInfo, ticket, ticketSize);

    // RightsId と TicketId が同一のチケットが既に存在していた場合
    if (m_TicketList.Find(ticketInfo.rightsId, ticketInfo.ticketId) != -1)
    {
        TicketInfo savedTicketInfo;
        GetTicketInfo(&savedTicketInfo, ticketInfo.rightsId, ticketInfo.ticketId);

        // インポート対象のチケットが保存されているものよりも古い場合
        NN_RESULT_THROW_UNLESS(savedTicketInfo.ticketVersion <= ticketInfo.ticketVersion, ResultTicketVersionInvalid());

        DeleteTicket(ticketInfo.rightsId, ticketInfo.ticketId);
    }
    else
    {
        // チケットの保存数が上限に達している場合
        NN_RESULT_THROW_UNLESS(!TicketDatabaseBase::IsFull(), ResultTicketFull());
    }

    TicketRecord record = { ticketInfo.rightsId, ticketInfo.ticketId, ticketInfo.accountId, ticketInfo.propertyMask, 0 };
    int index =  m_TicketList.InsertWithoutCheckingDuplication(record);
    Write(index, ticket, ticketSize);

    NN_RESULT_SUCCESS;
}

Result TicketDatabase::GetTitleKey(AesKey* outTitleKey, const RightsIdIncludingKeyId& rightsId, TicketId ticketId, int keyGeneration) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);
    NN_RESULT_THROW_UNLESS(index != -1, ResultRightsIdNotFound());

    std::unique_ptr<char[]> ticket(new char[MaxTicketSize]);
    size_t ticketSize;
    NN_RESULT_DO(GetTicketData(&ticketSize, ticket.get(), MaxTicketSize, rightsId, ticketId));
    NN_RESULT_DO(VerifyTicket(ticket.get(), MaxTicketSize));

#if defined(NN_BUILD_CONFIG_OS_WIN)
    memset(outTitleKey->_data, 0, sizeof(AesKey));

#else
    MemoryInputStream ticketReader(ticket.get(), static_cast<unsigned int>(MaxTicketSize));

    nn::escore::ETicket eTicket;
    nn::escore::ESError esError = eTicket.Set(ticketReader, nullptr, 0, false);
    NN_RESULT_THROW_UNLESS(esError == ES_ERR_OK, ResultTitleKeyCouldntDecrypt());

    nn::escore::ESTitleKeyType esTitleKeyType;
    esError = eTicket.GetTitleKeyType(&esTitleKeyType);
    NN_RESULT_THROW_UNLESS(esError == ES_ERR_OK, ResultTitleKeyCouldntDecrypt());
    NN_RESULT_THROW_UNLESS(static_cast<nn::es::KeyType>(esTitleKeyType) == nn::es::KeyType_AesKey || static_cast<nn::es::KeyType>(esTitleKeyType) == nn::es::KeyType_RsaKey, ResultTitleKeyCouldntDecrypt());

    nn::escore::ESTitleKey esTitleKey;
    esError = eTicket.GetTitleKey(&esTitleKey);
    NN_RESULT_THROW_UNLESS(esError == ES_ERR_OK, ResultTitleKeyCouldntDecrypt());

    nn::spl::AccessKey accessKey;

    switch (static_cast<nn::es::KeyType>(esTitleKeyType))
    {
    case nn::es::KeyType_AesKey:
    {
        NN_RESULT_TRY(nn::spl::PrepareCommonEsTitleKey(&accessKey, esTitleKey.aesKey, sizeof(AesKey), keyGeneration))
            NN_RESULT_CATCH_ALL
        {
            return ResultTitleKeyCouldntDecrypt();
        }
        NN_RESULT_END_TRY
            break;
    }
    case nn::es::KeyType_RsaKey:
    {
        // 0バイトに SHA256 をかけたバイト列を使う
        uint8_t LabelDigest[32] =
        {
            0xE3, 0xB0, 0xC4, 0x42, 0x98, 0xFC, 0x1C, 0x14, 0x9A, 0xFB, 0xF4, 0xC8, 0x99, 0x6F, 0xB9, 0x24,
            0x27, 0xAE, 0x41, 0xE4, 0x64, 0x9B, 0x93, 0x4C, 0xA4, 0x95, 0x99, 0x1B, 0x78, 0x52, 0xB8, 0x55,
        };

        Rsa2048DevicePublicKey publicKey;

        NN_RESULT_TRY(nn::spl::PrepareEsTitleKey(
            &accessKey,
            esTitleKey.rsaKey, sizeof(RsaKey),
            publicKey.GetModulus(), publicKey.GetModulusSize(),
            LabelDigest, sizeof(LabelDigest),
            keyGeneration
        ));
            NN_RESULT_CATCH_ALL
            {
                return ResultTitleKeyCouldntDecrypt();
            }
        NN_RESULT_END_TRY
        break;
    }
    default:
        NN_RESULT_THROW(ResultTitleKeyCouldntDecrypt());
    }

    memcpy(outTitleKey->_data, &accessKey, sizeof(AesKey));

#endif

    NN_RESULT_SUCCESS;
}

size_t TicketDatabase::GetTicketSize(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);

    // RightsId と TicketId が一致するチケットが見つからない場合
    if (index == -1)
    {
        return 0;
    }

    std::unique_ptr<char[]> ticket(new char[MaxTicketSize]);
    Read(ticket.get(), MaxTicketSize, index);

    TicketInfo ticketInfo;
    CreateTicketInfo(&ticketInfo, ticket.get(), MaxTicketSize);

    return static_cast<int>(ticketInfo.ticketSize);
}

Result TicketDatabase::GetTicketData(size_t* outSize, void* outTicketBuffer, size_t ticketBufferSize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);

    // RightsId と TicketId が一致するチケットが見つからない場合
    NN_RESULT_THROW_UNLESS(index != -1, ResultRightsIdNotFound());

    size_t ticketSize = GetTicketSize(rightsId, ticketId);
    NN_RESULT_THROW_UNLESS(ticketSize <= ticketBufferSize, ResultBufferNotEnough());

    Read(outTicketBuffer, ticketSize, index);

    *outSize = ticketSize;

    NN_RESULT_SUCCESS;
}

size_t TicketDatabase::GetEncryptedTicketSize(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    return static_cast<int>(CalculateEncryptionTextSizeAes128CbcWithPkcs7Padding(GetTicketSize(rightsId, ticketId)));
}

Result TicketDatabase::GetEncryptedTicketData(size_t* outSize, void* outTicketBuffer, size_t ticketBufferSize, void* outKey, size_t outKeySize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    size_t rawTicketSize;
    char rawTicketData[MaxTicketSize];

    NN_RESULT_DO(GetTicketData(&rawTicketSize, rawTicketData, sizeof(rawTicketData), rightsId, ticketId));
    NN_RESULT_THROW_UNLESS(rawTicketSize != 0, ResultRightsIdNotFound());

    // 乱数で AES 鍵を生成する
    uint8_t aesKey[16];
    NN_RESULT_DO(GenerateRandomBytes(aesKey, sizeof(aesKey)));

    const uint8_t Iv[16] = {};

    size_t encryptedSize = EncryptAes128CbcWithPkcs7Padding(
        outTicketBuffer, ticketBufferSize,
        aesKey, sizeof(aesKey),
        Iv, sizeof(Iv),
        rawTicketData, rawTicketSize
    );
    NN_RESULT_THROW_UNLESS(encryptedSize == static_cast<size_t>(GetEncryptedTicketSize(rightsId, ticketId)), ResultTicketEncryptionFailed());

    const uint8_t* pModulus;
    if (IsDevelopment())
    {
        pModulus = ModulusForDevelopment;
    }
    else
    {
        pModulus = ModulusForProduction;
    }

    bool isRsaEncryptionSucceeded = EncryptRsa2048OaepSha256(
        outKey, outKeySize,
        pModulus, ModulusSize,
        PublicExponent, sizeof(PublicExponent),
        aesKey, sizeof(aesKey),
        "", 0);
    NN_RESULT_THROW_UNLESS(isRsaEncryptionSucceeded, ResultKeyEncryptionFailed());

    *outSize = encryptedSize;
    NN_RESULT_SUCCESS;
}

bool TicketDatabase::GetCertificateName(char* outCaName, char* outXsName, size_t caNameBufferSize, size_t xsNameBufferSize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);

    // RightsId と TicketId が一致するチケットが見つからない場合
    if (index == -1)
    {
        return false;
    }

    std::unique_ptr<char[]> ticket(new char[MaxTicketSize]);
    Read(ticket.get(), MaxTicketSize, index);

    return GetCertificateNameImpl(outCaName, outXsName, caNameBufferSize, xsNameBufferSize, ticket.get(), MaxTicketSize);
}

int TicketDatabase::GetCertificateSize(const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);

    // RightsId と TicketId が一致するチケットが見つからない場合
    if (index == -1)
    {
        return 0;
    }

    char caName[CertificateNameLength + 1];
    char xsName[CertificateNameLength + 1];
    bool isSuccess = GetCertificateName(caName, xsName, CertificateNameLength + 1, CertificateNameLength + 1, rightsId, ticketId);
    if (!isSuccess)
    {
        return 0;
    }

    int caSize = m_CertificateDatabaseRef->GetCertificateSize(caName);
    int xsSize = m_CertificateDatabaseRef->GetCertificateSize(xsName);

    return caSize + xsSize;
}

int TicketDatabase::GetCertificateData(void* outCertificate, size_t outCertificateSize, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    uint8_t* outCertificatePtr = reinterpret_cast<uint8_t*>(outCertificate);

    char caName[CertificateNameLength + 1];
    char xsName[CertificateNameLength + 1];
    bool isSuccess = GetCertificateName(caName, xsName, CertificateNameLength + 1, CertificateNameLength + 1, rightsId, ticketId);
    if (!isSuccess)
    {
        return 0;
    }

    int caSize = m_CertificateDatabaseRef->GetCertificateSize(caName);
    int xsSize = m_CertificateDatabaseRef->GetCertificateSize(xsName);
    NN_SDK_ASSERT(static_cast<int>(outCertificateSize) >= caSize + xsSize);

    int outCaSize;
    NN_RESULT_DO(m_CertificateDatabaseRef->GetCertificate(&outCaSize, outCertificatePtr, caSize, caName));
    NN_SDK_ASSERT_EQUAL(caSize, outCaSize);

    int outXsSize;
    NN_RESULT_DO(m_CertificateDatabaseRef->GetCertificate(&outXsSize, &outCertificatePtr[caSize], xsSize, xsName));
    NN_SDK_ASSERT_EQUAL(xsSize, outXsSize);

    return caSize + xsSize;
}

bool TicketDatabase::GetTicketInfo(TicketInfo* outTicketInfo, const RightsIdIncludingKeyId& rightsId, TicketId ticketId) const NN_NOEXCEPT
{
    int index = m_TicketList.Find(rightsId, ticketId);

    // RightsId と TicketId が一致するチケットが見つからない場合
    if (index == -1)
    {
        return false;
    }

    fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(fs::OpenFile(&fileHandle, m_TicketFilePath.Get(), fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT { fs::CloseFile(fileHandle); };

    std::unique_ptr<char[]> ticket(new char[MaxTicketSize]);
    Read(ticket.get(), MaxTicketSize, index);

    CreateTicketInfo(outTicketInfo, ticket.get(), MaxTicketSize);

    return true;
}

bool TicketDatabase::HasDeviceLinkedTicket(RightsId rightsId) const NN_NOEXCEPT
{
    int index = 0;
    TicketRecord* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            // rightsId が一致しているかつ IsElicenseRequired フラグが立っていない
            if (list[i].rightsId.GetRightsId() == rightsId && !list[i].IsElicenseRequired())
            {
                return true;
            }
        }
    }

    return false;
}

Result TicketDatabase::VerifyTicket(const void* ticket, size_t ticketSize) const NN_NOEXCEPT
{
    char caName[CertificateNameLength + 1];
    char xsName[CertificateNameLength + 1];
    bool isSuccess = GetCertificateNameImpl(caName, xsName, CertificateNameLength + 1, CertificateNameLength + 1, ticket, ticketSize);
    NN_RESULT_THROW_UNLESS(isSuccess, ResultTicketInvalid());

    int caSize = m_CertificateDatabaseRef->GetCertificateSize(caName);
    std::unique_ptr<char[]> caCertificate(new char[caSize]);
    int outCaSize;
    NN_RESULT_DO(m_CertificateDatabaseRef->GetCertificate(&outCaSize, caCertificate.get(), caSize, caName));
    NN_SDK_ASSERT_EQUAL(caSize, outCaSize);

    int xsSize = m_CertificateDatabaseRef->GetCertificateSize(xsName);
    std::unique_ptr<char[]> xsCertificate(new char[xsSize]);
    int outXsSize;
    NN_RESULT_DO(m_CertificateDatabaseRef->GetCertificate(&outXsSize, xsCertificate.get(), xsSize, xsName));
    NN_SDK_ASSERT_EQUAL(xsSize, outXsSize);

    void* certificateList[2] =
    {
        caCertificate.get(),
        xsCertificate.get()
    };

    // チケットを検証する
    {
        MemoryInputStream ticketReader(ticket, static_cast<u32>(ticketSize));

        nn::escore::ETicket eTicket;
        nn::escore::ESError esError = eTicket.Set(ticketReader, const_cast<const void**>(certificateList), sizeof(certificateList), true);

        if (esError != ES_ERR_OK)
        {
            NN_RESULT_THROW_UNLESS(!(esError > ES_ERR_CERT_BASE), ResultTicketInvalid());
            NN_RESULT_THROW(ResultCertificateInvalid());
        }
    }

    NN_RESULT_SUCCESS;
}

bool TicketDatabase::GetCertificateNameImpl(char* outCaName, char* outXsName, size_t caNameBufferSize, size_t xsNameBufferSize, const void* ticket, size_t ticketSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(static_cast<int>(caNameBufferSize) >= CertificateNameLength + 1);
    NN_SDK_ASSERT(static_cast<int>(xsNameBufferSize) >= CertificateNameLength + 1);

    const char* CaLabel = "CA";
    const char* XsLabel = "XS";

    MemoryInputStream ticketReader(ticket, static_cast<u32>(ticketSize));
    u8 issuer[64];

    // issuer を取得する
    {
        nn::escore::ETicket eTicket;
        nn::escore::ESError esError = eTicket.Set(ticketReader, nullptr, 0, false);
        if (esError != ES_ERR_OK)
        {
            return false;
        }

        esError = eTicket.GetIssuerName(issuer);
        if (esError != ES_ERR_OK)
        {
            return false;
        }
    }

    // 証明書の名前を取得する
    char* caNamePtr;
    caNamePtr = strstr(reinterpret_cast<char*>(issuer), CaLabel);
    memcpy(outCaName, caNamePtr, CertificateNameLength);
    outCaName[CertificateNameLength] = '\0';

    char* xsNamePtr;
    xsNamePtr = strstr(reinterpret_cast<char*>(issuer), XsLabel);
    memcpy(outXsName, xsNamePtr, CertificateNameLength);
    outXsName[CertificateNameLength] = '\0';

    return true;
}

int TicketDatabase::GetTicketInfoList(TicketInfo outTicketInfoList[], size_t ticketInfoListCount, const RightsIdIncludingKeyId rightsIdList[], size_t rightsIdListCount) const NN_NOEXCEPT
{
    int recordCount = 0;
    int index = 0;

    TicketRecord* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 空のレコードだったらチェックしない
            if (list[i].IsEmpty())
            {
                continue;
            }

            for (size_t j = 0; j < rightsIdListCount && static_cast<size_t>(recordCount) < ticketInfoListCount; j++)
            {
                // rightsIdList の各エントリと一致しているか確認する
                if (list[i].rightsId == rightsIdList[j])
                {
                    TicketRecord record;

                    m_TicketList.Get(&record, index);
                    GetTicketInfo(&outTicketInfoList[recordCount], record.rightsId, record.ticketId);

                    recordCount++;
                }
            }
        }
    }

    return recordCount;
}

int TicketDatabase::ListLightTicketInfo(LightTicketInfo outTicketInfoList[], size_t count, const RightsIdIncludingKeyId& rightsId) const NN_NOEXCEPT
{
    int recordCount = 0;
    int index = 0;

    TicketRecord* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 空のレコードだったらチェックしない
            if (list[i].IsEmpty())
            {
                continue;
            }

            if (list[i].rightsId == rightsId)
            {
                outTicketInfoList[recordCount].rightsId = list[i].rightsId;
                outTicketInfoList[recordCount].ticketId = list[i].ticketId;
                outTicketInfoList[recordCount].accountId = list[i].accountId;
                outTicketInfoList[recordCount].propertyMask = list[i].propertyMask;
                recordCount++;

                if (recordCount >= static_cast<int>(count))
                {
                    return recordCount;
                }
            }
        }
    }

    return recordCount;
}

void TicketDatabase::DeleteAllDeviceLinkedTicketExcludingList(const TicketId exclusionTicketIdList[], size_t count) NN_NOEXCEPT
{
    int index = 0;
    TicketRecord* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            if (list[i].IsEmpty())
            {
                continue;
            }

            // 動的チケットは削除しない
            if (list[i].IsElicenseRequired())
            {
                continue;
            }

            bool isDelete = true;

            for (size_t j = 0; j < count; j++)
            {
                if (list[i].ticketId == exclusionTicketIdList[j])
                {
                    isDelete = false;
                    break;
                }
            }

            if (isDelete)
            {
                // チケットを削除
                m_TicketList.Erase(index);
            }
        }
    }
}

void TicketMetaRecordDatabase::Initialize() NN_NOEXCEPT
{
    TicketDatabaseBase::Initialize();
    ETicketServiceDatabase::Initialize();
}

Result TicketMetaRecordDatabase::ImportTicket(TicketMetaRecord record) NN_NOEXCEPT
{
    // RightsId と TicketId が同一のチケットが既に存在していた場合
    if (m_TicketList.Find(record.rightsId, record.ticketId) != -1)
    {
        DeleteTicket(record.rightsId, record.ticketId);
    }
    else
    {
        // チケットの保存数が上限に達している場合
        NN_RESULT_THROW_UNLESS(!TicketDatabaseBase::IsFull(), ResultTicketFull());
    }

    m_TicketList.InsertWithoutCheckingDuplication(record);

    NN_RESULT_SUCCESS;
}

bool TicketMetaRecordDatabase::HasTicketMetaRecord(RightsId rightsId, TicketMetaRecordType type) const NN_NOEXCEPT
{
    int index = 0;

    TicketMetaRecord* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 空のレコードだったらチェックしない
            if (list[i].IsEmpty())
            {
                continue;
            }

            if (list[i].rightsId.GetRightsId() == rightsId && list[i].type == type)
            {
                return true;
            }
        }
    }

    return false;
}

int TicketMetaRecordDatabase::ListTicketMetaRecordInfo(TicketMetaRecord ticketMetaRecordList[], size_t count, const RightsIdIncludingKeyId& rightsId, TicketMetaRecordType type) const NN_NOEXCEPT
{
    int recordCount = 0;
    int index = 0;

    TicketMetaRecord* list;
    while (int readRecordCount = m_TicketList.ReadTicketList(&list, index))
    {
        for (int i = 0; i < readRecordCount; i++, index++)
        {
            // 空のレコードだったらチェックしない
            if (list[i].IsEmpty())
            {
                continue;
            }

            if (IsSameRightsId(list[i].rightsId, rightsId, type))
            {
                TicketMetaRecord record;
                m_TicketList.Get(&record, index);

                if (record.type == type)
                {
                    ticketMetaRecordList[recordCount] = record;
                    recordCount++;

                    if (recordCount >= static_cast<int>(count))
                    {
                        return recordCount;
                    }
                }
            }
        }
    }

    return recordCount;
}

bool TicketMetaRecordDatabase::IsSameRightsId(const RightsIdIncludingKeyId& lhs, const RightsIdIncludingKeyId& rhs, TicketMetaRecordType type) const NN_NOEXCEPT
{
    switch (type)
    {
    case nn::es::TicketMetaRecordType_Prepurchase:
        return lhs.GetRightsId() == rhs.GetRightsId();
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}
