﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/es.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/es/es_IETicketService.sfdl.h>
#include "es_ELicenseArchiveStore.h"
#include "es_ELicenseManager.h"
#include "es_ELicenseList.h"
#include "es_RightsAvailabilityMaster.h"
#include "es_TicketDatabase.h"
#include "es_UserList.h"
#include "es_ActiveRightsContextManager.h"

namespace nn { namespace es {

class ETicketServiceImpl
{
public:
    ETicketServiceImpl() NN_NOEXCEPT;

    // es_Api.h
    nn::Result ImportTicket(const nn::sf::InBuffer& ticket, const nn::sf::InBuffer& certificate) NN_NOEXCEPT;
    nn::Result ImportTicketCertificateSet(const nn::sf::InBuffer& ticketCertificateSet) NN_NOEXCEPT;
    nn::Result DeleteTicket(const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList) NN_NOEXCEPT;
    nn::Result DeletePersonalizedTicket(nn::es::AccountId accountId) NN_NOEXCEPT;
    nn::Result DeleteAllCommonTicket() NN_NOEXCEPT;
    nn::Result DeleteAllPersonalizedTicket() NN_NOEXCEPT;
    nn::Result DeleteAllPersonalizedTicketExcludingList(const nn::sf::InArray<nn::es::TicketId>& ticketIdExclusionList) NN_NOEXCEPT;
    nn::Result CountCommonTicket(nn::sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT;
    nn::Result CountPersonalizedTicket(nn::sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT;
    nn::Result ListCommonTicket(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList) NN_NOEXCEPT;
    nn::Result ListPersonalizedTicket(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList) NN_NOEXCEPT;
    nn::Result ListMissingPersonalizedTicket(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::TicketId>& outTicketIdList, const nn::sf::InArray<nn::es::TicketId>& ticketIdList) NN_NOEXCEPT;
    nn::Result GetCommonTicketSize(nn::sf::Out<std::uint64_t> pOutSize, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result GetCommonTicketData(nn::sf::Out<std::uint64_t> pOutSize, const nn::sf::OutBuffer& pOutBuffer, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result OwnTicket(const nn::sf::OutArray<bool>& outOwnTicketList, const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList) NN_NOEXCEPT;
    nn::Result GetTicketInfo(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::TicketInfo>& outTicketInfoList, const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList) NN_NOEXCEPT;
    nn::Result ListLightTicketInfo(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::LightTicketInfo>& outTicketInfoList, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result SignData(nn::sf::Out<nn::es::Sign> pOutSign, nn::sf::Out<nn::es::Certificate> pOutCertificate, const nn::sf::InBuffer& pData) NN_NOEXCEPT;
    nn::Result GetCommonTicketAndCertificateSize(nn::sf::Out<std::uint64_t> pOutTicketSize, nn::sf::Out<std::uint64_t> pOutCertificateSize, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result GetCommonTicketAndCertificateData(nn::sf::Out<std::uint64_t> pOutTicketSize, nn::sf::Out<std::uint64_t> pOutCertificateSize, const nn::sf::OutBuffer& pOutTicketBuffer, const nn::sf::OutBuffer& pOutCertificateBuffer, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result ImportPrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT;
    nn::Result DeletePrepurchaseRecord(const nn::es::PrepurchaseRecord& record) NN_NOEXCEPT;
    nn::Result DeleteAllPrepurchaseRecord() NN_NOEXCEPT;
    nn::Result CountPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT;
    nn::Result ListPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList) NN_NOEXCEPT;
    nn::Result ListPrepurchaseRecordInfo(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::PrepurchaseRecord>& outPrepurchaseRecordList, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result CountDesignatedTicket(nn::sf::Out<std::int32_t> pOutCount, nn::es::RightsId rightsId) NN_NOEXCEPT;
    nn::Result ListDesignatedTicket(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList, nn::es::RightsId rightsId) NN_NOEXCEPT;
    nn::Result CountDesignatedPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount, nn::es::RightsId rightsId) NN_NOEXCEPT;
    nn::Result ListDesignatedPrepurchaseRecord(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::RightsIdIncludingKeyId>& outRightsIdList, nn::es::RightsId rightsId) NN_NOEXCEPT;
    nn::Result GetEncryptedTicketSize(nn::sf::Out<std::uint64_t> pOutSize, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;
    nn::Result GetEncryptedTicketData(nn::sf::Out<std::uint64_t> pOutTicketId, nn::sf::Out<std::uint64_t> pOutSize, const nn::sf::OutBuffer& pOutBuffer, const nn::sf::OutBuffer& pOutKey, const nn::es::RightsIdIncludingKeyId& rightsId) NN_NOEXCEPT;

    // es_RightsApi.h
    nn::Result CheckRightsStatusIncludingKeyId(const nn::sf::OutArray<nn::es::RightsStatus>& outStatusList, const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList, const nn::sf::InArray<nn::es::ELicenseUserId>& userIdList, bool onlyAllAccountRights) NN_NOEXCEPT;
    nn::Result CheckRightsStatus(const nn::sf::OutArray<nn::es::RightsStatus>& outStatusList, const nn::sf::InArray<nn::es::RightsId>& rightsIdList, const nn::sf::InArray<nn::es::ELicenseUserId>& userIdList, bool onlyAllAccountRights) NN_NOEXCEPT;
    nn::Result RegisterTitleKey(const nn::sf::InArray<nn::es::RightsIdIncludingKeyId>& rightsIdList, const nn::sf::InArray<std::int32_t>& keyGenerationList) NN_NOEXCEPT;
    nn::Result UnregisterAllTitleKey() NN_NOEXCEPT;
    nn::Result RegisterAccountRestrictedRightsUser(nn::es::ELicenseUserId userId) NN_NOEXCEPT;
    nn::Result UnregisterAllAccountRestrictedRightsUser() NN_NOEXCEPT;
    nn::Result ListAccountRestrictedRightsUser(nn::sf::Out<bool> outHasAccount, nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::es::ELicenseUserId>& outUserIdList) NN_NOEXCEPT;

    // es_ELicenseApi.h
    nn::Result BeginImportELicenseArchive(nn::sf::Out<nn::es::ELicenseImportContext> pOutContext, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT;
    nn::Result ImportELicenseArchive(const nn::es::ELicenseImportContext& context, const nn::sf::InBuffer& eLicenseArchive) NN_NOEXCEPT;
    nn::Result EndImportELicenseArchive(nn::sf::Out<nn::es::ELicenseArchiveId> pOutELicenseArchiveId, const nn::es::ELicenseImportContext& context) NN_NOEXCEPT;
    nn::Result EndImportELicenseArchiveForDebug(nn::sf::Out<nn::es::ELicenseArchiveId> pOutELicenseArchiveId, const nn::es::ELicenseImportContext& context) NN_NOEXCEPT;
    nn::Result CountELicense(nn::sf::Out<std::int32_t> pOutCount) NN_NOEXCEPT;
    nn::Result ListELicenseIds(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::ELicenseId>& outELicenseIdList) NN_NOEXCEPT;
    nn::Result ListELicenseInfo(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::ELicenseInfoWrapper>& outELicenseInfoList, const nn::sf::InArray<nn::es::ELicenseId>& eLicenseIdList) NN_NOEXCEPT;
    nn::Result ListELicenseInfoForDebug(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::es::ELicenseInfoForSystemWrapper>& outELicenseInfoList, const nn::sf::InArray<nn::es::ELicenseId>& eLicenseIdList) NN_NOEXCEPT;
    nn::Result GetELicenseArchiveSizeForDebug(nn::sf::Out<std::uint64_t> pOutSize, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT;
    nn::Result GetELicenseArchiveDataForDebug(nn::sf::Out<std::uint64_t> pOutSize, const nn::sf::OutBuffer& pOutBuffer, nn::es::ELicenseOwnerId ownerId) NN_NOEXCEPT;
    nn::Result DeleteAllELicenseArchiveForDebug() NN_NOEXCEPT;

    // es_LocalConcurrencyCheckApi.h
    nn::Result GetChallengeOfLocalConcurrencyCheck(nn::sf::Out<nn::es::LocalConcurrencyCheckChallenge> outValue) NN_NOEXCEPT;
    nn::Result AuthorizeLocalConcurrencyCheckData(nn::sf::Out<nn::es::LocalConcurrencyCheckAuthenticationInfo> outInfo, const nn::sf::InArray<nn::es::LocalConcurrencyCheckData>& authenticatedDataList, const nn::es::LocalConcurrencyCheckData& unauthenticatedData) NN_NOEXCEPT;
    nn::Result GetLocalConcurrencyCheckData(nn::sf::Out<nn::es::LocalConcurrencyCheckData> outData, nn::es::LocalConcurrencyCheckChallenge challenge) NN_NOEXCEPT;
    nn::Result VerifyLocalConcurrencyAuthentication(const nn::es::LocalConcurrencyCheckAuthenticationInfo& info) NN_NOEXCEPT;

    nn::Result InitializeFile() NN_NOEXCEPT;

    // es_ActiveRightsContext.h
    nn::Result CreateActiveRightsContext(nn::sf::Out<nn::sf::SharedPointer<nn::es::IActiveRightsContext>> pOut) NN_NOEXCEPT;

private:
    template <typename T, typename ConvertELicenseInfoFunc>
    nn::Result ListELicenseInfoImpl(int* pOutCount, T* outELicenseInfoList, int eLicenseInfoCount, const nn::es::ELicenseId eLicenseIdList[], int eLicenseIdCount, ConvertELicenseInfoFunc convertFunc) NN_NOEXCEPT;

    void SetupTicketDB() NN_NOEXCEPT;
    void SetupDeviceCertificateAndDeviceKey() NN_NOEXCEPT;

    TicketDatabase m_CommonTicketDatabase;
    TicketDatabase m_PersonalizedTicketDatabase;
    CertificateDatabase m_CertificateDatabase;
    TicketMetaRecordDatabase m_TicketMetaRecordDatabase;
    ELicenseArchiveStore m_ELicenseArchiveStore;
    ELicenseList m_ELicenseList;

    ELicenseManager m_ELicenseManager;
    UserList m_RestrictedRightsUserList;

    nn::os::SdkMutex m_Mutex;
};

}} // namespace nn::es
