﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/err_SystemErrorArg.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>
#include <nn/settings/settings_Language.h>

namespace nn { namespace err {

SystemErrorArg::SystemErrorArg() NN_NOEXCEPT
    : m_pErrorContext(nullptr)
{
    m_StartupParam.common.errorType = ErrorType::SystemError;
    m_StartupParam.common.isJumpEnabled = false;
    m_StartupParam.errorCode.category = 0;
    m_StartupParam.errorCode.number = 0;
    util::Strlcpy(m_StartupParam.dialogMessage, "", SystemErrorArg::MessageLengthMax);
    util::Strlcpy(m_StartupParam.fullScreenMessage, "", SystemErrorArg::MessageLengthMax);
    util::Strlcpy(m_StartupParam.languageCode.string, "", sizeof(m_StartupParam.languageCode.string));
}

SystemErrorArg::SystemErrorArg(ErrorCode errorCode, const char* dialogViewMessage, const char* fullScreenViewMessage, const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
    : m_pErrorContext(nullptr)
{
    NN_SDK_REQUIRES(errorCode.IsValid());
    NN_SDK_REQUIRES_NOT_NULL(dialogViewMessage);
    NN_SDK_REQUIRES_NOT_NULL(fullScreenViewMessage);

    m_StartupParam.common.errorType = ErrorType::SystemError;
    m_StartupParam.common.isJumpEnabled = false;
    m_StartupParam.errorCode = errorCode;

    nn::util::Strlcpy(m_StartupParam.dialogMessage, dialogViewMessage, SystemErrorArg::MessageLengthMax);
    nn::util::Strlcpy(m_StartupParam.fullScreenMessage, fullScreenViewMessage, SystemErrorArg::MessageLengthMax);
    nn::util::Strlcpy(m_StartupParam.languageCode.string, languageCode.string, sizeof(m_StartupParam.languageCode.string));
}

void SystemErrorArg::SetErrorCode(ErrorCode errorCode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(errorCode.IsValid());
    m_StartupParam.errorCode = errorCode;
}

void SystemErrorArg::SetDialogMessage(const char* message) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(message);
    nn::util::Strlcpy(m_StartupParam.dialogMessage, message, SystemErrorArg::MessageLengthMax);
}

void SystemErrorArg::SetFullScreenMessage(const char* message) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(message);
    nn::util::Strlcpy(m_StartupParam.fullScreenMessage, message, SystemErrorArg::MessageLengthMax);
}

void SystemErrorArg::SetLanguageCode(const nn::settings::LanguageCode& languageCode) NN_NOEXCEPT
{
    nn::util::Strlcpy(m_StartupParam.languageCode.string, languageCode.string, sizeof(m_StartupParam.languageCode.string));
}

ErrorCode SystemErrorArg::GetErrorCode() const NN_NOEXCEPT
{
    return m_StartupParam.errorCode;
}

const char* SystemErrorArg::GetDialogMessage() const NN_NOEXCEPT
{
    return m_StartupParam.dialogMessage;
}

const char* SystemErrorArg::GetFullScreenMessage() const NN_NOEXCEPT
{
    return m_StartupParam.fullScreenMessage;
}

nn::settings::LanguageCode SystemErrorArg::GetLanguageCode() const NN_NOEXCEPT
{
    return m_StartupParam.languageCode;
}

const ErrorViewerStartupParamForSystemError& SystemErrorArg::GetStartupParam() const NN_NOEXCEPT
{
    return m_StartupParam;
}

void SystemErrorArg::AttachErrorContext(ErrorContext* pErrorContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pErrorContext);
    m_pErrorContext = pErrorContext;
    m_StartupParam.hasErrorContext = true;
}

void SystemErrorArg::DetachErrorContext() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pErrorContext);
    m_pErrorContext = nullptr;
    m_StartupParam.hasErrorContext = false;
}

const err::ErrorContext* SystemErrorArg::GetErrorContext() const NN_NOEXCEPT
{
    return m_pErrorContext;
}

}}
