﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include "erptsrv_Journal.h"
#include "erptsrv_Report.h"

namespace nn   {
namespace erpt {
namespace srv  {

nn::util::IntrusiveList<
         JournalRecord, nn::util::IntrusiveListBaseNodeTraits<JournalRecord>
       > Journal::m_RecordList;
uint32_t Journal::m_RecordCount = 0;
uint32_t Journal::m_RecordCountPerType[ReportType_Last] = {0};
uint32_t Journal::m_UsedStorage = 0;

JournalMeta Journal::m_JournalMeta;

nn::Result Journal::Store(JournalRecord* pJournalRecordIn)
NN_NOEXCEPT
{
    ReportType type;
    uint32_t   count;

    for (auto itr  = m_RecordList.begin();
              itr != m_RecordList.end();
              itr++)
    {
        if (itr->m_Info.reportId == pJournalRecordIn->m_Info.reportId)
        {
            return nn::erpt::ResultAlreadyExist();
        }
    }

    if (m_RecordCount >= NumberOfReports)
    {
        // Replacement policy:
        // - find report type with highest number of stored reports
        // - discard the oldest report of that type

        type  = pJournalRecordIn->m_Info.reportType;
        count = m_RecordCountPerType[type];

        for (int i = ReportType_First;
                 i < ReportType_Last;
                 i++)
        {
            if (m_RecordCountPerType[i] > count)
            {
                type = static_cast<ReportType>(i);
                count = m_RecordCountPerType[i];
            }
        }

        for (auto itr  = m_RecordList.rbegin();
                  itr != m_RecordList.rend();
                  itr++)
        {
            if (itr->m_Info.reportType == type)
            {
                JournalRecord* pOldRecord = &(*itr);
                m_RecordList.erase(m_RecordList.iterator_to(*pOldRecord));

                m_RecordCount--;
                m_RecordCountPerType[type]--;

                if (pOldRecord->RemoveReference())
                {
                    if (pOldRecord->m_Info.reportFlags.Test<ReportFlag::Transmitted>())
                    {
                        m_JournalMeta.transmittedCount[type]++;
                    }
                    else
                    {
                        m_JournalMeta.untransmittedCount[type]++;
                    }

                    // adjust used storage counter
                    m_UsedStorage -= static_cast<uint32_t>(pOldRecord->m_Info.reportSize);

                    Stream::DeleteStream(Report::FileName(pOldRecord->m_Info.reportId).name);
                    delete pOldRecord;
                }

                break;
            }
        }
    }

    pJournalRecordIn->AddReference();

    m_RecordList.push_front(*pJournalRecordIn);
    m_RecordCount++;
    m_RecordCountPerType[pJournalRecordIn->m_Info.reportType]++;
    m_UsedStorage += static_cast<uint32_t>(pJournalRecordIn->m_Info.reportSize);

    return ResultSuccess();
}

JournalRecord* Journal::Retrieve(ReportId reportId)
NN_NOEXCEPT
{
    for (auto itr  = m_RecordList.begin();
              itr != m_RecordList.end();
              itr++)
    {
        if (itr->m_Info.reportId == reportId)
        {
            return &(*itr);
        }
    }

    return nullptr;
}

nn::Result Journal::Commit()
NN_NOEXCEPT
{
    Stream     stream;
    nn::Result result;

    if ((result = stream.OpenStream(
            JournalFileName,
            StreamMode_Write,
            JournalStreamBufferSize)).IsFailure())
    {
        return result;
    }

    if ((result = stream.WriteStream(
            reinterpret_cast<const uint8_t*>(&m_RecordCount),
            sizeof(m_RecordCount))).IsFailure())
    {
        stream.CloseStream();
        return result;
    }

    for (auto itr  = m_RecordList.crbegin();
              itr != m_RecordList.crend();
              itr++)
    {
        if ((result = stream.WriteStream(
                reinterpret_cast<const uint8_t*>(&(itr->m_Info)),
                sizeof(itr->m_Info))).IsFailure())
        {
            stream.CloseStream();
            return result;
        }
    }

    if ((result = stream.WriteStream(
                reinterpret_cast<const uint8_t*>(&m_JournalMeta),
                sizeof(m_JournalMeta))).IsFailure())
    {
        stream.CloseStream();
        return result;
    }

    stream.CloseStream();
    stream.CommitStream();

    return ResultSuccess();
}

nn::Result Journal::GetReportList(ReportList* pReportList, ReportType filterType)
NN_NOEXCEPT
{
    uint32_t reportCount = 0;

    for (auto itr  = m_RecordList.cbegin();
              itr != m_RecordList.cend();
              itr++)
    {
        if (filterType == ReportType_Any ||
            filterType == itr->m_Info.reportType)
        {
            pReportList->Report[reportCount] = itr->m_Info;
            reportCount++;
        }
    }

    pReportList->reportCount = reportCount;

    return ResultSuccess();
}

void Journal::CleanupReports()
NN_NOEXCEPT
{
    for (auto itr  = m_RecordList.begin();
              itr != m_RecordList.end();
              )
    {
        JournalRecord* pRecord = &(*itr);
        itr = m_RecordList.erase(itr);
        Stream::DeleteStream(Report::FileName(pRecord->m_Info.reportId).name);
        delete pRecord;
    }

    m_RecordCount = 0;
    m_UsedStorage = 0;

    std::memset(m_RecordCountPerType, 0x0, sizeof(m_RecordCountPerType));
}

nn::Result Journal::Restore()
NN_NOEXCEPT
{
    nn::Result     result;
    Stream         stream;
    ReportInfo     info;
    JournalRecord* pRecord;
    uint32_t       recordCount;
    uint32_t       readCount;

    // to ensure reserved fields are set to 0
    std::memset(&m_JournalMeta, 0x0, sizeof(m_JournalMeta));

    m_JournalMeta.journalId      = util::GenerateUuid();
    m_JournalMeta.version        = JournalVersion;
    m_UsedStorage                = 0;

    if ((result = stream.OpenStream(
            JournalFileName,
            StreamMode_Read,
            JournalStreamBufferSize)).IsFailure())
    {
        return result;
    }

    if ((result = stream.ReadStream(
            &readCount,
            reinterpret_cast<uint8_t*>(&recordCount),
            sizeof(recordCount))).IsFailure())
    {
        return result;
    }

    if (readCount != sizeof(recordCount) || recordCount > NumberOfReports)
    {
        return nn::erpt::ResultCorruptJournal();
    }

    for (uint32_t i = 0; i < recordCount; i++)
    {
        if ((result = stream.ReadStream(
                &readCount,
                reinterpret_cast<uint8_t*>(&info),
                sizeof(info))).IsFailure())
        {
            CleanupReports();
            return result;
        }

        if (readCount != sizeof(info) ||
            info.reportType < ReportType_First ||
            info.reportType >= ReportType_Last)
        {
            CleanupReports();
            return nn::erpt::ResultCorruptJournal();
        }

        if ((pRecord = new JournalRecord(info)) == nullptr)
        {
            CleanupReports();
            return nn::erpt::ResultOutOfMemory();
        }

        if (pRecord->m_Info.reportSize == 0)
        {
            if (Stream::GetStreamSize(&pRecord->m_Info.reportSize, Report::FileName(pRecord->m_Info.reportId).name).IsFailure())
            {
                CleanupReports();
                return nn::erpt::ResultCorruptJournal();
            }
        }

        Store(pRecord);
    }

    // load journal meta data.
    // read may fail first time after update,
    // in this case we will start with the default values
    stream.ReadStream(
            &readCount,
            reinterpret_cast<uint8_t*>(&m_JournalMeta),
            sizeof(m_JournalMeta));

    return ResultSuccess();
}

nn::Result Journal::Delete(ReportId reportId)
NN_NOEXCEPT
{
    for( auto itr = m_RecordList.begin(); itr != m_RecordList.end(); itr++ )
    {
        JournalRecord* pRecord = &(*itr);
        if( pRecord->m_Info.reportId == reportId )
        {
            m_RecordList.erase(itr);
            m_RecordCount -= 1;
            m_RecordCountPerType[pRecord->m_Info.reportType] -= 1;
            if( pRecord->RemoveReference() )
            {
                Stream::DeleteStream(Report::FileName(reportId).name);
                m_UsedStorage -= static_cast<uint32_t>(pRecord->m_Info.reportSize);
                delete pRecord;
            }
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(erpt::ResultInvalidArgument());
}

uint32_t Journal::GetTransmittedCount(ReportType type)
NN_NOEXCEPT
{
    if( !(type >= ReportType_First && type < ReportType_Last) )
    {
        return 0;
    }
    return m_JournalMeta.transmittedCount[type];
}

uint32_t Journal::GetUnstransmittedCount(ReportType type)
NN_NOEXCEPT
{
    if( !(type >= ReportType_First && type < ReportType_Last) )
    {
        return 0;
    }
    return m_JournalMeta.untransmittedCount[type];
}

util::Uuid Journal::GetJournalId()
NN_NOEXCEPT
{
    return m_JournalMeta.journalId;
}

uint32_t Journal::GetStoredReportCount(ReportType type)
NN_NOEXCEPT
{
    if( !(type >= ReportType_First && type < ReportType_Last) )
    {
        return 0;
    }
    return m_RecordCountPerType[type];
}

int64_t Journal::GetMaxReportSize()
NN_NOEXCEPT
{
    int64_t maxSize = 0;
    for( auto itr = m_RecordList.begin(); itr != m_RecordList.end(); itr++ )
    {
        if( itr->m_Info.reportSize > maxSize )
        {
            maxSize = itr->m_Info.reportSize;
        }
    }
    return maxSize;
}

uint32_t Journal::GetUsedStorage()
NN_NOEXCEPT
{
    return m_UsedStorage;
}

}}}
