﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/erpt/erpt_Result.h>

#include "erptsrv_IReport.h"
#include "erptsrv_Report.h"

namespace nn   {
namespace erpt {
namespace srv  {

nn::Result IReport::Open(ReportId reportId)
NN_NOEXCEPT
{
    nn::Result     result;
    JournalRecord* pJournalRecord;

    if (m_pReport)
    {
        return nn::erpt::ResultAlreadyInitialized();
    }
    if ((pJournalRecord = Journal::Retrieve(reportId)) == nullptr)
    {
        return nn::erpt::ResultNotFound();
    }
    if ((m_pReport = new Report(pJournalRecord)) == nullptr)
    {
        return nn::erpt::ResultOutOfMemory();
    }
    if ((result = m_pReport->Open(ReportOpenType_Read)).IsFailure())
    {
        delete m_pReport;
        m_pReport = nullptr;
    }

    return result;
}

nn::Result IReport::Read(nn::sf::Out<uint32_t> readCountOut, const nn::sf::OutBuffer& readBuffer)
NN_NOEXCEPT
{
    return m_pReport ?
           m_pReport->Read(&(*readCountOut), reinterpret_cast<uint8_t*>(readBuffer.GetPointerUnsafe()), static_cast<uint32_t>(readBuffer.GetSize())) :
           nn::erpt::ResultNotInitialized();
}

nn::Result IReport::SetFlags(ReportFlagSet flagsIn)
NN_NOEXCEPT
{
    return m_pReport ?
           m_pReport->SetFlags(flagsIn) :
           nn::erpt::ResultNotInitialized();
}

nn::Result IReport::GetFlags(nn::sf::Out<ReportFlagSet> pFlagsOut)
NN_NOEXCEPT
{
    return m_pReport ?
           m_pReport->GetFlags(&(*pFlagsOut)) :
           nn::erpt::ResultNotInitialized();
}

nn::Result IReport::GetSize(nn::sf::Out<int64_t> sizeOut)
NN_NOEXCEPT
{
    return m_pReport ?
           m_pReport->GetSize(&(*sizeOut)) :
           nn::erpt::ResultNotInitialized();
}

nn::Result IReport::Close()
NN_NOEXCEPT
{
    if (m_pReport)
    {
        m_pReport->Close();
        delete m_pReport;
        m_pReport = nullptr;
    }

    return ResultSuccess();
}

IReport::IReport()
NN_NOEXCEPT :
    m_pReport(nullptr)
{

}

IReport::~IReport()
NN_NOEXCEPT
{
    Close();
}

}}}
