﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/time.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/util/util_StringUtil.h>

#include "erptsrv_Context.h"
#include "erptsrv_Report.h"
#include "erptsrv_ContextRecord.h"

namespace nn   {
namespace erpt {
namespace srv  {

nn::util::IntrusiveList<
          Context, nn::util::IntrusiveListBaseNodeTraits<Context>
        > Context::s_CategoryList;

char Context::s_SerialNumber[24]     = "Unknown";
char Context::s_OsVersion[24]        = "Unknown";
char Context::s_PrivateOsVersion[96] = "Unknown";
util::optional<time::SteadyClockTimePoint> Context::s_InitialLaunchSettingsCompletionTime = nullptr;
util::optional<time::SteadyClockTimePoint> Context::s_PowerOnTime = nullptr;
util::optional<time::SteadyClockTimePoint> Context::s_AwakeTime = nullptr;

ReportId Context::GenerateReportId()
    NN_NOEXCEPT
{
    ReportId reportId{};
    reportId.u.uuidRFC4122 = util::GenerateUuid();
    return reportId;
}

nn::Result Context::AddCategoryToReport(
    Report* pReportIn)
NN_NOEXCEPT
{
    nn::Result result;

    if (m_RecordList.empty())
    {
        return ResultSuccess();
    }

    for (auto itr  = m_RecordList.begin();
              itr != m_RecordList.end();
              itr++)
    {
        for (uint32_t i = 0; i < itr->m_Context.fieldCount; i++)
        {
            FieldEntry* pField       = &(itr->m_Context.fields[i]);
            uint8_t*    pArrayBuffer = itr->m_Context.pArrayBuffer;

            switch (pField->type)
            {
            case FieldType_Bool:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.boolField);
                break;
            case FieldType_NumericU64:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldU64);
                break;
            case FieldType_NumericI64:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldI64);
                break;
            case FieldType_NumericU32:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldU32);
                break;
            case FieldType_NumericI32:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldI32);
                break;
            case FieldType_NumericU16:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldU16);
                break;
            case FieldType_NumericI16:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldI16);
                break;
            case FieldType_NumericU8:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldU8);
                break;
            case FieldType_NumericI8:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            pField->u.numericFieldI8);
                break;
            case FieldType_String:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<char*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size);
                break;
            case FieldType_U8Array:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<uint8_t*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size);
                break;
            case FieldType_U32Array:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<uint32_t*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size / sizeof(uint32_t));
                break;
            case FieldType_U64Array:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<uint64_t*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size / sizeof(uint64_t));
                break;
            case FieldType_I8Array:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<int8_t*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size / sizeof(int8_t));
                break;
            case FieldType_I32Array:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<int32_t*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size / sizeof(int32_t));
                break;
            case FieldType_I64Array:
                result = Cipher::AddField(
                            pReportIn,
                            pField->id,
                            reinterpret_cast<int64_t*>(&pArrayBuffer[pField->u.arrayField.startIndex]),
                            pField->u.arrayField.size / sizeof(int64_t));
                break;
            default:
                result = nn::erpt::ResultInvalidArgument();
                break;
            }

            if (result.IsFailure())
            {
                return result;
            }
        }
    }

    return ResultSuccess();
} // NOLINT(impl/function_size)

nn::Result Context::AddContextToCategory(
    ContextRecord* pContextRecord)
NN_NOEXCEPT
{
    if (m_RecordCount < m_MaxRecordCount)
    {
        m_RecordList.push_front(*pContextRecord);
        m_RecordCount++;
    }
    else
    {
        ContextRecord* tmp = &m_RecordList.back();
        m_RecordList.pop_back();
        m_RecordList.push_front(*pContextRecord);
        delete tmp;
    }

    return ResultSuccess();
}

nn::Result Context::AddContextToCategory(
    const ContextEntry* pContextIn,
    const uint8_t*      pDataIn,
    uint32_t            dataLength)
NN_NOEXCEPT
{
    nn::Result     result;
    ContextRecord* pContextRecord;

    if ((pContextRecord = new ContextRecord()) == nullptr)
    {
        return nn::erpt::ResultOutOfMemory();
    }

    if ((result = pContextRecord->Initialize(pContextIn, pDataIn, dataLength)).IsSuccess())
    {
        AddContextToCategory(pContextRecord);
    }
    else
    {
        delete pContextRecord;
    }

    return result;
}

nn::Result Context::CreateReportFile(
    JournalRecord* pJournalRecordIn)
NN_NOEXCEPT
{
    nn::Result result;
    Report*    pReport;

    if ((pReport = new Report(pJournalRecordIn)) == nullptr)
    {
        return nn::erpt::ResultOutOfMemory();
    }

    if ((result = pReport->Open(ReportOpenType_Create)).IsSuccess())
    {
        if ((result = Cipher::Begin(pReport, ContextRecord::GetRecordCount())).IsSuccess())
        {
            for (auto itr  = s_CategoryList.begin();
                      itr != s_CategoryList.end();
                      itr++)
            {
                if ((result = itr->AddCategoryToReport(pReport)).IsFailure())
                {
                    break;
                }
            }
        }

        Cipher::End(pReport);

        pReport->Close();

        pReport->GetSize(&pJournalRecordIn->m_Info.reportSize);

        if (result.IsSuccess() && (result = Journal::Store(pJournalRecordIn)).IsSuccess())
        {
            // at this point report and
            // its journal record are stored in fs
            Journal::Commit();
        }

        if (result.IsFailure())
        {
            pReport->Delete();
        }
    }

    delete pReport;

    return result;
}

nn::Result Context::CreateJournalRecord(
    JournalRecord**       pJournalRecordOut,
    ReportType            type,
    const ReportMetaData* pMeta)
NN_NOEXCEPT
{
    JournalRecord*      pJournalRecord;
    nn::time::PosixTime timeStamp;
    nn::time::PosixTime timeStampNet;
    nn::Result          result;

    if ((result = nn::time::StandardUserSystemClock::GetCurrentTime(&timeStamp)).IsFailure())
    {
        return result;
    }

    if ((nn::time::StandardNetworkSystemClock::GetCurrentTime(&timeStampNet)).IsFailure())
    {
        timeStampNet.value = 0;
    }

    if ((pJournalRecord = new JournalRecord()) == nullptr)
    {
        return nn::erpt::ResultOutOfMemory();
    }

    pJournalRecord->m_Info.reportTimeStamp    = timeStamp;
    pJournalRecord->m_Info.reportTimeStampNet = timeStampNet;
    pJournalRecord->m_Info.reportType         = type;
    pJournalRecord->m_Info.reportId           = GenerateReportId();
    pJournalRecord->m_Info.reportFlags        = MakeNoReportFlags();

    if (pMeta)
    {
        pJournalRecord->m_Info.reportMetaData = *pMeta;
    }

    *pJournalRecordOut = pJournalRecord;

    return ResultSuccess();
}

nn::Result Context::SubmitErrorInfoAutoContext(
    JournalRecord*        pJournalRecord,
    ContextRecord*        pContextRecord
)
NN_NOEXCEPT
{
    char name[64];

    pJournalRecord->m_Info.reportId.u.uuidRFC4122.ToString(name, sizeof(name));

    pContextRecord->Add(
        ReportIdentifier,
        name,
        static_cast<uint32_t>(nn::util::Strnlen(name, sizeof(name))));

    pContextRecord->Add(
        OccurrenceTimestamp,
        pJournalRecord->m_Info.reportTimeStamp.value);

    pContextRecord->Add(
        OccurrenceTimestampNet,
        pJournalRecord->m_Info.reportTimeStampNet.value);

    pContextRecord->Add(
        OsVersion,
        s_OsVersion,
        static_cast<uint32_t>(nn::util::Strnlen(s_OsVersion, sizeof(s_OsVersion))));

    pContextRecord->Add(
        PrivateOsVersion,
        s_PrivateOsVersion,
        static_cast<uint32_t>(nn::util::Strnlen(s_PrivateOsVersion, sizeof(s_PrivateOsVersion))));

    pContextRecord->Add(
        SerialNumber,
        s_SerialNumber,
        static_cast<uint32_t>(nn::util::Strnlen(s_SerialNumber, sizeof(s_SerialNumber))));

    pContextRecord->Add(
        ReportVisibilityFlag,
        pJournalRecord->m_Info.reportType == ReportType_Visible ?
        true : false);

    time::SteadyClockTimePoint currentTimePoint;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));

    pContextRecord->Add(SteadyClockCurrentTimePointValue, currentTimePoint.value);
    pContextRecord->Add(SteadyClockInternalOffset, time::GetStandardSteadyClockInternalOffset().GetSeconds());

    if( s_InitialLaunchSettingsCompletionTime )
    {
        int64_t elapsedSeconds;
        if( time::GetSpanBetween(&elapsedSeconds, *s_InitialLaunchSettingsCompletionTime, currentTimePoint).IsSuccess() )
        {
            pContextRecord->Add(ElapsedTimeSinceInitialLaunch, elapsedSeconds);
        }
    }

    if( s_PowerOnTime )
    {
        int64_t elapsedSeconds;
        if( time::GetSpanBetween(&elapsedSeconds, *s_PowerOnTime, currentTimePoint).IsSuccess() )
        {
            pContextRecord->Add(ElapsedTimeSincePowerOn, elapsedSeconds);
        }
    }

    if( s_AwakeTime )
    {
        int64_t elapsedSeconds;
        if( time::GetSpanBetween(&elapsedSeconds, *s_AwakeTime, currentTimePoint).IsSuccess() )
        {
            pContextRecord->Add(ElapsedTimeSinceLastAwake, elapsedSeconds);
        }
    }
    pContextRecord->Add(OccurrenceTick, os::GetSystemTick().GetInt64Value());

    return SubmitContext(pContextRecord);
}

nn::Result Context::ValidateReportContext(
    const ContextEntry* pContextIn)
NN_NOEXCEPT
{
    bool foundErrorCode = false;

    if (pContextIn->category != ErrorInfo)
    {
        NN_SDK_LOG("[erpt]: create report API requires context initialized with category '%s'\n",
                   CategoryString[ErrorInfo]
        );
        return nn::erpt::ResultRequiredContextMissing();
    }

    if (pContextIn->fieldCount > FieldsPerContext)
    {
        return nn::erpt::ResultInvalidArgument();
    }

    for (uint32_t i = 0; i < pContextIn->fieldCount; i++)
    {
        if (pContextIn->fields[i].id == ErrorCode)
        {
            foundErrorCode = true;
        }
    }

    if (!foundErrorCode)
    {
        NN_SDK_LOG("[erpt]: create report API requires at least '%s' field to be passed inside context\n",
                   FieldString[ErrorCode]
        );
        return nn::erpt::ResultRequiredFieldMissing();
    }

    return ResultSuccess();
}

nn::Result Context::SubmitContext(
    const ContextEntry* pContextIn,
    const uint8_t*      pDataIn,
    uint32_t            dataLength)
NN_NOEXCEPT
{
    for (auto itr  = s_CategoryList.begin();
              itr != s_CategoryList.end();
              itr++)
    {
        if (itr->m_Category == pContextIn->category)
        {
            return itr->AddContextToCategory(pContextIn, pDataIn, dataLength);
        }
    }

    return nn::erpt::ResultCategoryNotFound();
}

nn::Result Context::SubmitContext(
    ContextRecord* pContextRecordIn)
NN_NOEXCEPT
{
    for (auto itr  = s_CategoryList.begin();
              itr != s_CategoryList.end();
              itr++)
    {
        if (itr->m_Category == pContextRecordIn->m_Context.category)
        {
            return itr->AddContextToCategory(pContextRecordIn);
        }
    }

    return nn::erpt::ResultCategoryNotFound();
}

nn::Result Context::SubmitReportDefaults(
    const ContextEntry* pContextIn)
NN_NOEXCEPT
{
    nn::Result     result;
    ContextRecord* pContextRecord;

    #if !defined(NN_BUILD_CONFIG_OS_WIN)
    ReportDefaultValue defaultValues[] =
    {
        { .id = AbortFlag, .type = FieldType_Bool, .value.boolField = false, .exists = false },
    };
    #else
    ReportDefaultValue defaultValues[1];
    defaultValues[0].id              = AbortFlag;
    defaultValues[0].type            = FieldType_Bool;
    defaultValues[0].value.boolField = false;
    defaultValues[0].exists          = false;
    #endif

    for (uint32_t i = 0; i < sizeof(defaultValues) / sizeof(defaultValues[0]); i++)
    {
        for (uint32_t j = 0; j < pContextIn->fieldCount; j++)
        {
            if (defaultValues[i].id == pContextIn->fields[j].id)
            {
                defaultValues[i].exists = true;
                break;
            }
        }
    }

    if ((pContextRecord = new ContextRecord(ErrorInfoDefaults)) != nullptr)
    {
        for (uint32_t i = 0; i < sizeof(defaultValues) / sizeof(defaultValues[0]); i++)
        {
            if (!defaultValues[i].exists)
            {
                switch (defaultValues[i].type)
                {
                case FieldType_NumericU64:
                    pContextRecord->Add(defaultValues[i].id, defaultValues[i].value.numericFieldU64);
                    break;
                case FieldType_NumericU32:
                    pContextRecord->Add(defaultValues[i].id, defaultValues[i].value.numericFieldU32);
                    break;
                case FieldType_NumericI64:
                    pContextRecord->Add(defaultValues[i].id, defaultValues[i].value.numericFieldI64);
                    break;
                case FieldType_NumericI32:
                    pContextRecord->Add(defaultValues[i].id, defaultValues[i].value.numericFieldI32);
                    break;
                case FieldType_Bool:
                    pContextRecord->Add(defaultValues[i].id, defaultValues[i].value.boolField);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }

        if ((result = SubmitContext(pContextRecord)).IsFailure())
        {
            delete pContextRecord;
        }
    }
    else
    {
        result = nn::erpt::ResultOutOfMemory();
    }

    return result;
}

nn::Result Context::CreateReport(
    ReportType            type,
    const ContextEntry*   pContextIn,
    const uint8_t*        pDataIn,
    uint32_t              dataLength,
    const ReportMetaData* pMetaIn)
NN_NOEXCEPT
{
    nn::Result     result;
    JournalRecord* pJournalRecord;
    ContextRecord* pContextRecord;

    if ((result = ValidateReportContext(pContextIn)).IsFailure() ||
        (result = SubmitContext(pContextIn, pDataIn, dataLength)).IsFailure() ||
        (result = SubmitReportDefaults(pContextIn)).IsFailure() ||
        (result = CreateJournalRecord(&pJournalRecord, type, pMetaIn)).IsFailure())
    {
        return result;
    }

    pJournalRecord->AddReference();

    if ((pContextRecord = new ContextRecord(ErrorInfoAuto)) != nullptr)
    {
        if ((result = SubmitErrorInfoAutoContext(pJournalRecord, pContextRecord)).IsSuccess())
        {
            result = CreateReportFile(pJournalRecord);
        }
        else
        {
            delete pContextRecord;
        }
    }
    else
    {
        result = nn::erpt::ResultOutOfMemory();
    }

    if (pJournalRecord->RemoveReference())
    {
        delete pJournalRecord;
    }

    return result;
}

nn::Result Context::SetSerialNumberAndOsVersion(
    const char* pSerialNumber,     uint32_t serialNumberLength,
    const char* pOsVersion,        uint32_t osVersionLength,
    const char* pPrivateOsVersion, uint32_t privateOsVersionLength)
NN_NOEXCEPT
{
#if !defined (NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013)
    NN_STATIC_ASSERT(sizeof(Context::s_SerialNumber) >= sizeof(settings::system::SerialNumber::string));
    NN_STATIC_ASSERT(sizeof(Context::s_OsVersion) >= sizeof(settings::system::FirmwareVersion::displayVersion));
#endif

    if (serialNumberLength     > sizeof(s_SerialNumber) ||
        osVersionLength        > sizeof(s_OsVersion)    ||
        privateOsVersionLength > sizeof(s_PrivateOsVersion))
    {
        return nn::erpt::ResultInvalidArgument();
    }

    std::memcpy(s_SerialNumber,     pSerialNumber,      serialNumberLength);
    std::memcpy(s_OsVersion,        pOsVersion,         osVersionLength);
    std::memcpy(s_PrivateOsVersion, pPrivateOsVersion,  privateOsVersionLength);

    return ResultSuccess();
}

void Context::SetInitialLaunchSettingsCompletionTime(
    const time::SteadyClockTimePoint& timePoint
)
NN_NOEXCEPT
{
    s_InitialLaunchSettingsCompletionTime = timePoint;
}

void Context::ClearInitialLaunchSettingsCompletionTime()
NN_NOEXCEPT
{
    s_InitialLaunchSettingsCompletionTime = nullptr;
}

void Context::UpdatePowerOnTime()
NN_NOEXCEPT
{
    s_PowerOnTime.emplace();
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&(*s_PowerOnTime)));
}

void Context::UpdateAwakeTime()
NN_NOEXCEPT
{
    s_AwakeTime.emplace();
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&(*s_AwakeTime)));
}

Context::Context(CategoryId id, uint32_t maxRecordCount)
NN_NOEXCEPT :
    m_Category(id),
    m_MaxRecordCount(maxRecordCount),
    m_RecordCount(0)
{
    s_CategoryList.push_front(*this);
}

Context::~Context()
NN_NOEXCEPT
{
    s_CategoryList.erase(s_CategoryList.iterator_to(*this));
}

}}}
