﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/erpt/erpt_MultipleCategoryContext.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_Server.h>

#include "erpt_Session.h"

namespace nn   {
namespace erpt {

void MultipleCategoryContext::Reset() NN_NOEXCEPT
{
    m_TotalFieldCount = 0;
    m_ArrayFreeCount = m_ArrayBufferSize;
    m_ArrayBufferOffset = 0;

    m_Context.categoryCount = 0;
    std::memset(m_Context.categories, 0, sizeof(m_Context.categories));
    std::memset(m_Context.fieldCounts, 0, sizeof(m_Context.fieldCounts));
    std::memset(m_Context.fields, 0, sizeof(m_Context.fields));
}

nn::Result MultipleCategoryContext::MoveToNextCategory(nn::erpt::CategoryId id) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.categoryCount < CategoriesPerMultipleCategoryContext, ResultUnknown()); // TODO: Result

    m_ArrayBufferOffset = (m_ArrayBufferSize - m_ArrayFreeCount);
    m_Context.categories[m_Context.categoryCount] = id;
    m_Context.categoryCount += 1;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, bool valueBool)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id   = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_Bool;
    m_Context.fields[m_TotalFieldCount].u.boolField = valueBool;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, uint64_t valueU64)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id   = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericU64;
    m_Context.fields[m_TotalFieldCount].u.numericFieldU64 = valueU64;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, uint32_t valueU32)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id   = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericU32;
    m_Context.fields[m_TotalFieldCount].u.numericFieldU32 = valueU32;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, uint16_t valueU16)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericU16;
    m_Context.fields[m_TotalFieldCount].u.numericFieldU16 = valueU16;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, uint8_t valueU8)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericU8;
    m_Context.fields[m_TotalFieldCount].u.numericFieldU8 = valueU8;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, int64_t valueI64)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id   = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericI64;
    m_Context.fields[m_TotalFieldCount].u.numericFieldI64 = valueI64;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, int32_t valueI32)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id   = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericI32;
    m_Context.fields[m_TotalFieldCount].u.numericFieldI32 = valueI32;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, int16_t valueI16)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericI16;
    m_Context.fields[m_TotalFieldCount].u.numericFieldI16 = valueI16;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, int8_t valueI8)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_TotalFieldCount].id = id;
    m_Context.fields[m_TotalFieldCount].type = FieldType_NumericI8;
    m_Context.fields[m_TotalFieldCount].u.numericFieldI8 = valueI8;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, const uint8_t* pBuffer, uint32_t bufferLength, FieldType type)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCounts[m_Context.categoryCount - 1] < FieldsPerContext, ResultOutOfFieldSpace());
    NN_RESULT_THROW_UNLESS(bufferLength <= m_ArrayFreeCount, ResultOutOfArraySpace());

    uint32_t startIndex = m_ArrayBufferSize - m_ArrayFreeCount;

    m_Context.fields[m_TotalFieldCount].id   = id;
    m_Context.fields[m_TotalFieldCount].type = type;
    m_Context.fields[m_TotalFieldCount].u.arrayField.size = bufferLength;
    m_Context.fields[m_TotalFieldCount].u.arrayField.startIndex = (startIndex - m_ArrayBufferOffset);
    m_Context.arrayBufferCounts[m_Context.categoryCount - 1] += bufferLength;
    m_ArrayFreeCount -= bufferLength;
    m_Context.fieldCounts[m_Context.categoryCount - 1]++;
    m_TotalFieldCount++;

    std::memcpy(m_pArrayBuffer + startIndex, pBuffer, bufferLength);

    NN_RESULT_SUCCESS;
}

nn::Result MultipleCategoryContext::Add(FieldId id, const uint8_t* pBuffer, uint32_t bufferLength)
NN_NOEXCEPT
{
    return Add(id, pBuffer, bufferLength, FieldType_U8Array);
}

nn::Result MultipleCategoryContext::Add(FieldId id, const uint32_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_U32Array);
}

nn::Result MultipleCategoryContext::Add(FieldId id, const uint64_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_U64Array);
}

nn::Result MultipleCategoryContext::Add(FieldId id, const int32_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_I32Array);
}

nn::Result MultipleCategoryContext::Add(FieldId id, const int64_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_I64Array);
}

nn::Result MultipleCategoryContext::Add(FieldId id, const char* pBuffer, uint32_t bufferLength)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pBuffer), bufferLength, FieldType_String);
}

nn::Result MultipleCategoryContext::SubmitContext()
NN_NOEXCEPT
{
    nn::sf::SharedPointer<sf::IContext> pContextImpl;

    NN_RESULT_DO(GetContextObject(&pContextImpl));

    nn::sf::InBuffer data(reinterpret_cast<char*>(m_pArrayBuffer), m_ArrayBufferSize - m_ArrayFreeCount);

    NN_RESULT_DO(pContextImpl->SubmitMultipleCategoryContext(m_Context, data));
    NN_RESULT_SUCCESS;
}

MultipleCategoryContext::MultipleCategoryContext() NN_NOEXCEPT
    : m_TotalFieldCount(0)
    , m_Context()
    , m_ArrayBufferOffset(0)
{
    m_Context.version = ERRVERSION;

    m_pArrayBuffer = m_ArrayBuffer;
    m_ArrayBufferSize = sizeof(m_ArrayBuffer);
    m_ArrayFreeCount = sizeof(m_ArrayBuffer);
}

MultipleCategoryContext::MultipleCategoryContext(uint8_t* arrayBuffer, uint32_t arrayBufferSize) NN_NOEXCEPT
    : m_TotalFieldCount(0)
    , m_Context()
    , m_ArrayBufferOffset(0)
{
    m_Context.version = ERRVERSION;

    m_pArrayBuffer = arrayBuffer;
    m_ArrayBufferSize = arrayBufferSize;
    m_ArrayFreeCount = arrayBufferSize;
}

}}
