﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/erpt/erpt_Manager.h>

#include "erpt_Session.h"

namespace nn   {
namespace erpt {

nn::Result Manager::GetReportList(ReportList& reportList)
NN_NOEXCEPT
{
    nn::sf::OutBuffer list(reinterpret_cast<char*>(&reportList), sizeof(ReportList));
    return m_Impl != nullptr ?
           m_Impl->GetReportList(list, ReportType_Any) :
           nn::erpt::ResultNotInitialized();
}

nn::Result Manager::GetReportList(ReportList& reportList, ReportType filterType)
NN_NOEXCEPT
{
    nn::sf::OutBuffer list(reinterpret_cast<char*>(&reportList), sizeof(ReportList));
    return m_Impl != nullptr ?
           m_Impl->GetReportList(list, filterType) :
           nn::erpt::ResultNotInitialized();
}

nn::os::SystemEvent* Manager::GetEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

nn::Result Manager::CleanupReports()
NN_NOEXCEPT
{
    return m_Impl != nullptr ?
           m_Impl->CleanupReports() :
           nn::erpt::ResultNotInitialized();
}

nn::Result Manager::DeleteReport(ReportId reportId)
NN_NOEXCEPT
{
    return m_Impl != nullptr ?
           m_Impl->DeleteReport(reportId) :
           nn::erpt::ResultNotInitialized();
}

nn::Result Manager::GetStorageUsageStatistics(StorageUsageStatistics* pOutValue)
NN_NOEXCEPT
{
    return m_Impl != nullptr ?
           m_Impl->GetStorageUsageStatistics(pOutValue) :
           nn::erpt::ResultNotInitialized();
}

nn::Result Manager::Initialize()
NN_NOEXCEPT
{
    nn::Result result;
    nn::sf::NativeHandle sfHandle;
    nn::sf::SharedPointer<nn::erpt::sf::ISession> p_Session;

    if (m_Impl != nullptr)
    {
        return nn::erpt::ResultAlreadyInitialized();
    }

    if ((result = GetSessionObject(&p_Session)).IsSuccess() &&
        (result = p_Session->OpenManager(&m_Impl)).IsSuccess())
    {
        if ((result = m_Impl->GetEvent(&sfHandle)).IsSuccess())
        {
            m_SystemEvent.AttachReadableHandle(
                            sfHandle.GetOsHandle(),
                            sfHandle.IsManaged(),
                            nn::os::EventClearMode_AutoClear);

            sfHandle.Detach();
        }
        else
        {
            m_Impl    = nullptr;
            p_Session = nullptr;
            ReleaseSessionObject();
        }
    }

    return result;
}

void Manager::Finalize()
NN_NOEXCEPT
{
    if (m_Impl != nullptr)
    {
        m_Impl = nullptr;
        ReleaseSessionObject();
        nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    }
}

Manager::Manager()
NN_NOEXCEPT :
    m_Impl(nullptr)
{

}

Manager::~Manager()
NN_NOEXCEPT
{
    Finalize();
}

}}
