﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/util/util_StringUtil.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>

#include <nn/erpt/erpt_Context.h>
#include <nn/erpt/erpt_Result.h>
#include <nn/erpt/server/erpt_Server.h>

#include <nn/err/err_SystemApi.h>
#include <nn/err/err_Types.h>

#include "erpt_Session.h"

namespace nn   {
namespace erpt {

nn::Result Context::Add(FieldId id, bool valueBool)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_Bool;
    m_Context.fields[m_Context.fieldCount].u.boolField = valueBool;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, uint64_t valueU64)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericU64;
    m_Context.fields[m_Context.fieldCount].u.numericFieldU64 = valueU64;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, uint32_t valueU32)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericU32;
    m_Context.fields[m_Context.fieldCount].u.numericFieldU32 = valueU32;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, uint16_t valueU16)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericU16;
    m_Context.fields[m_Context.fieldCount].u.numericFieldU16 = valueU16;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, uint8_t valueU8)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericU8;
    m_Context.fields[m_Context.fieldCount].u.numericFieldU8 = valueU8;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, int64_t valueI64)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericI64;
    m_Context.fields[m_Context.fieldCount].u.numericFieldI64 = valueI64;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, int32_t valueI32)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericI32;
    m_Context.fields[m_Context.fieldCount].u.numericFieldI32 = valueI32;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, int16_t valueI16)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericI16;
    m_Context.fields[m_Context.fieldCount].u.numericFieldI16 = valueI16;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, int8_t valueI8)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());

    m_Context.fields[m_Context.fieldCount].id = id;
    m_Context.fields[m_Context.fieldCount].type = FieldType_NumericI8;
    m_Context.fields[m_Context.fieldCount].u.numericFieldI8 = valueI8;
    m_Context.fieldCount++;

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, const uint8_t* pBuffer, uint32_t bufferLength, FieldType type)
NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Context.fieldCount < FieldsPerContext, ResultOutOfFieldSpace());
    NN_RESULT_THROW_UNLESS(bufferLength <= m_Context.arrayFreeCount, ResultOutOfArraySpace());

    uint32_t startIndex = m_Context.arrayBufferSize - m_Context.arrayFreeCount;

    m_Context.fields[m_Context.fieldCount].id   = id;
    m_Context.fields[m_Context.fieldCount].type = type;
    m_Context.fields[m_Context.fieldCount].u.arrayField.size = bufferLength;
    m_Context.fields[m_Context.fieldCount].u.arrayField.startIndex = startIndex;
    m_Context.arrayFreeCount -= bufferLength;
    m_Context.fieldCount++;

    std::memcpy(m_Context.pArrayBuffer + startIndex, pBuffer, bufferLength);

    NN_RESULT_SUCCESS;
}

nn::Result Context::Add(FieldId id, const uint8_t* pBuffer, uint32_t bufferLength)
NN_NOEXCEPT
{
    return Add(id, pBuffer, bufferLength, FieldType_U8Array);
}

nn::Result Context::Add(FieldId id, const uint32_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_U32Array);
}

nn::Result Context::Add(FieldId id, const uint64_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_U64Array);
}

nn::Result Context::Add(FieldId id, const int8_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_I8Array);
}

nn::Result Context::Add(FieldId id, const int32_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_I32Array);
}

nn::Result Context::Add(FieldId id, const int64_t* pArray, uint32_t elementCount)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pArray), elementCount * sizeof(pArray[0]), FieldType_I64Array);
}

nn::Result Context::Add(FieldId id, const char* pBuffer, uint32_t bufferLength)
NN_NOEXCEPT
{
    return Add(id, reinterpret_cast<const uint8_t*>(pBuffer), bufferLength, FieldType_String);
}

nn::Result Context::AddErrorCode(const nn::err::ErrorCode& errorCode)
NN_NOEXCEPT
{
    char errorString[nn::err::ErrorCode::StringLengthMax];
    nn::err::GetErrorCodeString(errorString, sizeof(errorString), errorCode);
    auto errorStringLength = static_cast<uint32_t>(strnlen(errorString, sizeof(errorString)));
    return Add(nn::erpt::FieldId::ErrorCode, errorString, errorStringLength);
}

nn::Result Context::AddErrorCode(nn::Result result)
NN_NOEXCEPT
{
    return AddErrorCode(nn::err::ConvertResultToErrorCode(result));
}

nn::Result Context::AddThreadName(const os::ThreadType* pThread) NN_NOEXCEPT
{
    static const int CurrentThreadNameLengthMax = 32;

    auto threadName = nn::os::GetThreadNamePointer(pThread);
    auto threadNameLength = (threadName ? static_cast<uint32_t>(strnlen(threadName, CurrentThreadNameLengthMax)) : 0);
    return Add(FieldId::ThreadName, threadName, threadNameLength);
}

nn::Result Context::AddCurrentThreadName() NN_NOEXCEPT
{
    return AddThreadName(os::GetCurrentThread());
}

nn::Result Context::AddErrorContext(const err::ErrorContext& errorContext) NN_NOEXCEPT
{
    switch( errorContext.type )
    {
    case err::ErrorContextType::Http:
        {
            NN_RESULT_DO(Add(erpt::FieldId::ServerFqdn, errorContext.http.fqdn, static_cast<uint32_t>(util::Strnlen(errorContext.http.fqdn, sizeof(errorContext.http.fqdn)))));
            NN_RESULT_DO(Add(erpt::FieldId::ServerIpAddress, errorContext.http.ip, static_cast<uint32_t>(util::Strnlen(errorContext.http.ip, sizeof(errorContext.http.ip)))));
            NN_RESULT_SUCCESS;
        }
        break;
    case err::ErrorContextType::FileSystem:
        {
            NN_RESULT_DO(Add(erpt::FieldId::FileSystemPath, errorContext.fileSystem.path, static_cast<uint32_t>(util::Strnlen(errorContext.fileSystem.path, sizeof(errorContext.fileSystem.path)))));
            if( errorContext.fileSystem.fsResultValue != nn::ResultSuccess::InnerSuccessValue )
            {
                uint32_t resultBacktrace[] = { errorContext.fileSystem.fsResultValue };
                NN_RESULT_DO(Add(erpt::FieldId::ResultBacktrace, resultBacktrace, NN_ARRAY_SIZE(resultBacktrace)));
            }
            NN_RESULT_SUCCESS;
        }
        break;
    case err::ErrorContextType::WebMediaPlayer:
        {
            NN_RESULT_DO(Add(erpt::FieldId::WebMediaPlayerOpenUrl, errorContext.webMediaPlayer.openUrl, static_cast<uint32_t>(util::Strnlen(errorContext.webMediaPlayer.openUrl, sizeof(errorContext.webMediaPlayer.openUrl)))));
            NN_RESULT_DO(Add(erpt::FieldId::WebMediaPlayerLastSocketErrors, errorContext.webMediaPlayer.lastSocketErrors, NN_ARRAY_SIZE(errorContext.webMediaPlayer.lastSocketErrors)));
            NN_RESULT_SUCCESS;
        }
        break;
    case err::ErrorContextType::LocalContentShare:
        {
            NN_RESULT_THROW_UNLESS(errorContext.localContentShare.numKey <= NN_ARRAY_SIZE(errorContext.localContentShare.keyList), ResultInvalidArgument());
            char applicationIdStr[17];
            util::SNPrintf(applicationIdStr, sizeof(applicationIdStr), "%016llx", errorContext.localContentShare.applicationId.value);
            NN_RESULT_DO(Add(erpt::FieldId::LcsApplicationId, applicationIdStr,
                static_cast<uint32_t>(util::Strnlen(applicationIdStr, static_cast<int>(sizeof(applicationIdStr))))));
            uint64_t idList[] = {
                errorContext.localContentShare.keyList[0].id,
                errorContext.localContentShare.keyList[1].id,
            };
            NN_RESULT_DO(Add(erpt::FieldId::LcsContentMetaKeyIdList, idList, static_cast<uint32_t>(errorContext.localContentShare.numKey)));
            uint32_t versionList[] = {
                errorContext.localContentShare.keyList[0].version,
                errorContext.localContentShare.keyList[1].version,
            };
            NN_RESULT_DO(Add(erpt::FieldId::LcsContentMetaKeyVersionList, versionList, static_cast<uint32_t>(errorContext.localContentShare.numKey)));
            uint8_t typeList[] = {
                static_cast<uint8_t>(errorContext.localContentShare.keyList[0].type),
                static_cast<uint8_t>(errorContext.localContentShare.keyList[1].type),
            };
            NN_RESULT_DO(Add(erpt::FieldId::LcsContentMetaKeyTypeList, typeList, static_cast<uint32_t>(errorContext.localContentShare.numKey)));
            uint8_t installTypeList[] = {
                static_cast<uint8_t>(errorContext.localContentShare.keyList[0].installType),
                static_cast<uint8_t>(errorContext.localContentShare.keyList[1].installType),
            };
            NN_RESULT_DO(Add(erpt::FieldId::LcsContentMetaKeyInstallTypeList, installTypeList, static_cast<uint32_t>(errorContext.localContentShare.numKey)));
            NN_RESULT_DO(Add(erpt::FieldId::ResultBacktrace, &errorContext.localContentShare.resultInnerValue, 1u));
            NN_RESULT_DO(Add(erpt::FieldId::LcsIpAddress, errorContext.localContentShare.ip));
            NN_RESULT_DO(Add(erpt::FieldId::LcsSenderFlag, errorContext.localContentShare.isSender));
            NN_RESULT_DO(Add(erpt::FieldId::LcsApplicationRequestFlag, errorContext.localContentShare.isApplicationRequest));
            NN_RESULT_DO(Add(erpt::FieldId::LcsHasExFatDriverFlag, errorContext.localContentShare.hasExFatDriver));
            NN_RESULT_SUCCESS;
        }
        break;
    case err::ErrorContextType::None:
        {
            NN_RESULT_SUCCESS;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::Result Context::ResetContext(CategoryId category)
NN_NOEXCEPT
{
    m_Context.category        = category;
    m_Context.fieldCount      = 0;
    m_Context.arrayFreeCount = m_Context.arrayBufferSize;

    NN_RESULT_SUCCESS;
}

nn::Result Context::SubmitContext()
NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_Context.category != erpt::CategoryId::ErrorInfo);

    nn::sf::SharedPointer<sf::IContext> pContextImpl;

    NN_RESULT_DO(GetContextObject(&pContextImpl));

    nn::sf::InBuffer context(reinterpret_cast<char*>(&m_Context), sizeof(m_Context));
    nn::sf::InBuffer data(reinterpret_cast<char*>(m_Context.pArrayBuffer), m_Context.arrayBufferSize - m_Context.arrayFreeCount);

    NN_RESULT_DO(pContextImpl->SubmitContext(context, data));
    NN_RESULT_SUCCESS;
}

nn::Result Context::CreateReport(ReportType type, ReportMetaData metaData)
NN_NOEXCEPT
{
    nn::sf::SharedPointer<sf::IContext> pContextImpl;

    NN_RESULT_DO(GetContextObject(&pContextImpl));

    nn::sf::InBuffer context(reinterpret_cast<char*>(&m_Context), sizeof(m_Context));
    nn::sf::InBuffer data(reinterpret_cast<char*>(m_Context.pArrayBuffer), m_Context.arrayBufferSize - m_Context.arrayFreeCount);
    nn::sf::InBuffer meta(reinterpret_cast<char*>(&metaData), sizeof(metaData));

    NN_RESULT_DO(pContextImpl->CreateReport(type, context, data, meta));
    NN_RESULT_SUCCESS;
}

nn::Result Context::CreateReport(ReportType type)
NN_NOEXCEPT
{
    nn::sf::SharedPointer<sf::IContext> pContextImpl;

    NN_RESULT_DO(GetContextObject(&pContextImpl));

    nn::sf::InBuffer context(reinterpret_cast<char*>(&m_Context), sizeof(m_Context));
    nn::sf::InBuffer data(reinterpret_cast<char*>(m_Context.pArrayBuffer), m_Context.arrayBufferSize - m_Context.arrayFreeCount);
    nn::sf::InBuffer meta(nullptr, 0);

    NN_RESULT_DO(pContextImpl->CreateReport(type, context, data, meta));
    NN_RESULT_SUCCESS;
}

Context::Context(CategoryId category)
NN_NOEXCEPT
{
    m_Context.version         = ERRVERSION;
    m_Context.fieldCount      = 0;
    m_Context.category        = category;
    m_Context.pArrayBuffer    = m_ArrayBuffer;
    m_Context.arrayBufferSize = sizeof(m_ArrayBuffer);
    m_Context.arrayFreeCount  = sizeof(m_ArrayBuffer);
}

Context::Context(CategoryId category, uint8_t* arrayBuffer, uint32_t arrayBufferSize)
NN_NOEXCEPT
{
    m_Context.version         = ERRVERSION;
    m_Context.fieldCount      = 0;
    m_Context.category        = category;
    m_Context.pArrayBuffer    = arrayBuffer;
    m_Context.arrayBufferSize = arrayBufferSize;
    m_Context.arrayFreeCount  = arrayBufferSize;
}

}}
