﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/erpt.h>
#include <nn/erpt/erpt_MultipleCategoryContext.h>
#include <nn/err/err_SystemApi.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiTelemetry.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include <nnc/result/result_CConverter.h>

#include "eclct_Nifm.h"
#include "eclct_Util.h"

namespace nn    {
namespace eclct {
namespace       {

const char* Combine(char* pBuffer, uint32_t bufferLength, const char* pString1, const char* pString2)
NN_NOEXCEPT
{
    nn::util::SNPrintf(pBuffer, bufferLength, "%s%s", pString1, pString2);
    return pBuffer;
}


const char* ConnectionStatusString(bool isConnectd, nn::nifm::NetworkInterfaceType interfaceType)
NN_NOEXCEPT
{
    if (isConnectd)
    {
        if (interfaceType == nn::nifm::NetworkInterfaceType_Ethernet)
        {
            return "Ethernet";
        }
        else if (interfaceType == nn::nifm::NetworkInterfaceType_Ieee80211)
        {
            return "Wireless";
        }
        else
        {
            return "Invalid";
        }
    }
    else
    {
        return "Not Connected";
    }
}

const char* ErrorCodeString(char* pOutString, int *pOutSize, int size, Result result)
NN_NOEXCEPT
{
    nn::err::GetErrorCodeString(pOutString, size, nn::err::ConvertResultToErrorCode(result));
    *pOutSize = nn::util::Strnlen(pOutString, size);

    NN_SDK_ASSERT(size > *pOutSize);

    return pOutString;
}

void UpdateConnectionStatus(erpt::MultipleCategoryContext& context, bool isConnectionStatusValid, Result result, nn::nifm::NetworkInterfaceType interfaceType)
NN_NOEXCEPT
{
    static const char NotConnected[] = "Not Connected";

    context.MoveToNextCategory(nn::erpt::CategoryId::ConnectionStatusInfo);

    if (result.IsFailure())
    {
        int size;
        char errorCodeString[nn::err::ErrorCode::StringLengthMax];
        ErrorCodeString(errorCodeString, &size, sizeof(errorCodeString), result);
        context.Add(nn::erpt::FieldId::NifmErrorCode, errorCodeString, size);
    }

    if (isConnectionStatusValid)
    {
        char tmp[sizeof(NotConnected)];
        context.Add(
            nn::erpt::FieldId::ConnectionStatus,
            Combine(tmp, sizeof(tmp), ConnectionStatusString(result.IsSuccess(), interfaceType), ""),
            static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
        );
    }
    else
    {
        context.Add(nn::erpt::FieldId::ConnectionStatus, NotConnected, sizeof(NotConnected) - 1);
    }
}

const char* AuthTypeString(nn::nifm::Authentication authentication)
NN_NOEXCEPT
{
    switch (authentication)
    {
    case nn::nifm::Authentication_Open:
        return "Open";
    case nn::nifm::Authentication_Shared:
        return "Shared";
    case nn::nifm::Authentication_Wpa:
        return "Wpa";
    case nn::nifm::Authentication_WpaPsk:
        return "Wpa-Psk";
    case nn::nifm::Authentication_Wpa2:
        return "Wpa2";
    case nn::nifm::Authentication_Wpa2Psk:
        return "Wpa2-Psk";
    case nn::nifm::Authentication_Unknown:
        return "Unknown";
    case nn::nifm::Authentication_Invalid:
        NN_FALL_THROUGH;
    default:
        return "Invalid";
    }
}

const char* EncryptionString(nn::nifm::Encryption encryption)
NN_NOEXCEPT
{
    switch (encryption)
    {
    case nn::nifm::Encryption_None:
        return "(None)";
    case nn::nifm::Encryption_Wep:
        return "(Wep)";
    case nn::nifm::Encryption_Tkip:
        return "(Tkip)";
    case nn::nifm::Encryption_Aes:
        return "(Aes)";
    case nn::nifm::Encryption_Invalid:
        NN_FALL_THROUGH;
    default:
        return "(Invalid)";
    }
}

void UpdateAccessPointInfo(erpt::MultipleCategoryContext& context, bool isAccessPointInfoValid, const nn::nifm::AccessPointData& accessPointData)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::AccessPointInfo);

    if (isAccessPointInfoValid)
    {
        const char* ssid = reinterpret_cast<const char*>(accessPointData.ssid.hex);
        uint32_t ssidLength = accessPointData.ssid.length;
        context.Add(nn::erpt::FieldId::AccessPointSSID, ssid, ssidLength);

        char tmp[sizeof("Wpa2-Psk(Invalid)")];

        context.Add(
            nn::erpt::FieldId::AccessPointSecurityType,
            Combine(tmp, sizeof(tmp), AuthTypeString(accessPointData.authentication),
                EncryptionString(accessPointData.encryption)),
            static_cast<uint32_t>(nn::util::Strnlen(tmp, sizeof(tmp)))
        );

        context.Add(nn::erpt::FieldId::AccessPointChannel, static_cast<uint16_t>(accessPointData.channel));
    }
}

void UpdateRadioStrengthInfo(erpt::MultipleCategoryContext& context, bool isRadioStrengthInfoValid, nn::nifm::LinkLevel linkLevel, int32_t rssi)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::RadioStrengthInfo);

    if (isRadioStrengthInfoValid)
    {
        context.Add(nn::erpt::FieldId::RadioStrength, static_cast<uint32_t>(linkLevel));
        context.Add(nn::erpt::FieldId::AccessPointRssi, rssi);
    }
}

const char* MacAddressString(char* pOutString, int *pOutSize, int size, nn::nifm::MacAddress macAddress)
NN_NOEXCEPT
{
    *pOutSize = nn::util::SNPrintf(pOutString, size, "%02X-%02X-%02X-%02X-%02X-%02X",
        macAddress.data[0], macAddress.data[1], macAddress.data[2],
        macAddress.data[3], macAddress.data[4], macAddress.data[5]);

    NN_SDK_ASSERT(size > *pOutSize);

    return pOutString;
}

void UpdateWirelessAPMacAddressInfo(erpt::MultipleCategoryContext& context, bool isWirelessAPMacAddressInfoValid, const nn::nifm::MacAddress& macAddress)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::WirelessAPMacAddressInfo);

    if (isWirelessAPMacAddressInfoValid)
    {
        char tmp[sizeof("AA-BB-CC-DD-EE-FF")];
        int outSize;

        MacAddressString(tmp, &outSize, sizeof(tmp), macAddress);
        context.Add(nn::erpt::FieldId::WirelessAPMacAddress, tmp, outSize);
    }
}

void UpdateStealthNetworkInfo(erpt::MultipleCategoryContext& context, bool isStealthNetworkInfoValid, bool nonBroadCast)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::StealthNetworkInfo);

    if (isStealthNetworkInfoValid)
    {
        context.Add(nn::erpt::FieldId::UseStealthNetworkFlag, nonBroadCast);
    }
}

void UpdateNXMacAddressInfo(erpt::MultipleCategoryContext& context, bool isNxMacAddressInfoValid, const nn::nifm::MacAddress& macAddress)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::NXMacAddressInfo);

    if (isNxMacAddressInfoValid)
    {
        char tmp[sizeof("AA-BB-CC-DD-EE-FF")];
        int outSize;

        MacAddressString(tmp, &outSize, sizeof(tmp), macAddress);
        context.Add(nn::erpt::FieldId::NXMacAddress, tmp, outSize);
    }
}

void UpdateLANAdapterMacAddressInfo(erpt::MultipleCategoryContext& context, bool isLanAdapterMacAddressInfoValid, const nn::nifm::MacAddress& macAddress)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::LANAdapterMacAddressInfo);

    if (isLanAdapterMacAddressInfoValid)
    {
        char tmp[sizeof("AA-BB-CC-DD-EE-FF")];
        int outSize;

        MacAddressString(tmp, &outSize, sizeof(tmp), macAddress);
        context.Add(nn::erpt::FieldId::LANAdapterMacAddress, tmp, outSize);
    }
}

const char* IpAddressString(char* pOutString, int* pOutSize, int size, nn::nifm::IpV4Address ipAddress)
NN_NOEXCEPT
{
    *pOutSize = nn::util::SNPrintf(pOutString, size, "%d.%d.%d.%d",
        ipAddress.data[0], ipAddress.data[1], ipAddress.data[2], ipAddress.data[3]);

    NN_SDK_ASSERT(size > *pOutSize);

    return pOutString;
}

void UpdateNetworkInfo(erpt::MultipleCategoryContext& context, bool isNetworkInfoValid, bool isIpAddressSettingValid, bool isDnsSettingValid,
    const nn::nifm::NetworkProfileData& networkProfileData,
    const nn::nifm::IpAddressSetting& ipAddressSetting, const nn::nifm::DnsSetting& dnsSetting)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::NetworkInfo);

    if (isNetworkInfoValid)
    {
        context.Add(nn::erpt::FieldId::IPAddressAcquisitionMethod, networkProfileData.ipSetting.ip.isAuto ? 1U : 0U);

        context.Add(nn::erpt::FieldId::DNSType, networkProfileData.ipSetting.dns.isAuto ? 1U : 0U);

        context.Add(nn::erpt::FieldId::ConnectAutomaticallyFlag, networkProfileData.isAutoConnect);

        context.Add(nn::erpt::FieldId::MTU, static_cast<uint32_t>(networkProfileData.ipSetting.mtu));

        bool isProxyUsed = networkProfileData.ipSetting.proxy.isEnabled;
        context.Add(nn::erpt::FieldId::UseProxyFlag, isProxyUsed);

        if (isProxyUsed)
        {
            context.Add(nn::erpt::FieldId::ProxyIPAddress, networkProfileData.ipSetting.proxy.proxy,
                nn::util::Strnlen(networkProfileData.ipSetting.proxy.proxy, nn::nifm::ProxySetting::ProxyNameSize));

            context.Add(nn::erpt::FieldId::ProxyPort, static_cast<uint32_t>(networkProfileData.ipSetting.proxy.port));

            context.Add(nn::erpt::FieldId::ProxyAutoAuthenticateFlag, networkProfileData.ipSetting.proxy.authentication.isEnabled);
        }

        if (isIpAddressSettingValid)
        {
            char tmp[sizeof("012.345.678.901")];
            int outSize;

            IpAddressString(tmp, &outSize, sizeof(tmp), ipAddressSetting.ipAddress);
            context.Add(nn::erpt::FieldId::CurrentIPAddress, tmp, outSize);

            IpAddressString(tmp, &outSize, sizeof(tmp), ipAddressSetting.subnetMask);
            context.Add(nn::erpt::FieldId::SubnetMask, tmp, outSize);

            IpAddressString(tmp, &outSize, sizeof(tmp), ipAddressSetting.defaultGateway);
            context.Add(nn::erpt::FieldId::GatewayIPAddress, tmp, outSize);
        }

        if (isDnsSettingValid)
        {
            char tmp[sizeof("012.345.678.901")];
            int outSize;

            IpAddressString(tmp, &outSize, sizeof(tmp), dnsSetting.preferredDns);
            context.Add(nn::erpt::FieldId::PriorityDNSIPAddress, tmp, outSize);

            IpAddressString(tmp, &outSize, sizeof(tmp), dnsSetting.alternateDns),
            context.Add(nn::erpt::FieldId::AlternateDNSIPAddress, tmp, outSize);
        }
    }
}

void UpdateLimitHighCapacityInfo(erpt::MultipleCategoryContext& context, bool limitHighCapacityInfoFlag, bool isLargeCapacity)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::LimitHighCapacityInfo);

    if (limitHighCapacityInfoFlag)
    {
        context.Add(nn::erpt::FieldId::LimitHighCapacityFlag, !isLargeCapacity);
    }
}

void UpdateNintendoZoneSSIDListVersionInfo(erpt::MultipleCategoryContext& context, bool nintendoZoneSSIDListVersionInfoFlag, const nn::nifm::SsidListVersion& ssidListVersion)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::NintendoZoneSSIDListVersionInfo);

    if (nintendoZoneSSIDListVersionInfoFlag)
    {
        context.Add(nn::erpt::FieldId::NintendoZoneSSIDListVersion,
            ssidListVersion.data, nn::util::Strnlen(ssidListVersion.data, nn::nifm::SsidListVersion::Size));
    }
}

void UpdateNintendoZoneConnectedInfo(erpt::MultipleCategoryContext& context, bool nintendoZoneConnectedInfoFlag, nn::nifm::NetworkProfileType networkProfileType)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::NintendoZoneConnectedInfo);

    if (nintendoZoneConnectedInfoFlag)
    {
        context.Add(nn::erpt::FieldId::NintendoZoneConnectedFlag,
            networkProfileType == nn::nifm::NetworkProfileType_SsidList);
    }
}

void UpdateNifmConnectionTestInfo(erpt::MultipleCategoryContext& context, bool additionalInfoFlag, nn::nifm::AdditionalInfo additionalInfo)
NN_NOEXCEPT
{
    context.MoveToNextCategory(nn::erpt::CategoryId::NifmConnectionTestInfo);

    if (additionalInfoFlag)
    {
        auto urllen = nn::util::Strnlen(additionalInfo.redirectUrl, nn::nifm::AdditionalInfo::RedirectUrlSize);
        if (urllen > 0)
        {
            context.Add(nn::erpt::FieldId::NifmConnectionTestRedirectUrl,
                additionalInfo.redirectUrl, urllen);
        }
    }
}

} // ~nn::eclct::<anonymous>

void UpdateNifmInfo()
NN_NOEXCEPT
{
    static uint8_t buffer[1024 + 256]; // erpt::FieldId::NifmConnectionTestRedirectUrl の分 + その他の文字列・配列フィールドの分（IP アドレスなど）
    nn::erpt::MultipleCategoryContext context(buffer, sizeof(buffer));

    nn::nifm::TelemetryInfo telemetryInfo;

    nn::nifm::GetTelemetoryInfo(&telemetryInfo);

    UpdateConnectionStatus(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::ConnectionStatusInfo>(),
        nn::result::ConvertFromC(telemetryInfo.result), telemetryInfo.networkProfileData.networkInterfaceType);

    UpdateAccessPointInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::AccessPointInfo>(),
        telemetryInfo.accessPointData);

    UpdateRadioStrengthInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::RadioStrengthInfo>(),
        telemetryInfo.accessPointData.linkLevel, telemetryInfo.accessPointData.rssi);

    UpdateWirelessAPMacAddressInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::WirelessAPMacAddressInfo>(),
        telemetryInfo.accessPointData.bssid);

    UpdateStealthNetworkInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::StealthNetworkInfo>(),
        telemetryInfo.networkProfileData.wirelessSetting.ssidConfig.nonBroadcast);

    UpdateNXMacAddressInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::NXMacAddressInfo>(),
        telemetryInfo.wirelessMacAddress);

    UpdateLANAdapterMacAddressInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::LANAdapterMacAddressInfo>(),
        telemetryInfo.ethernetMacAddress);

    UpdateNetworkInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::NetworkInfo>(),
        telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::NetworkInfoIpAddressSetting>(),
        telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::NetworkInfoDnsSetting>(),
        telemetryInfo.networkProfileData, telemetryInfo.ipAddressSetting, telemetryInfo.dnsSetting);

    UpdateLimitHighCapacityInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::LimitHighCapacityInfo>(),
        telemetryInfo.networkProfileData.isLargeCapacity);

    UpdateNintendoZoneSSIDListVersionInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::NintendoZoneSSIDListVersionInfo>(),
        telemetryInfo.ssidListVersion);

    UpdateNintendoZoneConnectedInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::NintendoZoneConnectedInfo>(),
        telemetryInfo.networkProfileData.networkProfileType);

    UpdateNifmConnectionTestInfo(context, telemetryInfo.flagSet.Test<nn::nifm::TelemetryFlag::ConnectionTestInfo>(),
        telemetryInfo.additionalInfo);

    NN_DETAIL_ECLCT_WARN_UNLESS_RESULT_SUCCESS(context.SubmitContext());
}

}}
