﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief クライアントから消費型サービスアイテムを利用するためのAPIの定義
 */

#pragma once
#include <nn/nn_Result.h>
#include <nn/account/account_Types.h>
#include <nn/ec/ec_ConsumableServiceItemTypes.h>
#include <nn/ec/ec_ShopServiceAccessor.h>
#include <nn/time/time_PosixTime.h>

namespace nn { namespace ec {

/**
    @brief サーバーに所有している未消費の権利情報を取得します。

    @param[out] pAsyncResponse      非同期レスポンスの格納先
    @param[in]  user                権利情報を取得するユーザ
    @param[in]  page                ページインデックス
    @param[in]  perPage             1 ページあたりの取得件数
    @param[in]  pAccessor           消費型サーバーに対するアクセッサ
    @param[in]  pWorkMemory         ワークメモリ
    @param[in]  workMemorySize      ワークメモリのサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::ec::ResultInsufficientWorkMemory,
            ワークメモリが足りません。pWorkMemory のメモリサイズを RequiredWorkMemorySizeForConsumableServiceItem 以上にしてください
        }
    @endretresult

    @pre
        - pAsyncResponse != nullptr
        - static_cast<bool>(user) == true
        - page >= 0
        - perPage > 0
        - perPage <= ConsumableServiceItemRightDataCountMax
        - pAccessor != nullptr
        - pWorkMemory != nullptr
        - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

    @details
        非同期処理をリクエストして、ec::ShopServiceAccessor::AsyncResponse を返します。
        処理が完了したかの確認、処理のキャンセルは ec::ShopServiceAccessor::AsyncResponse で行ってください。
        処理完了後は EndGetConsumableRightData で内容を取得してください。
        処理が成功するためには、nifm でネットワーク要求が出されていて、ShopServiceAccessor が初期化されている必要があります。

        user で指定したユーザが、サーバーに所有している権利情報を取得します。権利情報の一覧は、購入時刻の降順で表示されます。
        本 API では、1 ページあたりの取得する権利数と、取得するページインデックスを指定できます。

        本 API では、サーバーへの要求を行います。
        サーバーへの要求を伴う関数の呼び出しには、頻度制限があります。
        制限を超えた場合でも本関数はすぐに返りますが、要求処理の開始は遅延されます。
*/
Result BeginGetConsumableRightData(ShopServiceAccessor::AsyncResponse* pAsyncResponse, const nn::account::Uid& user, int page, int perPage, ShopServiceAccessor *pAccessor, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

/**
    @brief 権利情報の取得完了を待ちます

    @param[out] pOutRightDataCount  取得した権利情報の数
    @param[out] outRightDataArray   取得した権利情報の格納先
    @param[in]  maxRightDataCount   取得する権利情報の最大数
    @param[in]  pAsyncResponse      完了を待つ非同期レスポンス
    @param[in]  pWorkMemory         ワークメモリ
    @param[in]  workMemorySize      ワークメモリのサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::ec::ResultCancel,
            サーバーとの通信がキャンセルされました。
        }
        @handleresult{
            nn::ec::ResultInsufficientWorkMemory,
            ワークメモリが足りません。pWorkMemory のメモリサイズを RequiredWorkMemorySizeForConsumableServiceItem 以上にしてください
        }
    @endretresult

    @pre
        - pOutRightDataCount != nullptr
        - outRightDataArray != nullptr
        - maxRightDataCount > 0
        - maxRightDataCount <= ConsumableServiceItemRightDataCountMax
        - pAsyncResponse != nullptr
        - pWorkMemory != nullptr
        - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

    @details
        pAsyncResponse で指定した非同期処理が完了していなければ、API 内部で完了を待機します。

        レスポンスに含まれている権利数に関わらず、一度に maxRightDataCount 個まで権利情報を取得できます。
        全ての権利情報を取得する場合、BeginGetConsumableRightData 実行時に指定した perPage 分以上のバッファを用意してください。
*/
Result EndGetConsumableRightData(int* pOutRightDataCount, ConsumableServiceItemRightData outRightDataArray[], int maxRightDataCount, ShopServiceAccessor::AsyncResponse* pAsyncResponse, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

/**
    @brief 提供可能な権利情報をサーバーに問い合わせます。

    @param[out] pAsyncResponse      非同期レスポンスの格納先
    @param[in]  user                権利情報を取得するユーザ
    @param[in]  rightDataArray      提供可否判定する権利情報
    @param[in]  rightDataCount      提供可否判定する権利情報の数
    @param[in]  pAccessor           消費型サーバーに対するアクセッサ
    @param[in]  pWorkMemory         ワークメモリ
    @param[in]  workMemorySize      ワークメモリのサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::ec::ResultInsufficientWorkMemory,
            ワークメモリが足りません。pWorkMemory のメモリサイズを RequiredWorkMemorySizeForConsumableServiceItem 以上にしてください
        }
    @endretresult

    @pre
        - pAsyncResponse != nullptr
        - static_cast<bool>(user) == true
        - rightDataArray != nullptr
        - rightDataArray[i].IsValid == true
        - rightDataCount > 0
        - rightDataCount <= ConsumableServiceItemRightDataCountMax
        - pAccessor != nullptr
        - pWorkMemory != nullptr
        - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

    @details
        非同期処理をリクエストして、ec::ShopServiceAccessor::AsyncResponse を返します。
        処理が完了したかの確認、処理のキャンセルは ec::ShopServiceAccessor::AsyncResponse で行ってください。
        処理完了後は EndProvidableRightDataInquiry で内容を取得してください。
        処理が成功するためには、nifm でネットワーク要求が出されていて、ShopServiceAccessor が初期化されている必要があります。

        本 API では、サーバーへの要求を行います。
        サーバーへの要求を伴う関数の呼び出しには、頻度制限があります。
        制限を超えた場合でも本関数はすぐに返りますが、要求処理の開始は遅延されます。
*/
Result BeginProvidableRightDataInquiry(ShopServiceAccessor::AsyncResponse* pAsyncResponse, const nn::account::Uid& user, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor *pAccessor, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

/**
    @brief 提供可能な権利情報の取得完了を待ちます

    @param[out] pOutProvidableRightDataCount  提供可能な権利情報の数
    @param[out] outProvidableRightDataArray   提供可能な権利情報の格納先
    @param[in]  maxRightDataCount             取得する権利情報の最大数
    @param[in]  rightDataArray                提供可否判定する権利情報
    @param[in]  rightDataCount                提供可否判定する権利情報の数
    @param[in]  pAsyncResponse                完了を待つ非同期レスポンス
    @param[in]  pWorkMemory                   ワークメモリ
    @param[in]  workMemorySize                ワークメモリのサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::ec::ResultCancel,
            サーバーとの通信がキャンセルされました。
        }
        @handleresult{
            nn::ec::ResultInsufficientWorkMemory,
            ワークメモリが足りません。pWorkMemory のメモリサイズを RequiredWorkMemorySizeForConsumableServiceItem 以上にしてください
        }
    @endretresult

    @pre
        - pOutProvidableRightDataCount != nullptr
        - outProvidableRightDataArray != nullptr
        - maxRightDataCount > 0
        - rightDataArray != nullptr
        - rightDataArray[i].IsValid == true
        - rightDataCount > 0
        - rightDataCount <= ConsumableServiceItemRightDataCountMax
        - pAsyncResponse != nullptr
        - pWorkMemory != nullptr
        - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

    @details
        pAsyncResponse で指定した非同期処理が完了していなければ、API 内部で完了を待機します。

        レスポンスに含まれている権利数に関わらず、一度に maxRightDataCount 個まで権利情報を取得できます。
        全ての権利情報を取得する場合、BeginProvidableRightDataInquiry 実行時に指定した rightDataCount 分以上のバッファを用意してください。

        提供可能と判断された権利は、以後、IsServiceProvidableOnLocal() == true を満たします。
*/
Result EndProvidableRightDataInquiry(int* pOutProvidableRightDataCount, ConsumableServiceItemRightData outProvidableRightDataArray[], int maxRightDataCount, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor::AsyncResponse* pAsyncResponse, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

/**
    @brief サーバーに所有している権利情報を ConsumableServiceItemRightData 指定で消費します。

    @param[out] pAsyncResponse      非同期レスポンスの格納先
    @param[in]  user                権利情報を取得するユーザ
    @param[in]  rightDataArray      消費する権利情報
    @param[in]  rightDataCount      消費する権利情報の数
    @param[in]  pAccessor           消費型サーバーに対するアクセッサ
    @param[in]  pWorkMemory         ワークメモリ
    @param[in]  workMemorySize      ワークメモリのサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::ec::ResultRequestCountLimit,
            1 度に消費する権利数が許容される数を超過しました。
            1 度に消費する権利数を ec::ConsumableServiceItemRightDataCountMax 以下にしてください。
        }
        @handleresult{
            nn::ec::ResultInsufficientWorkMemory,
            ワークメモリが足りません。pWorkMemory のメモリサイズを RequiredWorkMemorySizeForConsumableServiceItem 以上にしてください
        }
    @endretresult

    @pre
        - pAsyncResponse != nullptr
        - static_cast<bool>(user) == true
        - rightDataArray != nullptr
        - rightDataArray[i].IsValid == true
        - rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax
        - pAccessor != nullptr
        - pWorkMemory != nullptr
        - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

    @details
        非同期処理をリクエストして、ec::ShopServiceAccessor::AsyncResponse を返します。
        処理が完了したかの確認、処理のキャンセルは ec::ShopServiceAccessor::AsyncResponse で行ってください。
        処理完了後は EndConsumeRightData で内容を取得してください。
        処理が成功するためには、nifm でネットワーク要求が出されていて、ShopServiceAccessor が初期化されている必要があります。

        user で指定したユーザが、サーバーに所有している権利情報を消費します。
        rightDataArray に入力した権利情報のうち、IsServiceProvidableOnLocal() == true を満たすものは API 内で権利消費がスキップされます。
        消費した権利は以後、BeginGetConsumableRightData() では参照できなくなります。

        本 API では、サーバーへの要求を行います。
        サーバーへの要求を伴う関数の呼び出しには、頻度制限があります。
        制限を超えた場合でも本関数はすぐに返りますが、要求処理の開始は遅延されます。
*/
Result BeginConsumeRightData(ShopServiceAccessor::AsyncResponse* pAsyncResponse, const nn::account::Uid& user, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor *pAccessor, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

/**
    @brief 権利情報の消費完了を待ちます

    @param[out] pOutConsumedRightDataCount  消費できた権利情報の数
    @param[out] outConsumedRightDataArray   消費できた権利情報の格納先
    @param[in]  maxRightDataCount           取得する権利情報の最大数
    @param[in]  rightDataArray              消費する権利情報
    @param[in]  rightDataCount              消費する権利情報の数
    @param[in]  pAsyncResponse              完了を待つ非同期レスポンス
    @param[in]  pWorkMemory                 ワークメモリ
    @param[in]  workMemorySize              ワークメモリのサイズ

    @retresult
        @handleresult{
            nn::ResultSuccess,
            処理に成功しました。
        }
        @handleresult{
            nn::ec::ResultCancel,
            サーバーとの通信がキャンセルされました。
        }
        @handleresult{
            nn::ec::ResultInsufficientWorkMemory,
            ワークメモリが足りません。pWorkMemory のメモリサイズを RequiredWorkMemorySizeForConsumableServiceItem 以上にしてください
        }
    @endretresult

    @pre
        - pOutConsumedRightDataCount != nullptr
        - outConsumedRightDataArray != nullptr
        - maxRightDataCount > 0
        - rightDataArray != nullptr
        - rightDataArray[i].IsValid == true
        - rightDataCount > 0 && rightDataCount <= ConsumableServiceItemRightDataCountMax
        - pAsyncResponse != nullptr
        - pWorkMemory != nullptr
        - workMemorySize >= RequiredWorkMemorySizeForConsumableServiceItem

    @details
        pAsyncResponse で指定した非同期処理が完了していなければ、API 内部で完了を待機します。

        消費できた権利は、以後、IsServiceProvidableOnLocal() == true を満たします。
*/
Result EndConsumeRightData(int* pOutConsumedRightDataCount, ConsumableServiceItemRightData outConsumedRightDataArray[], int maxRightDataCount, const ConsumableServiceItemRightData rightDataArray[], int rightDataCount, ShopServiceAccessor::AsyncResponse* pAsyncResponse, void* pWorkMemory, size_t workMemorySize) NN_NOEXCEPT;

}}
