﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ec/detail/ec_ShowShopPageApi.h>
#include <nn/ec/detail/ec_TypesInternal.h>
#include <nn/ec/detail/ec_Utils.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_StringView.h>

#include <cstdlib>
#include <cstdarg>

#if defined( NN_BUILD_CONFIG_TOOLCHAIN_GCC )
#define STRTOULL strtoull
#elif defined( NN_BUILD_CONFIG_OS_WIN )
#define STRTOULL _strtoui64
#else
#define STRTOULL std::strtoull
#endif

namespace nn { namespace ec { namespace detail {

namespace
{
    static const size_t MaxPathLength = 3071;
    static const size_t MaxBasePathLength = 2048;
    static const size_t MaxParamsTailLength = 2048;

    struct FixedStringData
    {
        char* buffer;
        size_t remainLength;
    };

    // key=value 形式のデータとして用いることができる文字かどうかを返します。
    NN_FORCEINLINE bool IsValidCharForKeyValue(char ch) NN_NOEXCEPT
    {
        // 参考: RFC 3986:
        //   query         = *( pchar / "/" / "?" )
        //   pchar         = unreserved / pct-encoded / sub-delims / ":" / "@"
        //   unreserved    = ALPHA / DIGIT / "-" / "." / "_" / "~"
        //   pct-encoded   = "%" HEXDIG HEXDIG
        //   sub-delims    = "!" / "$" / "&" / "'" / "(" / ")"
        //                 / "*" / "+" / "," / ";" / "="
        // ※ RFC 2234:
        //   ALPHA         = %x41-5A / %x61-7A   ; A-Z / a-z
        //   DIGIT         = %x30-39             ; 0-9
        //   HEXDIG        =  DIGIT / "A" / "B" / "C" / "D" / "E" / "F"
        return
            // unreserved
            (ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') || (ch >= '0' && ch <= '9') ||
            ch == '.' || ch == '-' || ch == '_' || ch == '~' ||
            // pct-encoded
            ch == '%'; // 「"%" HEXDIG HEXDIG」という書式までは簡単のためチェックしない
            // sub-delims
            // → key, value ではまず用いられないので指定不可とする
            // pchar
            // → 「:」と「@」は key, value ではまず用いられないので指定不可とする
            // query
            // → 「/」と「?」も key, value ではまず用いられないので指定不可とする
    }

    // QueryStringとして用いることができる文字かどうかを返します。
    NN_FORCEINLINE bool IsValidCharForQueryString(char ch) NN_NOEXCEPT
    {
        return IsValidCharForKeyValue(ch) ||
            // sub-delims
            ch == '!' || ch == '$' || ch == '\'' || ch == '(' || ch == ')' ||
            ch == '*' || ch == '+' || ch == ',' || ch == ';' || ch == '&' || ch == '=' ||
            // pchar
            ch == ':' || ch == '@' ||
            // query
            ch == '/' || ch == '?';
    }

    // Pathとして用いることができる文字かどうかを返します。
    // ※ QueryStringが含まれても可
    NN_FORCEINLINE bool IsValidCharForPath(char ch) NN_NOEXCEPT
    {
        // 参考: RFC 3986: (ここで扱うのは path-abempty のみ)
        //   path-abempty  = *( "/" segment )
        //   segment       = *pchar
        return IsValidCharForQueryString(ch);
    }

    // return: 残りサイズが 0 より大きければ true
    inline bool AppendChar(FixedStringData& outString, char ch) NN_NOEXCEPT
    {
        if (outString.remainLength == 0)
        {
            return false;
        }
        *outString.buffer++ = ch;
        --outString.remainLength;
        return outString.remainLength > 0;
    }

    // コピー不可能な文字が指定されたらそこでコピーをストップするテンプレート関数
    // return: 1文字以上書き込まれ、かつ残りサイズが 0 より大きければ true (0 になってもぎりぎりまではコピーを行う)
    template <bool (* IsValidCharFunction)(char ch)>
    inline bool AppendStringWithValidation(FixedStringData& outString, const char* stringToAdd, size_t length) NN_NOEXCEPT
    {
        if (outString.remainLength < length)
        {
            length = outString.remainLength;
        }
        size_t writtenLength = 0;
        while (writtenLength < length)
        {
            if (!(*IsValidCharFunction)(*stringToAdd))
            {
                break;
            }
            *outString.buffer = *stringToAdd;
            ++outString.buffer;
            --outString.remainLength;
            ++stringToAdd;
            ++writtenLength;
        }
        return writtenLength > 0 && outString.remainLength > 0;
    }

    bool AppendKeyValueToBuffer(FixedStringData& outString, const char* key, const char* value, size_t valueLen) NN_NOEXCEPT
    {
        // key
        {
            size_t keyLen = std::strlen(key);
            if (keyLen == 0 || !AppendStringWithValidation<IsValidCharForKeyValue>(outString, key, keyLen))
            {
                return false;
            }
        }
        if (!AppendChar(outString, '='))
        {
            return false;
        }
        // value
        {
            return AppendStringWithValidation<IsValidCharForKeyValue>(outString, value, valueLen);
        }
    }

    inline bool AppendKeyValueToBuffer(FixedStringData& outPath, const char* key, const char* value) NN_NOEXCEPT
    {
        return AppendKeyValueToBuffer(outPath, key, value, std::strlen(value));
    }
}

size_t NsUidToString(char* outBuffer, size_t bufferLength, nn::ec::NsUid id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES(bufferLength > 0);

    if (bufferLength == 0)
    {
        return 0;
    }

    // NsUidの文字列表現は符号なし10進数なので
    // 10進数を文字列に変換する方法で生成する
    size_t result = static_cast<size_t>(nn::util::SNPrintf(outBuffer, bufferLength, "%llu", static_cast<uint64_t>(id.value)));
    if (result >= bufferLength)
    {
        result = bufferLength - 1;
    }
    outBuffer[result] = 0;
    return result;
}

size_t GetNsUidLengthAsString(nn::ec::NsUid id) NN_NOEXCEPT
{
    // 10進数の桁数を返す
    uint64_t value = id.value;
    size_t result = 0;
    while (true)
    {
        value /= 10;
        ++result;
        if (value == 0)
        {
            break;
        }
    }
    return result;
}

size_t AppIdToString(char* outBuffer, size_t bufferLength, nn::ApplicationId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    // 16桁固定にするため17文字以上のバッファーが必要
    NN_SDK_REQUIRES(bufferLength >= 17);

    if (bufferLength < 17)
    {
        return 0;
    }

    // 16進数を16桁の文字列に変換する方法で生成する
    size_t result = static_cast<size_t>(nn::util::SNPrintf(outBuffer, bufferLength, "%016llX", static_cast<uint64_t>(id.value)));
    if (result >= bufferLength)
    {
        result = bufferLength - 1;
    }
    outBuffer[result] = 0;
    return result;
}

size_t CouponIdToString(char* outBuffer, size_t bufferLength, CouponId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES(bufferLength > 0);

    if (bufferLength == 0)
    {
        return 0;
    }

    // CouponIdの文字列表現は符号なし10進数なので
    // 10進数を文字列に変換する方法で生成する
    size_t result = static_cast<size_t>(nn::util::SNPrintf(outBuffer, bufferLength, "%llu", static_cast<uint64_t>(id.value)));
    if (result >= bufferLength)
    {
        result = bufferLength - 1;
    }
    outBuffer[result] = 0;
    return result;
}

void GeneratePath(ShowShopPageArg& shopPageArg, const char* basePath, SourceId sourceId, const char* paramsTail,
    const char* name, const char* value, ...) NN_NOEXCEPT
{
    std::va_list arg;
    size_t basePathLength = static_cast<size_t>(nn::util::Strnlen(basePath, MaxBasePathLength + 1));
    if (basePathLength > MaxBasePathLength)
    {
        basePathLength = MaxBasePathLength;
    }

    size_t paramsTailLength = (paramsTail == nullptr ? 0 :
        static_cast<size_t>(nn::util::Strnlen(paramsTail, MaxParamsTailLength + 1)));
    // 2048文字以下に制限する
    if (paramsTailLength > MaxParamsTailLength)
    {
        paramsTailLength = MaxParamsTailLength;
    }

    const bool hasSourceId = (sourceId != SourceId_Default);
    const bool hasAnyParameters = (name != nullptr);

    bool hasQSMark;
    bool isLastQSDelimiter;
    if (basePathLength > 0)
    {
        nn::util::string_view basePathView(basePath, basePathLength);
        hasQSMark = (basePathView.find('?', basePathLength) < basePathLength);
        isLastQSDelimiter = (basePathView.back() == '&' || basePathView.back() == '?');
    }
    else
    {
        // 文字列がないのでマークもない
        hasQSMark = false;
        isLastQSDelimiter = false;
    }

    NN_STATIC_ASSERT(MaxPathLength < ShowShopPageArg::RequestRelativeUrlMaxLength);
    char* bufferBase = reinterpret_cast<char*>(shopPageArg.GetRequestRelativeUrlBuffer());
    FixedStringData data = { bufferBase, MaxPathLength };
    data.buffer[MaxPathLength] = 0;

    AppendStringWithValidation<IsValidCharForPath>(data, basePath, basePathLength);

    // 何かしら引数が追加されるかどうかのチェック
    if (hasSourceId || hasAnyParameters || paramsTailLength > 0)
    {
        // 末尾が '?' か '&' であれば何もせず、それ以外の位置に '?' が含まれていたら末尾に '&'、
        // 含まれていなければ '?' を付加する
        if (!isLastQSDelimiter)
        {
            if (hasQSMark)
            {
                AppendChar(data, '&');
            }
            else
            {
                AppendChar(data, '?');
            }
        }
    }

    if (hasSourceId && data.remainLength > 0)
    {
        AppendKeyValueToBuffer(data, "from_scene",
            sourceId.data, nn::util::Strnlen(sourceId.data, sizeof(sourceId.data) / sizeof(sourceId.data[0])));
    }
    if (hasAnyParameters && data.remainLength > 0)
    {
        if (hasSourceId)
        {
            // 既に sourceId が追加されているので '&' を付加する
            AppendChar(data, '&'); // 少なくとも1文字は追加可能
        }

        AppendKeyValueToBuffer(data, name, value);
        // add more parameters
        {
            va_start(arg, value);
            while (data.remainLength > 0)
            {
                // key
                const char* nameMore = va_arg(arg, const char*);
                if (nameMore == nullptr)
                {
                    break;
                }
                const char* valueMore;
                size_t valueLength;
                AppendChar(data, '&'); // 少なくとも1文字は追加可能
                if (NN_DETAIL_EC_PICK_UP_VALUE_AND_LENGTH(&valueMore, &valueLength, arg))
                {
                    AppendKeyValueToBuffer(data, nameMore, valueMore, valueLength);
                }
                else
                {
                    AppendKeyValueToBuffer(data, nameMore, valueMore);
                }
            }
            va_end(arg);
        }
    }

    // append paramsTail
    {
        if (data.buffer != bufferBase && data.remainLength > 0 && paramsTailLength > 0)
        {
            if ((hasSourceId || hasAnyParameters) && *paramsTail != '&')
            {
                // 既に別の引数が追加されているので '&' を付加する
                AppendChar(data, '&');
            }
            AppendStringWithValidation<IsValidCharForQueryString>(data, paramsTail, paramsTailLength);
        }
    }
    if (data.remainLength > 0)
    {
        data.buffer[0] = 0;
    }
}

bool IsStringTerminated(char* str, size_t length) NN_NOEXCEPT
{
    for (int i = 0; i < length; i++)
    {
        if (str[i] == '\0')
        {
            return true;
        }
    }

    return false;
}

}}} // nn::ec::detail
