﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"
#include "dhcps_Constants.h"

/**
 * @file
 *
 * @brief This file is the header file for a basic network packet
 * parser.
 */

namespace nn { namespace dhcps { namespace detail {

class Coordinator;

/**
 * @brief Constants that are used to identify the protocol type for
 * a @a NetworkLayersContainer.
 */
enum class NetworkLayer : uint32_t
{
    BpfHeader     = 0, ///< 0
    Ethernet      = 1, ///< 1
    Internet      = 2, ///< 2
    UserDatagram  = 3, ///< 3
    BootProtocol  = 4, ///< 4
    Icmp          = 5, ///< 5
    Max           = 6, ///< 6
};

/**
 * @brief Returns a string representation of the
 * @ref NetworkLayersContainer value.
 *
 * @param[in] in The NetworkLayer value.
 *
 * @return A string representation for the @ref NetworkLayer value.
 */
const char* NetworkLayerToString(NetworkLayer in) NN_NOEXCEPT;

/**
 * @brief A structure that represents a single layer of a network
 * packet.
 *
 * @details Packets are represented as an array of
 * NetworkLayersContainer and are supported by a single contiguous
 * buffer.
 */
struct NetworkLayersContainer
{
    /**
     * @brief The @a NetworkLayer type parameter that specifies the
     * protocol this layer contains.
     */
    NetworkLayer type;

    /**
     * @brief The buffer that contains the layer bytes.
     */
    uint8_t* pBuffer;

    /**
     * @brief The size of the layer bytes.
     */
    size_t size;
};

/**
 * @brief Gets a copy of the provided layer from a given packet array.
 *
 * @param[out] pOutContainer On success this parameter contains a
 * copy of the @ref NetworkLayersContainer for the provided layer.
 *
 * @param[in] layer The provided @ref NetworkLayer.
 *
 * @param[in] pPacket An array of @ref NetworkLayersContainer that
 * represent a network packet.
 *
 * @param[in] count The number of @ref NetworkLayersContainer contained
 * in the @a pPacket array.
 *
 * @return Return 0 on success and -1 on error.
 */
int NetworkLayersGetLayer(NetworkLayersContainer* pOutContainer,
                          NetworkLayer protocol,
                          const NetworkLayersContainer pPacket[],
                          size_t count) NN_NOEXCEPT;

/**
 * @brief Converts layers contained in the packet array from network
 * to host byte order.
 *
 * @param[in,out] pOutPacket An array of @ref NetworkLayersContainer that
 * represent a network packet.
 *
 * @param[in] count The number of @ref NetworkLayersContainer contained
 * in the @a pPacket array.
 */
void NetworkLayersNtoh(NetworkLayersContainer pOutPacket[],
                       size_t count) NN_NOEXCEPT;

/**
 * @brief Converts layers contained in the packet array from host to
 * network byte order
 *
 * @param[in,out] pOutPacket An array of @ref NetworkLayersContainer that
 * represent a network packet.
 *
 * @param[in] count The number of @ref NetworkLayersContainer contained
 * in the @a pPacket array.
 */
void NetworkLayersHton(NetworkLayersContainer pOutPacket[],
                       size_t count) NN_NOEXCEPT;

/**
 * @brief Flips the source and destination fields each layer contained
 * in the packet array.
 *
 * @param[in,out] pPacket An array of @ref NetworkLayersContainer that
 * represent a network packet.
 *
 * @param[in] count The number of @ref NetworkLayersContainer contained
 * in the @a pPacket array.
 */
void NetworkLayersFlip(NetworkLayersContainer pOutPacket[],
                       size_t count) NN_NOEXCEPT;

/**
 * @brief This function calculates the checksum for the network
 * layers contained in in the stack
 *
 * @param[in,out] pPacket An array of @ref NetworkLayersContainer that
 * represent a network packet.
 *
 * @param[in] count The number of @ref NetworkLayersContainer contained
 * in the @a pPacket array.
 */
void NetworkLayersChecksum(struct NetworkLayersContainer pOutPacket[],
                           size_t count) NN_NOEXCEPT;

/**
 * @brief Copies a source packet array to a destination packet array
 * given certain conditions.
 *
 * @details The conditions for success are:
 * - The number of elements in the @a pOutPacket array must be
 * identical.
 * - The buffer size must meet or exceed the size of the buffer used
 * by @a pPacketSrc.
 *
 * @param[in,out] pOutPacket The destination array of
 * @ref NetworkLayersContainer that represent a network packet.
 *
 * @param[in] packetCount The number of @NetworkLayersContainer
 * contained in the @a pOutPacket array.
 *
 * @param[out] pOutBuffer The buffer that supports the @a pOutPacket array.
 *
 * @param[in] bufferSize The size of @a pOutBuffer in bytes.
 *
 * @param[in,out] pPacketSrc The source array of
 * @ref NetworkLayersContainer that represent a network packet.
 *
 * @param[in] packetSrcCount The number of @NetworkLayersContainer
 * contained in the @a pPacketSrc array.
 *
 * @return On success return the number of bytes copied. On error
 * return -1.
 */
size_t NetworkLayersCopy(NetworkLayersContainer pOutPacket[],
                         size_t packetCount,
                         uint8_t* pOutBuffer,
                         size_t bufferSize,
                         const NetworkLayersContainer pPacketSrc[],
                         size_t packetSrcCount) NN_NOEXCEPT;
/**
 * @brief This class parses and writes packets to the network device.
 */
class PacketParser
{
public:
    /**
     * @brief The states of the PacketParser.
     */
    enum class State : uint8_t
    {
        Uninitialized  = 0, ///< 0
        Initialized    = 1, ///< 1
        Ready          = 2, ///< 2
        OnEthernet     = 3, ///< 3
        OnInternet     = 4, ///< 4
        OnUserDatagram = 5, ///< 5
        OnBootProtocol = 6, ///< 6
        Error          = 7, ///< 7
    };

    /**
     * @brief Constructor
     */
    PacketParser();

    /**
     * @brief Initializes the PacketParser.
     *
     * @param[in] pCoordinator The event Coordinator.
     */
    void Initialize(Coordinator* pCoordinator) NN_NOEXCEPT;

    /**
     * @brief Finalizes the PacketParser.
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief This is function builds the Ethernet layer.
     *
     * @param[in,out] pOutPacket The destination array of
     * @ref NetworkLayersContainer that represent a network packet.
     *
     * @param[in] layerIndex The current index into the @a pOutPacket
     * packet array.
     *
     * @param[in] pBuffer The buffer containing packet layer bytes.
     *
     * @param[in] bytes The number of bytes left in the buffer.
     *
     * @return On success return the number of bytes processed by this
     * and subsequent functions. On error return -1.
     */
    ssize_t OnEthernetPacket(NetworkLayersContainer pOutPacket[],
                             size_t layerIndex,
                             uint8_t* pBuffer,
                             size_t bytes) NN_NOEXCEPT;
private:
    /**
     * @brief the State variable.
     */
    State m_State;

    /**
     * @brief The event @ref Coordinator.
     */
    Coordinator* m_pCoordinator;

    /**
     * @brief This function builds the Internet Protocol layer.
     *
     * @param[in,out] pOutPacket The destination array of
     * @ref NetworkLayersContainer that represent a network packet.
     *
     * @param[in] layerIndex The current index into the @a pOutPacket
     * packet array.
     *
     * @param[in] pBuffer The buffer containing packet layer bytes.
     *
     * @param[in] bytes The number of bytes left in the buffer.
     *
     * @return On success return the number of bytes processed by this
     * and subsequent functions. On error return -1.
     */
    ssize_t OnInternetPacket(NetworkLayersContainer pOutPacket[],
                             size_t layerIndex,
                             uint8_t* pBuffer,
                             size_t bytes) NN_NOEXCEPT;

    /**
     * @brief This function builds the UDP layer.
     *
     * @param[in,out] pOutPacket The destination array of
     * @ref NetworkLayersContainer that represent a network packet.
     *
     * @param[in] layerIndex The current index into the @a pOutPacket
     * packet array.
     *
     * @param[in] pBuffer The buffer containing packet layer bytes.
     *
     * @param[in] bytes The number of bytes left in the buffer.
     *
     * @return On success return the number of bytes processed by this
     * and subsequent functions. On error return -1.
     */
    ssize_t OnUserDatagramPacket(NetworkLayersContainer pOutPacket[],
                                 size_t layerIndex,
                                 uint8_t* pBuffer,
                                 size_t bytes) NN_NOEXCEPT;

    /**
     * @brief This function builds the Bootp request / DHCP layer.
     *
     * @param[in,out] pOutPacket The destination array of
     * @ref NetworkLayersContainer that represent a network packet.
     *
     * @param[in] layerIndex The current index into the @a pOutPacket
     * packet array.
     *
     * @param[in] pBuffer The buffer containing packet layer bytes.
     *
     * @param[in] bytes The number of bytes left in the buffer.
     *
     * @return On success return the number of bytes processed by this
     * and subsequent functions. On error return -1.
     */
    ssize_t OnBootProtocolRequestPacket(NetworkLayersContainer pOutPacket[],
                                        size_t layerIndex,
                                        uint8_t* pBuffer,
                                        size_t bytes) NN_NOEXCEPT;

    /**
     * @brief This function changes the state and in debug it prints
     * the transition.
     *
     * @param[in] newState This parameter is the new state to transition to.
     */
    void ChangeState(State newState) NN_NOEXCEPT;
};

}}}; // nn::dhcps::detail
