﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief This file is the DHCP Server Lease Table manager, which
 * manages the DHCP Binding table.
 */

namespace nn { namespace dhcps { namespace detail {

class Coordinator;

/**
 * @brief An enum class that contains all of the internal DHCP client
 * states.
 *
 * @details
 */
enum class ClientState : uint8_t
{
    /**
     * @brief The lease is not bound to a client and available for use.
     */
    Unbound   = static_cast<int>(nn::dhcps::State::Unbound),

    /**
     * @brief A lease was sent to the client in response to the DHCP
     * Offer message.
     */
    Offered   = static_cast<int>(nn::dhcps::State::Offered),

    /**
     * @brief The lease is bound to a client and neither t1 or t2
     * timers are expired.
     */
    Bound     = static_cast<int>(nn::dhcps::State::Bound),

    /**
     * @brief The lease is bound to a client  but the t1 timer expired.
     */
    Renewing  = static_cast<int>(nn::dhcps::State::Renewing),

    /**
     * @brief The lease is bound to a client but both t1 and t2 timers
     * expired.
     */
    Rebinding = static_cast<int>(nn::dhcps::State::Rebinding),

    /**
     * @brief Some identifier such as an IP is unavailable and cannot
     * be used in the DHCP lease pool.
     *
     * @details This value was originally applied to the self IP and
     * the broadcast IP. Since that time other steps have been taken
     * to avoid these cases in the lease pool. However the
     * functionality of a reserved slot may still prove useful and
     * so the constant remains.
     */
    Reserved,
};

/**
 * @brief Represents a single record for the DHCP lease table.
 */
struct LeaseRecord
{
    /**
     * @brief This field contains information used to identify a client.
     */
    DhcpClientIdentity client;

    /**
     * @brief The client state value.
     */
    ClientState state;

    /**
     * @brief This field is the lease timer specified in seconds since
     * system startup.
     *
     * @details
     * - When it elapses the lease state advances to the
     * @a ClientState::Unbound state and can be reused.
     */
    uint32_t lease;

    /**
     * @brief The "t1" timer specified in seconds since system startup.
     * When it elapses the lease state advances to the
     * @a ClientState::Renewing state.
     */
    uint32_t t1;

    /**
     * @brief The "t2" timer specified in seconds since system startup.
     * When it elapses the lease state advances to the
     * @a ClientState::Rebinding state.
     */
    uint32_t t2;

    /**
     * @brief On certain errors this value is incremented. It is also
     * used for sorting the lease table with higher values put near
     * the top of the table and lower values at the bottom. The effect
     * being that IP addresses allocation is more fair which reduces
     * duplicate IP problems and other errors caused by transient
     * network conditions.
     */
    uint8_t priority;
};

/**
 * @brief This class manages the DHCP lease table.
 */
class LeaseTableManager : public IEventHandler
{
public:
    /**
     * @brief The states of the LeaseTableManager.
     */
    enum class State : uint8_t
    {
        Uninitialized = 0, ///< 0
        Initialized   = 1, ///< 1
        Available     = 2, ///< 2
        Full          = 3, ///< 3
        Error         = 4, ///< 4
    };

    /**
     * @brief Constructor.
     */
    LeaseTableManager() NN_NOEXCEPT;

    /**
     * @brief Initializes LeaseTableManager with the provided
     * @ref Coordinator.
     *
     * @param[in] pCoordinator The provided Coordinator.
     *
     * @return On success return 0, on error return -1.
     */
    int Initialize(Coordinator* pCoordinator) NN_NOEXCEPT;

    /**
     * @brief Finalizes the LeaseTableManager.
     *
     * @return On success return 0, on error return -1.
     */
    int Finalize() NN_NOEXCEPT;

    /**
     * @brief The IEventHandler event handler.
     *
     * @param[in] e The event to handle.
     */
    void OnEvent(const InternalEvent& e) NN_NOEXCEPT;

    /**
     * @brief Get the desired time for the next timeout event.
     *
     * @param[out] pOutTimeval The desired time for the next timeout event.
     */
    virtual void GetTimeout(nn::socket::TimeVal* pOutTimeval) NN_NOEXCEPT;

    /**
     * @brief This function returns the number of records that can be
     * assigned to DHCP Clients by the @ref LeaseTableManager.
     */
    unsigned int GetTotalRecordCount() NN_NOEXCEPT;

    /**
     * @brief Get a @ref LeaseRecord by @a ClientIdentifierHash value.
     *
     * @param[out] pOutRecord An out pointer that contains a pointer
     * to the LeaseRecord on success
     *
     * @param[in] client The client that originated the message.
     *
     * @return On success return 0, on error return -1.
     */
    int GetRecordByClientIdentifierHash(LeaseRecord *& pOutRecord,
                                        const ClientIdentifierHash& client) NN_NOEXCEPT;

    /**
     * @brief Get a @ref LeaseRecord by the associated IP value.
     *
     * @param[out] pOutRecord An out pointer that contains a pointer
     * to the LeaseRecord on success.
     *
     * @param[in] ip The IP Address that you want to look up in
     * network-byte order.
     *
     * @return On success return 0, on error return -1.
     */
    int GetRecordByIpAddress(LeaseRecord *& pOutRecord,
                             const nn::socket::InAddr& ip) NN_NOEXCEPT;

    /**
     * @brief Get a @a LeaseRecord by the associated
     * @a EthernetMacAddress value.
     *
     * @param[out] pOutRecord An out pointer that contains a pointer
     * to the LeaseRecord on success.
     *
     * @param[in] mac The MAC Address that you want to look up.
     *
     * @return On success return 0, on error return -1.
     */
    int GetRecordByEthernetMacAddress(LeaseRecord *& pOutRecord,
                                      const EthernetMacAddress& mac) NN_NOEXCEPT;

    /**
     * @brief Find an available record in the table.
     *
     * @param[out] pOutRecord An out pointer that, on success, points
     * at a slot in the LeaseTable that can be used for allocation.
     *
     * @return On success return 0, on error return -1.
     */
    int AllocateRecord(LeaseRecord *& pOutRecord) NN_NOEXCEPT;

private:
    /**
     * @brief The current state.
     */
    State m_State;

    /**
     * @brief The event @ref Coordinator.
     */
    Coordinator* m_pCoordinator;

    /**
     * @brief Changes the state variable.
     *
     * @param[in] next The next state to transition to.
     */
    void ChangeState(State next) NN_NOEXCEPT;

    /**
     * @brief Handles @ref EventType::OnTimerExpired events.
     */
    void OnTimerExpired() NN_NOEXCEPT;

    /**
     * @brief Runs the internal state machine.
     *
     * @return The number of changes made.
     */
    unsigned int RunStateMachine() NN_NOEXCEPT;

    /**
     * @brief A count of actions run by the DHCP Manager. It
     * is used for scheduling timer events.
     */
    unsigned int m_ActionCount;

    /**
     * @brief a pointer to the lease table memory passed in from the
     * user at @ref Initialize().
     */
    LeaseRecord* m_pLeaseTable;

    /**
     * @brief The number of records for the configuration.
     */
    unsigned int m_TotalRecordCount;
};

/**
 * @brief Convert a private LeaseRecord to a public DhcpLease structure
 *
 * @param[out] pOutLease a pointer to a DhcpLease structure
 *
 * @param[in] record the LeaseRecord that you want converted.
 */
void LeaseRecordToDhcpLease(DhcpLease* pOutLease, const LeaseRecord& record) NN_NOEXCEPT;

/**
 * @brief Zeroes out a lease table record.
 */
void ZeroRecord(LeaseRecord* pRecord) NN_NOEXCEPT;

}}}; // nn::dhcps::detail
