﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief This file is the header file for the event api.
 */

namespace nn { namespace dhcps { namespace detail {

/**
 * @brief Defines global event constants.
 */
enum class EventType : uint32_t
{
    /**
     * @brief Event indicating a timer expired.
     */
    OnTimerExpired = 0,

    /**
     * @brief The network parser received a packet that could not
     * be parsed properly.
     */
    OnPacketParseError = 100,

    /**
     * @brief The @ref BPFManager read packet bytes.
     */
    OnPacketRead = 101,

    /**
     * @brief The @ref BPFManager sent packet bytes.
     */
    OnPacketWrite = 102,

    /**
     * @brief The @ref BPFManager or @ref UDPSocketManager is trying
     * to open the underlying file or socket as in @ref Start().
     */
    OnFileOpen = 200,

    /**
     * @brief The @ref Coordinator detected a file ready to read event
     * for @ref BPFManager or @ref UDPSocketManager.
     */
    OnFileRead = 201,

    /**
     * @brief The @ref Coordinator detected a file ready to write
     * event for @ref BPFManager or @ref UDPSocketManager.
     */
    OnFileWrite = 202,

    /**
     * @brief The @ref Coordinator detected a file exception event for
     * @ref BPFManager or @ref UDPSocketManager.
     */
    OnFileExcept = 203,

    /**
     * @brief The @ref BPFManager or @ref UDPSocketManager was
     * instructed to close their file as in @ref Stop().
     */
    OnFileClose = 204,

    /**
     * @brief The @ref BPFManager or @ref UDPSocketManager detected
     * an unrecoverable file error. This condition will @ref Stop()
     * the dhcp server.
     */
    OnFileError = 205,

    /**
     * @brief The minimum value of the public API "server" range.
     */
    PublicApiServerMinimum = static_cast<uint32_t>(Event::OnStart),

    /**
     * @brief The maxium value of the public API "server" range.
     */
    PublicApiServerMaximum = static_cast<uint32_t>(Event::OnUnrecoverableError),

    /**
     * @brief The minimum value of the public API "dhcp" range.
     */
    PublicApiDhcpMinimum = static_cast<uint32_t>(Event::OnLeaseOffered),

    /**
     * @brief The maximum value of the public API "dhcp' range.
     */
    PublicApiDhcpMaximum = static_cast<uint32_t>(Event::OnDhcpUnhandledMessage),
};

/**
 * @brief Determine whether or not the provided event type provided
 * falls within a public range.
 */
bool EventTypeIsPublic(EventType e) NN_NOEXCEPT;

/**
 * @brief Provide a string representation for a provided EventType.
 *
 * @details If the value is a public @ref Event then this
 * function also resolves those values.
 *
 * @param[in] in The provided EventType value.
 *
 * @return An EventType string representation.
 */
const char* EventTypeToString(EventType in) NN_NOEXCEPT;

/**
 * @brief A simple event container class.
 */
class InternalEvent
{
public:
    /**
     * @brief Constructor.
     */
    InternalEvent(EventType type, const void* value, size_t size) NN_NOEXCEPT;

    /**
     * @brief The type of event.
     *
     * @return The event type.
     */
    EventType GetType() const NN_NOEXCEPT;

    /**
     * @brief The a contextual size of the event.
     *
     * @details The context of this parameter changes depending on
     * the type of the event. Sometimes it refers to size in bytes
     * and in other cases refers to the number of elements in an array.
     *
     * @return The contextual size of the event.
     */
    size_t GetSize() const NN_NOEXCEPT;

    /**
     * @brief A pointer to the value of the event.
     *
     * @details You must cast the value from void* to the type
     * associated with the event type in @ref InternalEvent::GetType().
     */
    const void* GetValue() const NN_NOEXCEPT;

private:
    /**
     * @brief The type of event.
     */
    EventType m_Type;

    /**
     * @brief the value of the event
     */
    const void* m_Value;

    /**
     * @brief The contextual size of the event.
     */
    size_t m_Size;
};

/**
 * @brief This interface specifies obligations for event handlers.
 */
class IEventHandler
{
public:
    /**
     * @brief Called to process an event.
     *
     * @param[in] e The event to process.
     */
    virtual void OnEvent(const InternalEvent& e) NN_NOEXCEPT = 0;

    /**
     * @brief Returns the desired next timeout value.
     *
     * @param[out] pOutTimeval The desired time for the next timeout event.
     */
    virtual void GetTimeout(nn::socket::TimeVal* pOutTimeval) NN_NOEXCEPT = 0;
};

}}}; // nn::dhcps::detail
