﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcps_DhcpManager.h"

//#define NN_DETAIL_DHCPS_LOG_LEVEL NN_DETAIL_DHCPS_LOG_LEVEL_DEBUG
#define NN_DETAIL_DHCPS_LOG_MODULE_NAME "DhcpManagerUtils"
#include "dhcps_Log.h"

namespace nn { namespace dhcps { namespace detail {

extern Config g_Config;

enum DhcpOptionFlags : uint32_t
{
    Message          = 1,
    ServerIdentifier = 2,
    AddressLeaseTime = 4,
    RenewalTime      = 8,
    RebindingTime    = 16,
    SubnetMask       = 32,
    Broadcast        = 64,
    Router           = 128,
    DnsServers       = 256,
    InterfaceMtu     = 512,
    End              = 1024,
};

int CopyOptionValue(void* pOutValue,
                    size_t outSize,
                    const uint8_t* pBuffer,
                    size_t bufferSize,
                    DhcpOptionCode code) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutValue: %p, outSize: %zu, "
                                   "pBuffer: %p, bufferSize: %zu, "
                                   "code: %s (%u)\n",
                                   pOutValue, outSize,
                                   pBuffer, bufferSize,
                                   DhcpOptionCodeToString(code), code);

    int rc = -1;

    for (const uint8_t* pCursor = pBuffer; pCursor - pBuffer < bufferSize;)
    {
        const DhcpOptionContainer<uint8_t[]>* pOption =
            reinterpret_cast<const DhcpOptionContainer<uint8_t[]>*>(pCursor);

        if (DhcpOptionCode::Pad == pOption->type)
        {
            if (code == DhcpOptionCode::Pad)
            {
                rc = 0;
                goto bail;
            };
            ++pCursor;
            continue;
        }
        else if (DhcpOptionCode::End == pOption->type)
        {
            if (code == DhcpOptionCode::End)
            {
                rc = 0;
                goto bail;
            };
            break;
        }
        else if (code == pOption->type)
        {
            if (outSize < pOption->length)
            {
                NN_DETAIL_DHCPS_LOG_MAJOR("Error: buffer for option %s (%u) "
                                               "is too small (needed: %zu, provided: %zu)\n",
                                               DhcpOptionCodeToString(code), code,
                                               pOption->length, outSize);
            }
            else
            {
                memcpy(pOutValue, &pOption->value, pOption->length);
                rc = 0;
            };
            goto bail;
        };

        pCursor += (sizeof(pOption->type)   +
                    sizeof(pOption->length) +
                    pOption->length);
    };

bail:
    return rc;
};

int AddOptions(uint8_t* pOutCursor,
               size_t outSize,
               DhcpMessageType messageType,
               const LeaseRecord* pRecord,
               DhcpOptionFlags flags) NN_NOEXCEPT
{
#define NN_DETAIL_DHCPS_ADD_OPTION(flag, typeval, optionCode, value)                    \
    if (0 != (flags & flag))                                            \
    {                                                                   \
        if (left < sizeof(DhcpOptionContainer<typeval>))                \
        {                                                               \
            NN_DETAIL_DHCPS_LOG_MAJOR("not enough space left to "  \
                                           "add an option "             \
                                           "(left: %d)\n",              \
                                           left);                       \
            goto bail;                                                  \
        }                                                               \
        else                                                            \
        {                                                               \
            DhcpOptionContainer<typeval> container = {                  \
                optionCode,                                             \
                sizeof(typeval),                                        \
                value                                                   \
            };                                                          \
                                                                        \
            if (left < sizeof(container))                               \
            {                                                           \
                NN_DETAIL_DHCPS_LOG_MAJOR("left (%d) is less "     \
                                               "than container size "   \
                                               "(%d).\n", left,         \
                                               sizeof(container));      \
            };                                                          \
                                                                        \
            memcpy(pOutCursor, &container, sizeof(container));          \
            left -= sizeof(DhcpOptionContainer<typeval>);               \
            pOutCursor += sizeof(DhcpOptionContainer<typeval>);         \
        };                                                              \
    };

    int rc = -1;
    ssize_t left = outSize;
    uint32_t now = Time();

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::Message,
        DhcpMessageType, DhcpOptionCode::DhcpMessageType,
        messageType);

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::SubnetMask,
        uint32_t, DhcpOptionCode::SubnetMask,
        g_Config.GetNetmask().S_addr);

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::ServerIdentifier,
        uint32_t, DhcpOptionCode::ServerIdentifier,
        g_Config.GetIpAddress().S_addr);

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::Broadcast,
        uint32_t, DhcpOptionCode::BroadcastAddress,
        g_Config.GetBroadcast().S_addr);

    if (nullptr != pRecord)
    {
    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::AddressLeaseTime,
        uint32_t, DhcpOptionCode::AddressLeaseTime,
        nn::socket::InetHtonl(RelativeTime(now, pRecord->lease)));

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::RenewalTime, uint32_t,
        DhcpOptionCode::RenewalTime,
        nn::socket::InetHtonl(RelativeTime(now, pRecord->t1)));

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::RebindingTime, uint32_t,
        DhcpOptionCode::RebindingTime,
        nn::socket::InetHtonl(RelativeTime(now, pRecord->t2)));
};

    if (true == g_Config.GetHasGateway())
    {
    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::Router, uint32_t,
        DhcpOptionCode::Routers,
        g_Config.GetGateway().S_addr);
};

    if (1 == g_Config.GetDnsServerCount())
    {
    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::DnsServers, uint32_t,
        DhcpOptionCode::DomainNameServers,
        g_Config.GetDnsServers()[0].S_addr);
}
    else if (2 == g_Config.GetDnsServerCount())
    {
    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::DnsServers, uint64_t,
        DhcpOptionCode::DomainNameServers,
        *reinterpret_cast<const uint64_t*>(g_Config.GetDnsServers()));
};

    NN_DETAIL_DHCPS_ADD_OPTION(DhcpOptionFlags::InterfaceMtu,
        uint16_t, DhcpOptionCode::InterfaceMtu,
        nn::socket::InetHtons(g_Config.GetInterfaceMtu()));

    if (left < sizeof(DhcpOptionCode::End))
    {
    NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    goto bail;
};

    if (left < sizeof(DhcpOptionCode::End))
    {
    NN_DETAIL_DHCPS_LOG_MAJOR("\n");
    goto bail;
};

    *reinterpret_cast<DhcpOptionCode*>(pOutCursor) = DhcpOptionCode::End;
    left -= sizeof(DhcpOptionCode::End);

    rc = outSize - left;

bail:
    return rc;
#undef NN_DETAIL_DHCPS_ADD_OPTION
};

void CopyRequestToResponse(BootpDhcpMessage* pOutResponse, const BootpDhcpMessage* pRequest) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, pRequest: %p\n", pOutResponse, pRequest);

    pOutResponse->operationCode = BootProtocolOperationCode::Reply;
    pOutResponse->hardwareType = pRequest->hardwareType;
    pOutResponse->hardwareLength = pRequest->hardwareLength;
    pOutResponse->hops = pRequest->hops + 1;
    pOutResponse->transactionId = pRequest->transactionId;
    pOutResponse->flags = pRequest->flags;

    memmove(pOutResponse->clientHardwareAddress,
            pRequest->clientHardwareAddress,
            sizeof(pOutResponse->clientHardwareAddress));

    pOutResponse->magicCookie = nn::socket::InetHtonl(
        static_cast<uint32_t>(LibraryConstants::DhcpMagicCookie));
};

ssize_t CreateNakResponse(BootpDhcpMessage* pOutResponse,
                          size_t outResponseSize,
                          const BootpDhcpMessage* pRequest) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                                   "pRequest: %p\n", pOutResponse,
                                   outResponseSize, pRequest);

    CopyRequestToResponse(pOutResponse, pRequest);

    unsigned int options = DhcpOptionFlags::Message | DhcpOptionFlags::End;

    ssize_t rc = AddOptions(pOutResponse->options,
                            outResponseSize,
                            DhcpMessageType::Nak,
                            nullptr,
                            static_cast<DhcpOptionFlags>(options));

    if ( -1 == rc )
    {
        return rc;
    };
    return sizeof(*pOutResponse) + rc;
};

ssize_t CreateCannedResponse(BootpDhcpMessage* pOutResponse,
                             size_t outResponseSize,
                             const BootpDhcpMessage* pRequest,
                             DhcpMessageType messageType,
                             const LeaseRecord* pRecord) NN_NOEXCEPT
{
    NN_DETAIL_DHCPS_LOG_DEBUG("pOutResponse: %p, outResponseSize: %zu, "
                                   "pRequest: %p, messageType: %u, pRecord: %p\n",
                                   pOutResponse, outResponseSize, pRequest,
                                   messageType, pRecord);

    CopyRequestToResponse(pOutResponse, pRequest);

    if (pRecord->state == ClientState::Bound ||
        pRecord->state == ClientState::Renewing ||
        pRecord->state == ClientState::Rebinding)
    {
        pOutResponse->clientIpAddress.S_addr = pRecord->client.ipAddress.S_addr;
    };

    pOutResponse->yourIpAddress.S_addr = pRecord->client.ipAddress.S_addr;
    pOutResponse->serverIpAddress.S_addr = g_Config.GetIpAddress().S_addr;

    unsigned int options = (DhcpOptionFlags::Message          |
                            DhcpOptionFlags::ServerIdentifier |
                            DhcpOptionFlags::AddressLeaseTime |
                            DhcpOptionFlags::RenewalTime      |
                            DhcpOptionFlags::RebindingTime    |
                            DhcpOptionFlags::SubnetMask       |
                            DhcpOptionFlags::Broadcast        |
                            DhcpOptionFlags::Router           |
                            DhcpOptionFlags::DnsServers       |
                            DhcpOptionFlags::InterfaceMtu     |
                            DhcpOptionFlags::End);

    size_t rc = AddOptions(pOutResponse->options,
                           outResponseSize,
                           messageType,
                           pRecord,
                           static_cast<DhcpOptionFlags>(options));
    if ( rc == -1)
    {
        return rc;
    };
    return sizeof(*pOutResponse) + rc;
};


/**
 * @brief Given a byte array compute a hash for the DhcpClient
 *
 * @details If the @a pBytes is NULL or if the size is zero then the
 * default return value is zero. This special value indicates that no
 * DHCP client identifier option was present in the DHCP message.
 *
 * @param[in] pBytes Bytes containing the DHCP client identifier option.
 *
 * @param[in] size The number of bytes contained in @a pBytes.
 *
 * @return A client identifier hash or zero if no bytes are provided.
 */
uint64_t DhcpClientHash(const uint8_t *pBytes, size_t size) NN_NOEXCEPT
{
    const uint64_t Multiplier = 2654435789;
    const uint64_t HashSeed = 104395301;
    const int Prime23 = 23;
    const int Prime37 = 37;

    uint64_t result = HashSeed;

    for (unsigned idx = 0; idx < size; ++idx)
    {
        result += (pBytes[idx] * Multiplier) ^ (result >> Prime23);
    };

    if (result == HashSeed)
    {
        return 0;
    };

    return result ^ (result << Prime37);
};

ClientIdentifierHash ClientIdHashFromMessage(const BootpDhcpMessage* pMessage, size_t size) NN_NOEXCEPT
{
    uint64_t rc = 0;

    for (const uint8_t* pCursor = pMessage->options;
         pCursor < reinterpret_cast<const uint8_t*>(pMessage) + size; )
    {
        const DhcpOptionContainer<uint8_t[]>* pOption =
            reinterpret_cast<const DhcpOptionContainer<uint8_t[]>*>(pCursor);

        if (DhcpOptionCode::ClientIdentifier == pOption->type)
        {
            rc = DhcpClientHash(pOption->value, pOption->length);
            break;
        };

        pCursor += (sizeof(pOption->type)   +
                    sizeof(pOption->length) +
                    pOption->length);
    };

    return rc;
};

}}} // nn::dhcps::detail
