﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief This file contains constants used throughout the library.
 * Some constants are normal programming constants and others are
 * specific to the DHCP protocol.
 */

namespace nn { namespace dhcps { namespace detail {

/**
 * @brief Constant values used throughout the library.
 */
enum class LibraryConstants : uint32_t
{
    /**
     * @brief A reasonable constant for maximum message sizes.
     */
    DhcpMaxMtu                 = 1500,

    /**
     * @brief This value is Bootp "Vendor" options value which
     * indicates that the Bootp options are DHCP options.
     */
    DhcpMagicCookie            = 0x63825363,

    /**
     * @brief This value helps to determine whether or not the
     * DHCP flags were sent with broadcast enabled.
     */
    DhcpBroadcastShift         = 15,

    /**
     * @brief This value is a Internet Protocol broadcast address.
     */
    InternetBroadcast          = 0xFFFFFFFF,

    /**
     *@brief The number of octets / bytes in a MAC address.
     */
    MacAddressSize             = 6,

    /**
     * @brief The maximum size of an IP address string.
     */
    IpAddressStringBufferSize        = sizeof("255.255.255.255"),

    /**
     * @brief The maximum size of an Ethernet MAC address string.
     */
    MacAddressStringBufferSize       = sizeof("FF:FF:FF:FF:FF:FF"),

    /**
     * @brief The maximum number of devices to try when opening BPF.
     */
    MaximumDevices             = 16,

    /**
     * @brief The buffer size of the string to open MaximumDevices
     * BPF devices from /dev/xxxYY
     */
    MaximumDeviceStringBufferSize  = sizeof("/dev/xxxYY"),

    /**
     * @brief The maximum string length of a device string used to
     * open @a MaximumDevices BPF device (i.e. wl0, usb0; without
     * the /dev/)
     */
    MaximumInterfaceNameLength  = sizeof("xxxYY") - 1,
    /**
     * @brief A constant that aligns a structure on a byte boundary.
     */
    NetworkStructureAlignment  = 1,

    /**
     * @brief The size of the BOOTP client hardware address field.
     */
    BootpClientHardwareAddressSize = 16,

    /**
     * @brief The size of the BOOTP client hardware address field.
     */
    BootpServerNameSize = 64,

    /**
     * @brief The size of the BOOTP client file name field.
     */
    BootpFileNameSize = 128,
};

/**
 * @brief BootProtocol Operation Codes
 */
enum class BootProtocolOperationCode : uint8_t
{
    /**
     * @brief Indicates the Bootp message is a request.
     */
    Request = 1,

    /**
     * @brief Indicates the Bootp message is a reply.
     */
    Reply   = 2,
};

/**
 * @brief Provide a string representation for a given a BootProtocolOperationCode value.
 *
 * @param[in] in The given BootProtocolOperationCode value.
 *
 * @return A BootProtocolOperationCode string representation.
 */
const char* BootProtocolOperationCodeToString(BootProtocolOperationCode in) NN_NOEXCEPT;

/**
 * @brief The Boot Protocol Hardware Type field table
 * @details Values come from IANA ARP parameter assignment
 * https://www.iana.org/assignments/arp-parameters/arp-parameters.xhtml#arp-parameters-2
 */
enum class BootProtocolHardwareType : uint8_t
{
    Ethernet =  1, ///< Networking as defined by 802.3.
    Ieee802  =  6, ///< Non-Ethernet 802 such as Token Ring.
    Fddi     =  8, ///< Fiber Distributed Data Interface.
    Ieee1394 = 24, ///< Firewire.
};

/**
 * @brief Provide a string representation for a given a BootProtocolHardwareType value.
 *
 * @param[in] in The given BootProtocolHardwareType value.
 *
 * @return A BootProtocolHardwareType string representation.
 */
const char* BootProtocolHardwareToString(BootProtocolHardwareType in) NN_NOEXCEPT;

/**
 * @brief DHCP Option Codes.
 *
 * @details Values come from IANA DHCP & Bootp Options tables.
 * https://www.iana.org/assignments/bootp-dhcp-parameters/bootp-dhcp-parameters.xhtml#options
 */
enum class DhcpOptionCode : uint8_t
{
    Pad               =   0, ///< DHCP Padding option. [RFC2132] / 0 bytes data
    SubnetMask        =   1, ///< Subnet mask value. [RFC2132] / 4 bytes data.
    TimeOffset        =   2, ///< N/4 Time Offset in Seconds from UTC. [RFC2132] / N bytes data.
    Routers           =   3, ///< N/4 Router addresses. [RFC2132] / N bytes data.
    TimeServers       =   4, ///< N/4 Timeserver addresses. [RFC2132] / N bytes data.
    NameServers       =   5, ///< N/4 IEN-116 Server addresses. [RFC2132] / N bytes data.
    DomainNameServers =   6, ///< N/4 DNS Server addresses. [RFC2132] / N bytes data.
    LogServers        =   7, ///< N/4 Logging Server addresses. [RFC2132] / N bytes data.
    QuotesServers     =   8, ///< N/4 Quotes Server addresses. [RFC2132] / N bytes data.
    LprServers        =   9, ///< N/4 Printer Server addresses. [RFC2132] / N bytes data.
    ImpressServers    =  10, ///< N/4 Impress Server addresses. [RFC2132] / N bytes data.
    RlpServers        =  11, ///< N/4 RLP Server addresses. [RFC2132] / N bytes data.
    Hostname          =  12, ///< Hostname string. [RFC2132] / N bytes data.
    BootFileSize      =  13, ///< Size of boot file in 512 byte chunks. [RFC2132] / 2 bytes data.
    MeritDumpFile     =  14, ///< Client to dump and name the file to dump it to. [RFC2132] / N bytes data.
    DomainName        =  15, ///< The DNS domain name of the client. [RFC2132] / N bytes data.
    SwapServer        =  16, ///< Swap Server address. [RFC2132] / N bytes data.
    RootPath          =  17, ///< Path name for root disk. [RFC2132] / N bytes data.
    ExtensionFile     =  18, ///< Path name for more BOOTP info. [RFC2132] / N bytes data.
    ForwardOnOff      =  19, ///< Enable/Disable IP Forwarding. [RFC2132] / 1 bytes data.
    SrcRteOnOff       =  20, ///< Enable/Disable Source Routing. [RFC2132] / 1 bytes data.
    PolicyFilter      =  21, ///< Routing Policy Filters. [RFC2132] / N bytes data.
    MaxDgAssembly     =  22, ///< Max Datagram Reassembly Size. [RFC2132] / 2 bytes data.
    DefaultIpTtl      =  23, ///< Default IP Time to Live. [RFC2132] / 1 bytes data.
    MtuTimeout        =  24, ///< Path MTU Aging Timeout. [RFC2132] / 4 bytes data.
    MtuPlateau        =  25, ///< Path MTU Plateau Table. [RFC2132] / N bytes data.
    InterfaceMtu      =  26, ///< Interface MTU Size. [RFC2132] / 2 bytes data.
    AllSubnetsLocal   =  27, ///< All Subnets are Local. [RFC2132] / 1 bytes data.
    BroadcastAddress  =  28, ///< Broadcast Address. [RFC2132] / 4 bytes data.
    MaskDiscovery     =  29, ///< Perform Mask Discovery. [RFC2132] / 1 bytes data.
    MaskSupplier      =  30, ///< Provide Mask to Others. [RFC2132] / 1 bytes data.
    RouterDiscovery   =  31, ///< Perform Router Discovery. [RFC2132] / 1 bytes data.
    RouterRequest     =  32, ///< Router Solicitation Address. [RFC2132] / 4 bytes data.
    StaticRoute       =  33, ///< Static Routing Table. [RFC2132] / N bytes data.
    Trailers          =  34, ///< Trailer Encapsulation. [RFC2132] / 1 bytes data.
    ARPTimeout        =  35, ///< ARP Cache Timeout. [RFC2132] / 4 bytes data.
    Ethernet          =  36, ///< Ethernet Encapsulation. [RFC2132] / 1 bytes data.
    DefaultTcpTtl     =  37, ///< Default TCP Time to Live. [RFC2132] / 1 bytes data.
    KeepaliveTime     =  38, ///< TCP Keepalive Interval. [RFC2132] / 4 bytes data.
    KeepaliveData     =  39, ///< TCP Keepalive Garbage. [RFC2132] / 1 bytes data.
    NisDomain         =  40, ///< NIS Domain Name. [RFC2132] / N bytes data.
    NisServers        =  41, ///< NIS Server Addresses. [RFC2132] / N bytes data.
    NtpServers        =  42, ///< NTP Server Addresses. [RFC2132] / N bytes data.
    VendorSpecific    =  43, ///< Vendor Specific Information. [RFC2132] / N bytes data.
    NetbiosNameSrv    =  44, ///< NETBIOS Name Servers. [RFC2132] / N bytes data.
    NetbiosDistSrv    =  45, ///< NETBIOS Datagram Distribution. [RFC2132] / N bytes data.
    NetbiosNodeType   =  46, ///< NETBIOS Node Type. [RFC2132] / 1 bytes data.
    NetbiosScope      =  47, ///< NETBIOS Scope. [RFC2132] / N bytes data.
    XWindowFont       =  48, ///< X Window Font Server. [RFC2132] / N bytes data.
    XWindowManager    =  49, ///< X Window Display Manager. [RFC2132] / N bytes data.
    RequestedAddress  =  50, ///< Requested IP Address. [RFC2132] / 4 bytes data.
    AddressLeaseTime  =  51, ///< IP Address Lease Time. [RFC2132] / 4 bytes data.
    Overload          =  52, ///< Overload sname or file. [RFC2132] / 1 bytes data.
    DhcpMessageType   =  53, ///< DHCP Message Type. [RFC2132] / 1 bytes data.
    ServerIdentifier  =  54, ///< DHCP Server Identification. [RFC2132] / 4 bytes data.
    ParameterList     =  55, ///< Parameter Request List. [RFC2132] / N bytes data.
    DhcpMessage       =  56, ///< DHCP Error Message. [RFC2132] / N bytes data.
    DhcpMaxMessageSize=  57, ///< DHCP Maximum Message Size. [RFC2132] / 2 bytes data.
    RenewalTime       =  58, ///< DHCP Renewal (T1) Time. [RFC2132] / 4 bytes data.
    RebindingTime     =  59, ///< DHCP Rebinding (T2) Time. [RFC2132] / 4 bytes data.
    ClassId           =  60, ///< Class Identifier. [RFC2132] / N bytes data.
    ClientIdentifier  =  61, ///< Client Identifier. [RFC2132] / N bytes data.
    NetwareIpDomain   =  62, ///< NetWare/IP Domain Name. [2242] / N bytes data.
    NetwareIp         =  63, ///< NetWare/IP sub Options. [2242] / N bytes data.
    NisDomainName     =  64, ///< NIS+ v3 Client Domain Name. [RFC2132] / N bytes data.
    NisServerAddr     =  65, ///< NIS+ v3 Server Addresses. [RFC2132] / N bytes data.
    ServerName        =  66, ///< TFTP Server Name. [RFC2132] / N bytes data.
    BootfileName      =  67, ///< Boot File Name. [RFC2132] / N bytes data.
    HomeAgentAddrs    =  68, ///< Home Agent Addresses. [RFC2132] / N bytes data.
    SmtpServer        =  69, ///< Simple Mail Server Addresses. [RFC2132] / N bytes data.
    Pop3Server        =  70, ///< Post Office Server Addresses. [RFC2132] / N bytes data.
    NntpServer        =  71, ///< Network News Server Addresses. [RFC2132] N bytes data.
    WwwServer         =  72, ///< WWW Server Addresses. [RFC2132] / N bytes data.
    FingerServer      =  73, ///< Finger Server Addresses. [RFC2132] / N bytes data.
    IrcServer         =  74, ///< Chat Server Addresses. [RFC2132] / N bytes data.
    StreetTalkServer  =  75, ///< StreetTalk Server Addresses. [RFC2132] / N bytes data.
    StdaServer        =  76, ///< ST Directory Assist. Addresses. [RFC2132] / N bytes data.
    UserClass         =  77, ///< User Class Information. [RFC3004] / N bytes data.
    DirectoryAgent    =  78, ///< directory agent information. [RFC2610] / N bytes data.
    ServiceScope      =  79, ///< service location agent scope. [RFC2610] / N bytes data.
    RapidCommit       =  80, ///< Rapid Commit. [RFC4039] / 0 bytes data.
    ClientFQDN        =  81, ///< Fully Qualified Domain Name. [RFC4702] / N bytes data.
    RelayAgentInfo    =  82, ///< Relay Agent Information. [RFC3046] / N bytes data.
    Isns              =  83, ///< Internet Storage Name Service. [RFC4174] / N bytes data.
    NdsServers        =  85, ///< Novell Directory Services. [RFC2241] / N bytes data.
    NdsTreeName       =  86, ///< Novell Directory Services. [2241] / N bytes data.
    NdsContext        =  87, ///< Novell Directory Services. [2241] / N bytes data.
    BcmcsControllerDomainNameList =  88, ///< See [RFC4280].
    BcmcsControllerIpv4Address =  89,    ///< See [RFC4280].
    Authentication    =  90,             ///< Authentication. [RFC 3118] / N bytes data.
    ClientLastTransactionTime =  91,     ///< See [RFC4388].
    AssociatedIp      =  92, ///< See [RFC4388].
    ClientSystem      =  93, ///< Client System Architecture. [RFC4578] / N bytes data.
    ClientNDI         =  94, ///< Client Network Device Interface. [RFC4578] / N bytes data.
    Ldap              =  95, ///< Lightweight Directory Access Protocol. [RFC3679] / N bytes data.
    UuidGuid          =  97, ///< UUID/GUID-based Client Identifier. [RFC4578] / N bytes data.
    UserAuth          =  98, ///< Open Group's User Authentication. [RFC2485] / N bytes data.
    GeoconfCivic      =  99, ///<See [RFC4776].
    Pcode             = 100, ///< IEEE 1003.1 TZ String. [RFC4833] / N bytes data.
    Tcode             = 101, ///< Reference to the TZ Database. [RFC4833] / N bytes data.
    NetinfoAddress    = 112, ///< NetInfo Parent Server Address. [RFC3679] / N bytes data.
    NetinfoTag        = 113, ///< NetInfo Parent Server Tag. [RFC3679] / N bytes data.
    Url               = 114, ///< URL. [RFC3679] / N bytes data.
    AutoConfig        = 116, ///< DHCP Auto-Configuration. [RFC2563] / N bytes data.
    NameServiceSearch = 117, ///< Name Service Search. [RFC2937] / N bytes data.
    SubnetSelection   = 118, ///< Subnet Selection Option. [RFC3011] / 4 bytes data.
    DomainSearch      = 119, ///< DNS domain search list. [RFC3397] / N bytes data.
    SipServersDhcp    = 120, ///< SIP Servers DHCP Option. [RFC3361] / N bytes data.
    ClasslessStaticRoute = 121, ///< Classless Static Route Option. [RFC3442] / N bytes data.
    Ccc               = 122, ///< CableLabs Client Configuration. [RFC3495] / N bytes data.
    GeoConf           = 123, ///< GeoConf Option. [RFC6225] / 16 bytes data.
    ViVendorClass     = 124, ///< Vendor-Identifying Vendor Class. [RFC3925]
    ViVendorSpecificInformation = 125, ///< Vendor-Identifying Vendor-Specific Information. [RFC3925]
    PanaAgent         = 136, ///< See [RFC5192].
    V4Lost            = 137, ///< See [RFC5223].
    CapWapAcv4        = 138, ///< CAPWAP Access Controller addresses. [RFC5417] / N bytes data.
    Ipv4AddressMos    = 139, ///< a series of suboptions. [RFC5678] / N bytes data.
    Ipv4FqdnMos       = 140, ///< a series of suboptions. [RFC5678] / N bytes data.
    SipUAConfigurationServiceDomains = 141, ///< List of domain names to search for SIP User Agent Configuration. [RFC6011] / N bytes data.
    Ipv4AddressAndSf  = 142, ///< ANDSF IPv4 Address Option for DHCPv4. [RFC6153] / N bytes data.
    GeoLoc            = 144, ///< Geospatial Location with Uncertainty. [RFC6225] / 16 bytes data.
    ForceRenewOnceCapable = 145, ///< Forcerenew once Capable. [RFC6704] / 1 bytes data.
    RdnsSelection     = 146, ///< Information for selecting RDNSS. [RFC6731] / N bytes data.
    TftpServerAddress = 150, ///< See [RFC5859].
    StatusCode        = 151, ///< Status code and optional N byte text message describing status. [RFC6926] / N+1 bytes data.
    BaseTime          = 152, ///< Absolute time (seconds since Jan 1, 1970) message was sent. [RFC6926] / 4 bytes data.
    StartTimeOfState  = 153, ///< Number of seconds in the past when client entered current state. [RFC6926] / 4 bytes data.
    QueryStartTime    = 154, ///< Absolute time (seconds since Jan 1, 1970) for beginning of query. [RFC6926] / 4 bytes data.
    QueryEndTime      = 155, ///< Absolute time (seconds since Jan 1, 1970) for end of query. [RFC6926] / 4 bytes data.
    DhcpState         = 156, ///< State of IP address. [RFC6926] / 1 bytes data.
    DataSource        = 157, ///< Indicates information came from local or remote server. [RFC6926] / 1 bytes data.
    V4PcpServer       = 158, ///< Includes one or multiple lists of PCP server IP addresses. [RFC7291] / Variable; the minimum length is 5.
    V4PortParams      = 159, ///< This option is used to configure a set of ports bound to a shared IPv4 address. [RFC7618] / 4 bytes data.
    DhcpCaptivePortal = 160, ///< DHCP Captive-Portal. [RFC7710] / N bytes data.
    MudUrlV4          = 161, ///< Manufacturer Usage Descriptions. [draft-ietf-opsawg-mud] / N (variable) bytes data.
    IpTelephone       = 176, ///< IANA does not provide any documentation for this option.
    PacketCableHome   = 177, ///< IANA does not provide any documentation for this option.
    PxeLinuxMagic     = 208, ///< Magic string = F1:00:74:7E. [5071][Deprecated] / 4 bytes data.
    ConfigurationFile = 209, ///< Configuration file. [5071] / N bytes data.
    PathPrefix        = 210, ///< Path Prefix Option. [5071] / N bytes data.
    RebootTime        = 211, ///< Reboot Time. [5071] / 4 bytes data.
    Option6Rd         = 212, ///< OPTION_6RD with N/4 6rd BR addresses. [5969] / 18 + N bytes data.
    V4AccessDomain    = 213, ///< Access Network Domain Name. [5986] / N bytes data.
    SubnetAllocation  = 220, ///< Subnet Allocation Option. [6656] / N bytes data.
    VirtualSubnetVss  = 221, ///< See [RFC6607].
    End               = 255, ///< DHCP End of options. [RFC2132] / 0 bytes data.
};

/**
 * @brief Provide a string representation for a given a DhcpOptionCode value.
 *
 * @param[in] in The given DhcpOptionCode value.
 *
 * @return A DhcpOptionCode string representation.
 */
const char* DhcpOptionCodeToString(DhcpOptionCode in) NN_NOEXCEPT;

/**
 * @brief DHCP Mesage Code Constants
 * @details Values come from IANA DHCP & Bootp Options tables
 * https://www.iana.org/assignments/bootp-dhcp-parameters/bootp-dhcp-parameters.xhtml#message-type-53
 */
enum class DhcpMessageType : uint8_t
{
    Discover         =  1,  ///< DHCP client discover server. [RFC2132]
    Offer            =  2,  ///< DHCP server offer lease. [RFC2132]
    Request          =  3,  ///< DHCP Client request lease. [RFC2132]
    Decline          =  4,  ///< DHCP client declined lease. [RFC2132]
    Ack              =  5,  ///< DHCP client acknowledgement. [RFC2132]
    Nak              =  6,  ///< DHCP client negative-acknowledgment. [RFC2132]
    Release          =  7,  ///< DHCP client return lease to server. [RFC2132]
    Inform           =  8,  ///< DHCP client request information. [RFC2132]
    ForceRenew       =  9,  ///< DHCP server reneges lease and forces client to switch to renewing state. [RFC3203]
    LeaseQuery       = 10,  ///< DHCP relay-agent to server message to enquire about location of an IP endpoint. [RFC4388]
    LeaseUnassigned  = 11,  ///< DHCP server response to Lease query; indicates no active lease with resulting IP address but the DHCP server is authoritative for the IP. [RFC4388]
    LeaseUnknown     = 12,  ///< DHCP server response to LeaseQuery; indicates that the DHCP server has no knowledge of the information specified in the query.  [RFC4388]
    LeaseActive      = 13,  ///< DHCP server response to LeaseQuery; allows a relay agent to determine the IP endpoint location and remaining duration of the lease. [RFC4388]
    BulkLeaseQuery   = 14,  ///< DHCP requestor to server; save entire database to stable storage. [RFC6926]
    LeaseQueryDone   = 15,  ///< DHCP server to requestor indicating that BulkLeaseQuery is done. [RFC6926]
    ActiveLeaseQuery = 16,  ///< DHCP requestor to server; save only active database to stable storage. [RFC7724]
    LeaseQueryStatus = 17,  ///< DHCP requestor to server TCP keep-alive. [RFC7724]
    Tls              = 18,  ///< DHCP Transport Layer Security. [RFC7724]
};

/**
 * @brief Provide a string representation for a given a DhcpMessageType value.
 *
 * @param[in] in The given DhcpMessageType value.
 *
 * @return A DhcpMessageType string representation.
 */
const char* DhcpMessageTypeToString(DhcpMessageType in) NN_NOEXCEPT;


}}}; // nn::dhcps::detail
