﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "dhcps_Common.h"

/**
 * @file
 *
 * @brief Header file for global DHCP server configuration
 */

namespace nn { namespace dhcps { namespace detail {

/**
 * @brief The Config class contains values provided by the user and
 * values acquired from the socket interface that are used throughout
 * the DHCP server.
 */
class Config
{
public:
    /**
     * @brief The constructor.
     */
    Config() NN_NOEXCEPT;

    /**
     * @brief Copies values from the provided public-API
     * @a UserConfiguration structure to the Config object.
     *
     * @param[in] userConfig The provided @a UserConfiguration structure.
     *
     * @return Returns a success or failure result code.
     */
    Result FromUserConfiguration(const UserConfiguration& userConfig) NN_NOEXCEPT;

    /**
     * @brief Acquires the network configuration via the
     * @ref nn::socket::Ioctl interface.
     *
     * @return This function returns a success or failure result code.
     */
    Result AcquireNetworkValues() NN_NOEXCEPT;

    /**
     * @brief Sets memory for the lease table.
     *
     * @param[in] pMemory A pointer to the memory allocated for the lease
     * table.
     *
     * @param size The size of the memory allocated for the lease table.
     */
    void SetLeaseTableMemory(void* pMemory, size_t size) NN_NOEXCEPT;

    /**
     * @brief Prints the configuration object if debug logging is
     * enabled.
     */
    void Print() NN_NOEXCEPT;

    /**
     * @brief Gets the name of the provided network interface to
     * bind to such as "wl0", "usb0", etc.
     *
     * @return The configured interface name string.
     */
    const char* GetInterfaceName() const NN_NOEXCEPT;

    /**
     * @brief Gets a boolean value indicating whether or not a gateway
     * was provided.
     *
     * @return True or false indicating whether or not a gatway was
     * provided.
     */
    bool GetHasGateway() const NN_NOEXCEPT;

    /**
     * @brief Gets the gateway address.
     *
     * @return The gateway address in network-byte order.
     */
    nn::socket::InAddr GetGateway() const NN_NOEXCEPT;

    /**
     * @brief Gets the number of provided DNS servers.
     *
     * @details The minimum value is zero and the maximum value is 2.
     *
     * @return The number of provided DNS servers.
     */
    uint32_t GetDnsServerCount() const NN_NOEXCEPT;

    /**
     * @brief Gets a pointer to the DNS server array.
     *
     * @details This value is passed in from the UserConfiguration
     * structure and is in network-byte order.
     *
     * @return The array of DNS servers.
     */
    const nn::socket::InAddr* GetDnsServers() const NN_NOEXCEPT;

   /**
    * @brief Gets the lease table memory provided to the library by
    * the application.
    *
    * @return The provided lease table memory pointer.
    */
    void* GetLeaseTableMemory() NN_NOEXCEPT;

    /**
     * @brief Gets the lease table memory size provided to the library
     * by the application.
     *
     * @return The provided lease table memory size.
     */
    size_t GetLeaseTableMemorySize() const NN_NOEXCEPT;

    /**
     * @brief Gets the user-provided library-to-application callback
     * function pointer.
     *
     * @return The library-to-callback function pointer.
     */
    ApplicationCallback GetCallback() const NN_NOEXCEPT;

    /**
     * @brief Gets the provided context pointer, which is passed back
     * to the application through the library-to-application callback
     * function pointer.
     *
     * @return The provided context pointer.
     */
    void* GetContext() const NN_NOEXCEPT;

    /**
     * @brief Gets the amount of time in seconds that a record
     * persists in the offered state.
     *
     * @return The number of seconds that a record persists in the
     * offered state.
     */
    uint32_t GetOfferTimerSeconds() const NN_NOEXCEPT;

    /**
     * @brief Gets the total number of seconds offered for a lease.
     * It is also the number of seconds the lease extends when a
     * client renews or rebinds with a dhcp request message.
     *
     * @return The number of seconds offered for the lease.
     */
    uint32_t GetLeaseTimerSeconds() const NN_NOEXCEPT;

    /**
     * @brief Gets the ratio that defines the t1 timer value, which
     * corresponds to the @a nn::dhcps::State::Renewing state.
     *
     * @return The t1 timer ratio.
     */
    double GetT1Ratio() const NN_NOEXCEPT;

    /**
     * @brief Gets the ratio that defines the t2 timer value, which
     * corresponds to the @a nn::dhcps::State::Rebinding state.
     *
     * @return The t2 timer ratio.
     */
    double GetT2Ratio() const NN_NOEXCEPT;

    /**
     * @brief Gets the UDP port for the DHCP client.
     *
     * @details By default this value is the standard port 68.
     *
     * @return The DHCP client port.
     */
    uint16_t GetDhcpClientUdpPort() const NN_NOEXCEPT;

    /**
     * @brief Gets the UDP port for the DHCP server.
     *
     * @details By default this value is the standard port 67.
     *
     * @return The DHCP server port.
     */
    uint16_t GetDhcpServerUdpPort() const NN_NOEXCEPT;

    /**
     * @brief Gets the MAC address corresponding to the provided
     * network interface.
     *
     * @param[out] pOutAddress A pointer to a EthernetMacAddress.
     *
     * @return The Ethernet MAC address.
     */
    void GetInterfaceMacAddress(EthernetMacAddress* pOutAddress) const NN_NOEXCEPT;

    /**
     * @brief Gets acquired interface MTU size that corresponds to
     * provided network interface name.
     *
     * @return The MTU for the configured network interface.
     */
    uint16_t GetInterfaceMtu() const NN_NOEXCEPT;

    /**
     * @brief Gets the acquired IP address corresponding to the
     * provided network interface. The DHCP server also uses this
     * value as the source address for DHCP messages.
     *
     * @return The DHCP Server IP address.
     */
    nn::socket::InAddr GetIpAddress() const NN_NOEXCEPT;

    /**
     * @brief Gets the acquired netmask value for the provided network
     * interface. The DHCP server also uses this value as the netmask
     * DHCP option in DHCP ack messages.
     *
     * @return The netmask.
     */
    nn::socket::InAddr GetNetmask() const NN_NOEXCEPT;

    /**
     * @brief Gets the acquired network address for the provided
     * network interface.
     *
     * @return The network address.
     */
    nn::socket::InAddr GetNetwork() const NN_NOEXCEPT;

    /**
     * @brief Gets the acquired broadcast address for the provided
     * network interface.
     *
     * @return The broadcast address.
     */
    nn::socket::InAddr GetBroadcast() const NN_NOEXCEPT;

    /**
     * @brief Gets the provided beginning address of the DHCP range.
     *
     * @return The beginning address.
     */
    nn::socket::InAddr GetRangeBegin() const NN_NOEXCEPT;

    /**
     * @brief Gets the provided ending address of the DHCP range.
     *
     * @return The ending address.
     */
    nn::socket::InAddr GetRangeEnd() const NN_NOEXCEPT;

private:
    /**
     * @brief The name of the interface to bind to such as "wl0",
     * "usb0", etc.
     */
    char m_pInterfaceName[static_cast<size_t>(LibraryConstants::MaximumInterfaceNameLength)];

    /**
     * @brief Indicates whether or not the @a m_Gateway value should be
     * sent to DHCP clients.
     */
    bool m_HasGateway;

    /**
     * @brief The default gatway address in network-byte order that
     * is sent to DHCP clients if @a m_HasGateway is set to true.
     */
    nn::socket::InAddr m_Gateway;

    /**
     * @brief The number of zero-filled DNS servers present in the
     * @a m_pDnsServers array.
     */
    uint32_t m_DnsServerCount;

    /**
     * @brief Contains the DNS server IP addresses in network-
     * byte order. These values are sent to DHCP clients if @a
     * m_DnsServerCount is greater than zero.
     */
    nn::socket::InAddr m_pDnsServers[MaximumDnsServerCount];

   /**
    * @brief A pointer to the memory provided to the library by the
    * application for the lease table.
    */
    void* m_pLeaseTableMemory;

    /**
     * @brief The size of the lease table memory provided to the
     * library by the application.
     */
    size_t m_LeaseTableMemorySize;

    /**
     * @brief The callback function provided to the library by the
     * application. If this field is required and Events
     * are sent to the application via the callback.
     */
    ApplicationCallback m_pCallback;

    /**
     * @brief The context pointer passed back as an argument to the
     * application via the callback.
     */
    void* m_pContext;

    /**
     * @brief The amount of time in seconds that a record persists
     * in the @a nn::dhcps::State::Offered state.
     */
    uint32_t m_OfferTimerSeconds;

    /**
     * @brief The number of seconds an acknowledged lease persists in
     * the DHCP server until it expires, returns to the unbound state,
     * and is available for reuse. It is also the number of seconds
     * the lease extends when a client renews or rebinds.
     */
    uint32_t m_LeaseTimerSeconds;

    /**
     * @brief The ratio to determine the t1 timer, which corresponds
     * to the @a nn::dhcps::State::Renewing state. The recommended default
     * value is 1/2 or 50% of the total lease time.
     *
     * This value must be greater than zero and less than 1.
     */
    double m_T1Ratio;

    /**
     * @brief The ratio to determine the t2 timer, which corresponds
     * to the nn::dhcps::State::Rebinding state. The recommended default
     * value is 7/8 or 87.5% of the total lease time.
     *
     * This value must be greater than zero and less than 1.
     */
    double m_T2Ratio;

    /**
     * @brief The UDP port for the DHCP messages client.
     *
     * By default this value is the standard port 68.
     */
    uint16_t m_ClientUdpPort;

    /**
     * @brief The UDP port for the DHCP server.
     *
     * By default this value is the standard port 67.
     */
    uint16_t m_ServerUdpPort;

    /**
     * @brief The mac address that corresponds to the provided
     * network interface.
     */
    EthernetMacAddress m_InterfaceMacAddress;

    /**
     * @brief The MTU size that corresponds to the provided
     * network  interface.
     */
    uint16_t m_InterfaceMtu;

    /**
     * @brief The acquired IP address corresponding to the provided
     * network interface. This value is used as the source IP and the
     * server field of bootp messages.
     *
     * @details This value is in network-byte order.
     */
    nn::socket::InAddr m_IpAddress;

    /**
     * @brief The acquired netmask that corresponds to the provided
     * network interface. The DHCP server also uses this value as the
     * netmask DHCP option in DHCP ack messages.
     *
     * @details This value is in network-byte order.
     */
    nn::socket::InAddr m_Netmask;

    /**
     * @brief The acquired network address that corresponds to the
     * provided network interface. The DHCP server also uses this value
     * as the network address DHCP option in DHCP ACK messages.
     *
     * @details This value is in network-byte order.
     */
    nn::socket::InAddr m_Network;

    /**
     * @brief The acquired broadcast address that corresponds to the
     * provided network interface. The DHCP server also uses this
     * value as the broadcast address DHCP option in DHCP ack messages.
     *
     * @details This value is in network-byte order.
     */
    nn::socket::InAddr m_Broadcast;

    /**
     * @brief The provided range begin value is the IP address that
     * begins the lease table range.
     *
     * @details This value is in network-byte order.
     */
    nn::socket::InAddr m_RangeBegin;

    /**
     * @brief The provided range end value is the IP address that ends
     * the lease table range.
     *
     * @details This value is in network-byte order.
     */
    nn::socket::InAddr m_RangeEnd;
};

/**
 * @brief Detect whether or not a network change occurred.
 *
 * @details This function calls into bsdsocket via the @ref
 * Config::AcquireNetworkValues function and compares the current
 * network configuration to the global configuration. If there is a
 * difference between these two configurations then the function
 * returns true otherwise it returns false.
 *
 * @return True or false value indicating whether or not a network
 * change was detected.
 */
bool DetectNetworkChange() NN_NOEXCEPT;

}}}; // nn::dhcps::detail
