﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dauth/detail/dauth_ServiceImpl.h>

#include <mutex>

#include <nn/dauth/detail/dauth_Interface.sfdl.h>
#include <nn/dauth/detail/dauth_Result.h>
#include <nn/dauth/dauth_ServiceResource.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/settings/system/settings_FirmwareVersion.h>
#endif

#include "dauth_AsyncResultImpl.h"
#include "dauth_NdasDriver.h"

namespace nn { namespace dauth { namespace detail {
namespace {
bool IsAlmostExpired(TimeSpan t) NN_NOEXCEPT
{
    static const TimeSpan ExpirationMargin(TimeSpan::FromHours(3));
    return t <= (os::GetSystemTick().ToTimeSpan() + ExpirationMargin);
}

uint32_t GetSystemVersion() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    settings::system::FirmwareVersion fwVer;
    settings::system::GetFirmwareVersion(&fwVer);
    return static_cast<uint32_t>(fwVer.GetComparableVersion());
#else
    return 0x00u;
#endif
}

template <typename TokenCacheType>
class AuthenticationTask
    : public util::Executable
{
public:
    using TokenAquirer = Result(*)(TimeSpan*, int*, char*, size_t, uint64_t, CURL*, void*, size_t, const util::Cancelable*);

protected:
    virtual Result ExecuteImpl(void* rawBuffer, size_t rawBufferSize) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_SDK_ASSERT(rawBufferSize == sizeof(ExecutionResource));
        NN_UNUSED(rawBufferSize);
        auto& resource = *reinterpret_cast<ExecutionResource*>(rawBuffer);

        if (!m_Refresh)
        {
            NN_RESULT_DO(m_ErrorCache.Check(m_ClientId));
            if (m_TokenCache.IsAvailable(m_ClientId))
            {
                // キャッシュを利用するため再取得しない
                NN_RESULT_SUCCESS;
            }
        }

        struct Buffer
        {
            NN_ALIGNAS(std::alignment_of<std::max_align_t>::value)
                char tokenBuffer[RequiredBufferSizeForDeviceAuthenticationToken];
            NN_ALIGNAS(std::alignment_of<std::max_align_t>::value)
                char workBuffer[NdasDriver::RequiredWorkBufferSize];
        };
        NN_STATIC_ASSERT(sizeof(resource.buffer) >= sizeof(Buffer));
        NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(rawBuffer) % std::alignment_of<Buffer>::value == 0);
        auto& buffer = *reinterpret_cast<Buffer*>(resource.buffer);

        TimeSpan expiration;
        int length;
        auto r = m_Acquirer(
            &expiration, &length, buffer.tokenBuffer, sizeof(buffer.tokenBuffer),
            m_ClientId,
            resource.curlHandle, buffer.workBuffer, sizeof(buffer.workBuffer), this);
        if (ResultNdasStatusNo0007::Includes(r) || ResultNdasStatusNo0008::Includes(r))
        {
            // 次のエラーが返る場合は、サーバー負荷低減のためエラーをキャッシュする
            // - 0007 : System update is required.
            // - 0008 : Device has been banned.
            m_ErrorCache.Put(m_ClientId, r);
        }
        NN_RESULT_DO(r);

        m_TokenCache.Put(m_ClientId, expiration, buffer.tokenBuffer, length);
        NN_RESULT_SUCCESS;
    }

private:
    TokenAquirer m_Acquirer;
    uint64_t m_ClientId;
    bool m_Refresh;
    TokenCacheType& m_TokenCache;
    ErrorCache& m_ErrorCache;

public:
    AuthenticationTask(TokenAquirer acquirer, uint64_t clientId, bool refresh, TokenCacheType& tokenCache, ErrorCache& errorCache) NN_NOEXCEPT
        : m_Acquirer(acquirer)
        , m_ClientId(clientId)
        , m_Refresh(refresh)
        , m_TokenCache(tokenCache)
        , m_ErrorCache(errorCache)
    {
    }
};
} // ~nn::dauth::<anonymous>

template <size_t Length, typename Tag>
void TokenCache<Length, Tag>::Put(uint64_t clientId, TimeSpan expiration, const char* token, int length) NN_NOEXCEPT
{
    NN_SDK_ASSERT(length <= Length);
    NN_UNUSED(length);
    if (IsAlmostExpired(expiration))
    {
        // そもそもキャッシュすべきではない
        return;
    }

    auto f = [expiration, token](TokenType* p) {
        p->expiration = expiration;
        util::Strlcpy(p->data, token, sizeof(p->data));
    };
    Base::Put({clientId, GetSystemVersion()}, f);
}
template <size_t Length, typename Tag>
bool TokenCache<Length, Tag>::TryGet(TimeSpan* pOutExpiration, char* buffer, size_t bufferSize, uint64_t clientId) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(bufferSize >= Length + 1);

    TimeSpan expiration;
    auto f = [&expiration, buffer, bufferSize](const TokenType& t) {
        expiration = t.expiration;
        util::Strlcpy(buffer, t.data, static_cast<int>(bufferSize));
    };

    if (!( true
        && Base::TryGet({clientId, GetSystemVersion()}, f)
        && !IsAlmostExpired(expiration)))
    {
        return false;
    }
    *pOutExpiration = expiration;
    return true;
}
template <size_t Length, typename Tag>
bool TokenCache<Length, Tag>::IsAvailable(uint64_t clientId) const NN_NOEXCEPT
{
    TimeSpan expiration;
    auto f = [&expiration](const TokenType& t) {
        expiration = t.expiration;
    };

    if (!(true
        && Base::TryGet({clientId, GetSystemVersion()}, f)
        && !IsAlmostExpired(expiration)))
    {
        return false;
    }
    return true;
}

void ErrorCache::Put(uint64_t clientId, Result result) NN_NOEXCEPT
{
    NN_SDK_ASSERT(result.IsFailure());
    Base::Put({clientId, GetSystemVersion()}, result);
}
Result ErrorCache::Check(uint64_t clientId) const NN_NOEXCEPT
{
    Result result;
    if (!Base::TryGet(&result, {clientId, GetSystemVersion()}))
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(result);
}

ServiceImpl::ServiceImpl(MemoryResource& memoryResource, util::AbstractExecutor& executor) NN_NOEXCEPT
    : m_MemoryResource(memoryResource)
    , m_Executor(executor)
{
}
Result ServiceImpl::EnsureAuthenticationTokenCacheAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outAsync, uint64_t clientId, bool refresh) NN_NOEXCEPT
{
    auto p = sf::CreateSharedObjectEmplaced<IAsyncResult, AsyncResultImpl<AuthenticationTask<DeviceAuthenticationTokenCache>>>(
        &m_MemoryResource,
        detail::NdasDriver::AcquireDeviceAuthenticationToken, clientId, refresh, m_DeviceAuthenticationTokenCache, m_ErrorCache);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfMemoryForServiceObject());
    NN_RESULT_DO(p.GetImpl().Initialize(m_Executor));
    *outAsync = std::move(p);
    NN_RESULT_SUCCESS;
}
Result ServiceImpl::LoadAuthenticationTokenCache(sf::Out<int64_t> outExpiration, sf::OutBuffer buffer, uint64_t clientId) NN_NOEXCEPT
{
    TimeSpan expiration;
    if (m_DeviceAuthenticationTokenCache.TryGet(&expiration, buffer.GetPointerUnsafe(), buffer.GetSize(), clientId))
    {
        *outExpiration = expiration.GetSeconds();
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(ResultDeviceAuthenticationTokenCacheUnavailable());
}
Result ServiceImpl::EnsureEdgeTokenCacheAsync(sf::Out<sf::SharedPointer<IAsyncResult>> outAsync, uint64_t clientId) NN_NOEXCEPT
{
    auto p = sf::CreateSharedObjectEmplaced<IAsyncResult, AsyncResultImpl<AuthenticationTask<EdgeTokenCache>>>(
        &m_MemoryResource,
        detail::NdasDriver::AcquireEdgeToken, clientId, false, m_EdgeTokenCache, m_ErrorCache);
    NN_RESULT_THROW_UNLESS(p, ResultOutOfMemoryForServiceObject());
    NN_RESULT_DO(p.GetImpl().Initialize(m_Executor));
    *outAsync = std::move(p);
    NN_RESULT_SUCCESS;
}
Result ServiceImpl::LoadEdgeTokenCache(sf::Out<int64_t> outExpiration, sf::OutBuffer buffer, uint64_t clientId) NN_NOEXCEPT
{
    TimeSpan expiration;
    if (m_EdgeTokenCache.TryGet(&expiration, buffer.GetPointerUnsafe(), buffer.GetSize(), clientId))
    {
        *outExpiration = expiration.GetSeconds();
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW(ResultEdgeTokenCacheUnavailable());
}

}}} // ~namespace nn::dauth::detail
