﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>
#include <array>

#include <nn/nn_Result.h>
#include <nn/nn_Common.h>
#include <nn/dd.h>
#include <nn/os/os_Mutex.h>
#include <nn/codec/codec_OpusCommon.h>
#include <nn/codec/detail/codec_HardwareOpusCommon-spec.NX.h>

#include "codec_DeviceBuffer-spec.NX.h"

#include "../dsp/codec_UtilMailbox.h"
#include "../dsp/codec_SharedResource.h"

namespace nn {
namespace codec {
namespace detail {

class HardwareOpusDecoderImpl;

class HardwareOpusDecoderManagerImpl
{
public:
    static const std::size_t DecoderCountMax = HardwareOpusDecoderIpcConstants_SessionCountMax;

    // プロセス内部で呼び出される初期化/終了処理
    static nn::Result InitializeInternal() NN_NOEXCEPT;
    static nn::Result FinalizeInternal() NN_NOEXCEPT;
    static nn::Result SleepInternal() NN_NOEXCEPT;
    static nn::Result WakeInternal() NN_NOEXCEPT;
    static nn::Result InitializeAdspInternal() NN_NOEXCEPT;
    static nn::Result FinalizeAdspInternal() NN_NOEXCEPT;

    // sf を通じてサービスとして直接的/間接的に呼び出される関数
    static size_t GetWorkBufferSize(int channelCount) NN_NOEXCEPT;
    static nn::Result Map(uint32_t* pOutDeviceAddress, const void* virtualAddress, size_t size) NN_NOEXCEPT;
    static nn::Result Unmap(const void* virtualAddress, size_t size, uint32_t deviceAddress) NN_NOEXCEPT;
    static uint32_t MapMemoryOnSmmu(const void* virtualAddress, size_t size) NN_NOEXCEPT;
    static void UnmapMemoryFromSmmu(const void* virtualAddress, size_t size, uint32_t deviceAddress) NN_NOEXCEPT;
    static nn::Result MapMemoryOnAdsp(uint32_t deviceAddress, size_t size) NN_NOEXCEPT;
    static nn::Result UnmapMemoryFromAdsp(uint32_t deviceAddress, size_t size) NN_NOEXCEPT;

    static nn::Result Initialize(
        int sampleRate,
        int channelCount,
        uintptr_t workBufferAddress,
        size_t workBufferSize) NN_NOEXCEPT;

    static nn::Result Finalize(
        uintptr_t workBufferAddress,
        size_t workBufferSize) NN_NOEXCEPT;

    static nn::Result DecodeInterleaved(
        int* pOutputSampleCount,
        int16_t* outputBuffer,
        size_t outputBufferLength,
        int channelCount,
        const uint8_t* inputBuffer,
        size_t inputBufferSize,
        uintptr_t workBufferAddress,
        nn::TimeSpan* pPerf,
        bool isResetRequested) NN_NOEXCEPT;

    static size_t GetWorkBufferSizeForMultiStream(int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT;

    static nn::Result InitializeForMultiStream(
        int sampleRate,
        int channelCount,
        int totalStreamCount,
        int stereoStreamCount,
        const uint8_t channelMapping[],
        uintptr_t workBufferAddress,
        size_t workBufferSize) NN_NOEXCEPT;

    static nn::Result FinalizeForMultiStream(
        uintptr_t workBufferAddress,
        size_t workBufferSize) NN_NOEXCEPT;

    static nn::Result DecodeInterleavedForMultiStream(
        int* pOutputSampleCount,
        int16_t* outputBuffer,
        size_t outputBufferLength,
        int channelCount,
        const uint8_t* inputBuffer,
        size_t inputBufferSize,
        uintptr_t workBufferAddress,
        nn::TimeSpan* pPerf,
        bool isResetRequested) NN_NOEXCEPT;

    static bool RegisterDecoder(const HardwareOpusDecoderImpl* pDecoder) NN_NOEXCEPT;
    static void UnregisterDecoder(const HardwareOpusDecoderImpl* pDecoder) NN_NOEXCEPT;

    static inline bool IsInitialized() NN_NOEXCEPT
    {
        return m_IsInitialized;
    }
    static inline bool IsAdspInitialized() NN_NOEXCEPT
    {
        return m_IsAdspInitialized;
    }

private:

    static nn::Result GetWorkBufferSizeInternal(size_t* pSize, int channelCount) NN_NOEXCEPT;
    static nn::Result GetWorkBufferSizeForMultiStreamInternal(size_t* pSize, int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT;
    static nn::Result WaitForMessage(uint32_t* pMessage, int64_t intervalInMilli) NN_NOEXCEPT;
    static nn::Result MailboxErrorHanderOfReceiving(int errorCode) NN_NOEXCEPT;
    static nn::Result TryingSendMessage(uint32_t msg, int64_t intervalInMilli) NN_NOEXCEPT;
    static nn::Result MailboxErrorHanderOfSending(int errorCode) NN_NOEXCEPT;
    static nn::os::Mutex m_Mutex;
    static dsp::MboxWrapper m_Mailbox;
    static std::array<HardwareOpusDecoderImpl*, DecoderCountMax> m_DecoderList;
    static DeviceBuffer m_SharedMemory;
    static size_t m_RequiredWorkBufferSize[2];
    static uint m_ErrorCountOfSending;
    static uint m_ErrorCountOfReceiving;
    static bool m_IsInitialized;
    static bool m_IsAdspInitialized;
};

}  // namespace detail
}  // namespace codec
}  // namespace nn
