﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/result/result_HandlingUtility.h>
#include <nn/result/result_ErrorResult.h>
#include <nn/codec/codec_Result.private.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>
#include <nn/codec/detail/codec_Log.h>

#include <nne/audio/audio_Iova.h>

#include "codec_HardwareOpusDecoderImpl-spec.NX.h"
#include "codec_HardwareOpusDecoderManagerImpl-spec.NX.h"

namespace nn {
namespace codec {
namespace detail {

HardwareOpusDecoderImpl::HardwareOpusDecoderImpl() NN_NOEXCEPT
    : m_WorkBufferVirtualAddress(nullptr)
    , m_WorkBufferSize(0)
    , m_WorkBufferDeviceAddress(0)
    , m_SampleRate(0)
    , m_ChannelCount(0)
    , m_IsMemoryMappedOnAdsp(false)
    , m_IsInitialized(false)
{
}

HardwareOpusDecoderImpl::~HardwareOpusDecoderImpl() NN_NOEXCEPT
{
    m_IsInitialized = false;
}

nn::Result HardwareOpusDecoderImpl::Initialize(
    int sampleRate,
    int channelCount,
    void* workBufferAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !IsInitialized() );

    // セッションを登録する。
    // ToDo: 適切な result を使用する (ExcessiveSessionCount() とか)。
    if (false == HardwareOpusDecoderManagerImpl::RegisterDecoder(this))
    {
        NN_RESULT_THROW(ResultCpuOutOfResource());
    }

    NN_RESULT_DO(
        Map(sampleRate,
            channelCount,
            1,
            (channelCount == 1) ? 0 : 1,
            workBufferAddress,
            workBufferSize)
    );

    auto result = HardwareOpusDecoderManagerImpl::Initialize(
        sampleRate,
        channelCount,
        m_WorkBufferDeviceAddress, // 先頭を渡して、一度でマップする
        m_WorkBufferSize); // 全サイズを渡して、一度でマップする

    // 初期化に失敗した場合は継続不能
    // Map した領域を Unmap してエラーを result を返す。
    if ( result.IsFailure() )
    {
        // ToDo: ABORT を使わないようにできる？
        NN_ABORT_UNLESS_RESULT_SUCCESS(Unmap());
        HardwareOpusDecoderManagerImpl::UnregisterDecoder(this);
        NN_RESULT_THROW(result);
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderImpl::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );
    if (!IsMemoryMapped())
    {
        NN_RESULT_DO(Remap());
    }
    m_IsInitialized = false;
    auto result = HardwareOpusDecoderManagerImpl::Finalize(
        m_WorkBufferDeviceAddress,
        m_WorkBufferSize);
    HardwareOpusDecoderManagerImpl::UnregisterDecoder(this);
    HardwareOpusDecoderManagerImpl::UnmapMemoryFromSmmu(
        m_WorkBufferVirtualAddress,
        m_WorkBufferSize,
        m_WorkBufferDeviceAddress);
    return result;
}

nn::Result HardwareOpusDecoderImpl::Sleep() NN_NOEXCEPT
{
    return UnmapMemoryFromAdsp();
}

nn::Result HardwareOpusDecoderImpl::Wake() NN_NOEXCEPT
{
    // audio プロセスの Wake() 処理にかかる時間削減のため、ここでは Remap() しない。
    // return Remap();
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderImpl::DecodeInterleaved(
    int* pConsumedPacketSize,
    int* pOutputSampleCount,
    int16_t* outputBuffer,
    size_t outputBufferSize,
    const uint8_t* inputBuffer,
    size_t inputBufferSize,
    nn::TimeSpan* pPerf,
    bool isResetRequested) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );
    NN_SDK_ASSERT( m_InputBuffer.IsAvailable() );
    NN_SDK_ASSERT( m_OutputBuffer.IsAvailable() );

    NN_RESULT_THROW_UNLESS(inputBufferSize > OpusPacketInternal::HeaderSize, ResultOpusUnexpectedError());

    // Opus パケットのヘッダとフレームを分ける。
    // ToDo:
    //   どの層までヘッダを付けておくべきか、要検討。
    //     1. opus フレームが、20 ms より大きなサンプル数を持てるようにする。
    //     2. opus フレームは最大 20 ms のサンプルを持つことができ、複数フレームをデコーダに渡せるようにする。
    //   1 と 2 は排他的であるが、ヘッダを取り除いた後では 2 はできなくなる。
    OpusPacketInternal::Header header;
    OpusPacketInternal::GetHeaderFromPacket(&header, inputBuffer);
    const size_t packetSize = header.packetSize;

    // ToDo:
    //   正しいエラーコードを追加し、修正する (ResultSmallerInputBuffer)
    NN_RESULT_THROW_UNLESS(m_InputBuffer.GetSize() >= packetSize, ResultOpusBufferTooSmall());

    const size_t consumedSize = packetSize + OpusPacketInternal::HeaderSize;
    if ( consumedSize > inputBufferSize )
    {
        return ResultOpusBufferTooSmall();
    }

    if (!IsMemoryMapped())
    {
        NN_RESULT_DO(Remap());
    }

    std::memcpy(m_InputBuffer, inputBuffer + OpusPacketInternal::HeaderSize, packetSize);
    nn::dd::StoreDataCache(m_InputBuffer, packetSize);
    auto result = HardwareOpusDecoderManagerImpl::DecodeInterleaved(
        pOutputSampleCount,
        reinterpret_cast<int16_t*>(m_OutputBuffer.GetDeviceAddress()),
        m_OutputBuffer.GetSize(),
        m_ChannelCount,
        reinterpret_cast<const uint8_t*>(m_InputBuffer.GetDeviceAddress()),
        packetSize,
        m_WorkBufferDeviceAddress,
        pPerf,
        isResetRequested);

    NN_RESULT_DO(result);

    // 結果領域を無効化
    nn::dd::InvalidateDataCache(m_OutputBuffer, m_OutputBuffer.GetSize());
#if 0
    // データダンプ
    {
        const uint16_t* buffer = reinterpret_cast<const uint16_t*>(m_OutputBuffer.GetVirtualAddress());
        NN_UNUSED(buffer);
        for (int total = 0; total < outputSampleCount * channelCount; total += 8)
        {
            NN_DETAIL_CODEC_TRACE("[HostCpu] ");
            for ( int i = 0; i < 8; ++i )
            {
                NN_DETAIL_CODEC_TRACE(" %04x", buffer[i]);
            }
            NN_DETAIL_CODEC_TRACE("\n");
            buffer += 8;
        }
    }
#endif
    // ResultOpusUnexpectedError() としているが、ResultOpusInsufficientPcmBuffer() などを定義して返すべき (TBD)。
    const size_t copySize =  *pOutputSampleCount * sizeof(int16_t) * m_ChannelCount;
    std::memcpy(outputBuffer, m_OutputBuffer, copySize);
    *pConsumedPacketSize = consumedSize;

    NN_RESULT_SUCCESS;

}

nn::Result HardwareOpusDecoderImpl::SetContext(const uint8_t*address, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );

    if (!IsMemoryMapped())
    {
        NN_RESULT_DO(Remap());
    }
    // 実装する (TBD)
    NN_UNUSED(address);
    NN_UNUSED(size);
    // ここまで

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderImpl::InitializeForMultiStream(
    int sampleRate,
    int channelCount,
    int totalStreamCount,
    int stereoStreamCount,
    const uint8_t channelMapping[],
    void* workBufferAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !IsInitialized() );

    // セッションを登録する。
    // ToDo: 適切な result を使用する (ExcessiveSessionCount() とか)。
    if (false == HardwareOpusDecoderManagerImpl::RegisterDecoder(this))
    {
        NN_RESULT_THROW(ResultCpuOutOfResource());
    }

    NN_RESULT_DO(
        Map(sampleRate,
            channelCount,
            totalStreamCount,
            stereoStreamCount,
            workBufferAddress,
            workBufferSize)
    );

    auto result = HardwareOpusDecoderManagerImpl::InitializeForMultiStream(
        sampleRate,
        channelCount,
        totalStreamCount,
        stereoStreamCount,
        channelMapping,
        m_WorkBufferDeviceAddress, // 先頭を渡して、一度でマップする
        m_WorkBufferSize); // 全サイズを渡して、一度でマップする

    // 初期化に失敗した場合は継続不能
    // Map した領域を Unmap してエラーを result を返す。
    if ( result.IsFailure() )
    {
        // ToDo: ABORT を使わないようにできる？
        NN_ABORT_UNLESS_RESULT_SUCCESS(Unmap());
        HardwareOpusDecoderManagerImpl::UnregisterDecoder(this);
        NN_RESULT_THROW(result);
    }
    m_SampleRate = sampleRate;
    m_ChannelCount = channelCount;
    m_TotalStreamCount = totalStreamCount;
    m_StereoStreamCount = stereoStreamCount;
    m_IsInitialized = true;

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderImpl::FinalizeForMultiStream() NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );
    if (!IsMemoryMapped())
    {
        NN_RESULT_DO(Remap());
    }
    m_IsInitialized = false;
    auto result = HardwareOpusDecoderManagerImpl::FinalizeForMultiStream(
        m_WorkBufferDeviceAddress,
        m_WorkBufferSize);
    HardwareOpusDecoderManagerImpl::UnregisterDecoder(this);
    HardwareOpusDecoderManagerImpl::UnmapMemoryFromSmmu(
        m_WorkBufferVirtualAddress,
        m_WorkBufferSize,
        m_WorkBufferDeviceAddress);
    return result;
}

nn::Result HardwareOpusDecoderImpl::DecodeInterleavedForMultiStream(
    int* pConsumedPacketSize,
    int* pOutputSampleCount,
    int16_t* outputBuffer,
    size_t outputBufferSize,
    const uint8_t* inputBuffer,
    size_t inputBufferSize,
    nn::TimeSpan* pPerf,
    bool isResetRequested) NN_NOEXCEPT
{
    NN_SDK_ASSERT( IsInitialized() );
    NN_SDK_ASSERT( m_InputBuffer.IsAvailable() );
    NN_SDK_ASSERT( m_OutputBuffer.IsAvailable() );

    NN_RESULT_THROW_UNLESS(inputBufferSize > OpusPacketInternal::HeaderSize, ResultOpusUnexpectedError());

    // Opus パケットのヘッダとフレームを分ける。
    // ToDo:
    //   どの層までヘッダを付けておくべきか、要検討。
    //     1. opus フレームが、20 ms より大きなサンプル数を持てるようにする。
    //     2. opus フレームは最大 20 ms のサンプルを持つことができ、複数フレームをデコーダに渡せるようにする。
    //   1 と 2 は排他的であるが、ヘッダを取り除いた後では 2 はできなくなる。
    OpusPacketInternal::Header header;
    OpusPacketInternal::GetHeaderFromPacket(&header, inputBuffer);
    const size_t packetSize = header.packetSize;

    // ToDo:
    //   正しいエラーコードを追加し、修正する (ResultSmallerInputBuffer)
    NN_RESULT_THROW_UNLESS(m_InputBuffer.GetSize() >= packetSize, ResultOpusBufferTooSmall());

    const size_t consumedSize = packetSize + OpusPacketInternal::HeaderSize;
    if ( consumedSize > inputBufferSize )
    {
        return ResultOpusBufferTooSmall();
    }

    if (!IsMemoryMapped())
    {
        NN_RESULT_DO(Remap());
    }

    std::memcpy(m_InputBuffer, inputBuffer + OpusPacketInternal::HeaderSize, packetSize);
    nn::dd::StoreDataCache(m_InputBuffer, packetSize);
    auto result = HardwareOpusDecoderManagerImpl::DecodeInterleavedForMultiStream(
        pOutputSampleCount,
        reinterpret_cast<int16_t*>(m_OutputBuffer.GetDeviceAddress()),
        m_OutputBuffer.GetSize(),
        m_ChannelCount,
        reinterpret_cast<const uint8_t*>(m_InputBuffer.GetDeviceAddress()),
        packetSize,
        m_WorkBufferDeviceAddress,
        pPerf,
        isResetRequested);

    NN_RESULT_DO(result);

    // 結果領域を無効化
    nn::dd::InvalidateDataCache(m_OutputBuffer, m_OutputBuffer.GetSize());
#if 0
    // データダンプ
    {
        const uint16_t* buffer = reinterpret_cast<const uint16_t*>(m_OutputBuffer.GetVirtualAddress());
        NN_UNUSED(buffer);
        for (int total = 0; total < outputSampleCount * channelCount; total += 8)
        {
            NN_DETAIL_CODEC_TRACE("[HostCpu] ");
            for ( int i = 0; i < 8; ++i )
            {
                NN_DETAIL_CODEC_TRACE(" %04x", buffer[i]);
            }
            NN_DETAIL_CODEC_TRACE("\n");
            buffer += 8;
        }
    }
#endif
    // ResultOpusUnexpectedError() としているが、ResultOpusInsufficientPcmBuffer() などを定義して返すべき (TBD)。
    const size_t copySize =  *pOutputSampleCount * sizeof(int16_t) * m_ChannelCount;
    std::memcpy(outputBuffer, m_OutputBuffer, copySize);
    *pConsumedPacketSize = consumedSize;

    NN_RESULT_SUCCESS;

}
nn::Result HardwareOpusDecoderImpl::Map(
    int sampleRate,
    int channelCount,
    int totalStreamCount,
    int stereoStreamCount,
    void* workBufferVirtualAddress,
    size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT( !IsInitialized() );

    NN_SDK_REQUIRES(nn::util::is_aligned(workBufferSize, nn::dd::DeviceAddressSpaceMemoryRegionAlignment));
    NN_SDK_REQUIRES(nn::util::is_aligned(reinterpret_cast<uintptr_t>(workBufferVirtualAddress), nn::dd::DeviceAddressSpaceMemoryRegionAlignment));

    const uint32_t workBufferDeviceAddress =
        HardwareOpusDecoderManagerImpl::MapMemoryOnSmmu(workBufferVirtualAddress, workBufferSize);

    nn::dd::InvalidateDataCache(workBufferVirtualAddress, workBufferSize);

    auto virtualAddress = static_cast<uint8_t*>(workBufferVirtualAddress) + workBufferSize; // 末尾から割り当てる
    auto deviceAddress = workBufferDeviceAddress + workBufferSize; // 末尾から割り当てる
    auto size = workBufferSize;

    // | OpusDecoeer ワークバッファ | 出力バッファ | 入力バッファ | と並べる。
    {
        const int bufferSize = GetOutputBufferSize(sampleRate, channelCount);
        virtualAddress -= bufferSize;
        deviceAddress -= bufferSize;
        size -= bufferSize;

        NN_SDK_ASSERT_ALIGNED(bufferSize, dsp::OneCacheLineExpected);
        NN_SDK_ASSERT_ALIGNED(virtualAddress, dsp::OneCacheLineExpected);
        NN_SDK_ASSERT_ALIGNED(deviceAddress, dsp::OneCacheLineExpected);

        m_OutputBuffer.SetParameters(virtualAddress, deviceAddress, bufferSize);
    }
    {
        const int bufferSize = GetInputBufferSize(sampleRate, totalStreamCount, stereoStreamCount);
        virtualAddress -= bufferSize;
        deviceAddress -= bufferSize;
        size -= bufferSize;

        NN_SDK_ASSERT_ALIGNED(bufferSize, dsp::OneCacheLineExpected);
        NN_SDK_ASSERT_ALIGNED(virtualAddress, dsp::OneCacheLineExpected);
        NN_SDK_ASSERT_ALIGNED(deviceAddress, dsp::OneCacheLineExpected);

        m_InputBuffer.SetParameters(virtualAddress, deviceAddress, bufferSize);
    }

    NN_SDK_ASSERT_ALIGNED(workBufferSize, dsp::OneCacheLineExpected);
    NN_SDK_ASSERT_ALIGNED(virtualAddress, dsp::OneCacheLineExpected);
    NN_SDK_ASSERT_ALIGNED(deviceAddress, dsp::OneCacheLineExpected);

    m_WorkBufferSize = workBufferSize;
    m_WorkBufferVirtualAddress = workBufferVirtualAddress;
    m_WorkBufferDeviceAddress = workBufferDeviceAddress;

    m_IsMemoryMappedOnAdsp = true;

    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderImpl::Remap() NN_NOEXCEPT
{
    return MapMemoryFromAdsp();
}

nn::Result HardwareOpusDecoderImpl::Unmap() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_WorkBufferVirtualAddress != nullptr);
    NN_SDK_ASSERT(m_WorkBufferSize > 0);
    NN_SDK_ASSERT(m_WorkBufferDeviceAddress != 0);
    NN_SDK_ASSERT(IsMemoryMapped());
    auto result = UnmapMemoryFromAdsp();
    HardwareOpusDecoderManagerImpl::UnmapMemoryFromSmmu(
        m_WorkBufferVirtualAddress,
        m_WorkBufferSize,
        m_WorkBufferDeviceAddress);
    return result;
}

// Initialize()/Finalize() の間で使う
nn::Result HardwareOpusDecoderImpl::MapMemoryFromAdsp() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_WorkBufferSize > 0);
    NN_SDK_ASSERT(m_WorkBufferDeviceAddress != 0);
    if (IsMemoryMapped())
    {
        NN_RESULT_SUCCESS;
    }
    const auto result = HardwareOpusDecoderManagerImpl::MapMemoryOnAdsp(
        m_WorkBufferDeviceAddress,
        m_WorkBufferSize);
    m_IsMemoryMappedOnAdsp = true;
    return result;
}

// Initialize()/Finalize() の間で使う
nn::Result HardwareOpusDecoderImpl::UnmapMemoryFromAdsp() NN_NOEXCEPT
{
    NN_SDK_ASSERT(IsInitialized());
    NN_SDK_ASSERT(m_WorkBufferSize > 0);
    NN_SDK_ASSERT(m_WorkBufferDeviceAddress != 0);
    if (!IsMemoryMapped())
    {
        NN_RESULT_SUCCESS;
    }
    m_IsMemoryMappedOnAdsp = false;
    return HardwareOpusDecoderManagerImpl::UnmapMemoryFromAdsp(
        m_WorkBufferDeviceAddress,
        m_WorkBufferSize);
}

}  // namespace detail
}  // namespace codec
}  // namespace nn
