﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new> // replacement new

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/codec/detail/codec_HardwarePlatform.h>

#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/dd/dd_DeviceAddressSpace.h>

#include <nn/codec/codec_Result.private.h>
#include <nn/codec/detail/codec_IHardwareOpusDecoderManager.h>

#include "codec_CreateHardwareOpusDecoderManager.h"

#else  // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
#include <nn/codec/codec_OpusMultiStreamDecoder.h>
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

#include <nn/codec/codec_OpusPacket.h>
#include <nn/codec/codec_HardwareOpusMultiStreamDecoder.h>
#include <nn/codec/detail/codec_OpusPacketInternal.h>

namespace nn { namespace codec {

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
NN_STATIC_ASSERT(sizeof(HardwareOpusMultiStreamDecoderType) >= sizeof(OpusMultiStreamDecoderType));
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

#if defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
namespace {

    nn::sf::SharedPointer<detail::IHardwareOpusDecoderManager> CreateHardwareOpusDecoderManager() NN_NOEXCEPT
    {
        //return CreateHardwareOpusDecoderManagerByDfc();
        return CreateHardwareOpusDecoderManagerByHipc();
    }

} // anonymous
#endif // defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

HardwareOpusMultiStreamDecoder::HardwareOpusMultiStreamDecoder(int32_t option) NN_NOEXCEPT
    : m_Decoder()
    , m_Option(option)
{
}

HardwareOpusMultiStreamDecoder::~HardwareOpusMultiStreamDecoder() NN_NOEXCEPT
{
}

OpusResult HardwareOpusMultiStreamDecoder::Initialize(int sampleRate, int channelCount, int totalStreamCount, int stereoStreamCount, const uint8_t channelMapping[], void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    return InitializeHardwareOpusMultiStreamDecoder(&m_Decoder, sampleRate, channelCount, totalStreamCount, stereoStreamCount, channelMapping, buffer, size, m_Option);
}

void HardwareOpusMultiStreamDecoder::Finalize() NN_NOEXCEPT
{
    return FinalizeHardwareOpusMultiStreamDecoder(&m_Decoder);
}

size_t GetHardwareOpusMultiStreamDecoderWorkBufferSize(int sampleRate, int channelCount, int totalStreamCount, int stereoStreamCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(channelCount >= 1 && channelCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(totalStreamCount + stereoStreamCount <= channelCount);
    NN_SDK_REQUIRES(totalStreamCount >= 1 && totalStreamCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(stereoStreamCount >= 0 && stereoStreamCount <= totalStreamCount);
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusMultiStreamDecoderWorkBufferSize(sampleRate, channelCount, totalStreamCount, stereoStreamCount);
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    auto manager = CreateHardwareOpusDecoderManager();
    int size;
    detail::HardwareOpusMultiStreamDecoderParameterInternal parameter =
    {
        sampleRate,
        channelCount,
        totalStreamCount,
        stereoStreamCount,
        { 0 },
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(manager->GetWorkBufferSizeForMultiStream(&size, parameter));
    return static_cast<size_t>(nn::util::align_up(size, nn::os::MemoryPageSize));
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

OpusResult InitializeHardwareOpusMultiStreamDecoder(HardwareOpusMultiStreamDecoderType* pOutDecoder, int sampleRate, int channelCount, int totalStreamCount, int stereoStreamCount, const uint8_t channelMapping[], void* buffer, size_t size, int32_t option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDecoder);
    NN_SDK_REQUIRES(sampleRate == 48000 || sampleRate == 24000 || sampleRate == 16000 || sampleRate == 12000 || sampleRate == 8000);
    NN_SDK_REQUIRES(totalStreamCount >= 1);
    NN_SDK_REQUIRES(totalStreamCount + stereoStreamCount <= OpusStreamCountMax);
    NN_SDK_REQUIRES(stereoStreamCount >= 0 && stereoStreamCount <= totalStreamCount);
    NN_SDK_REQUIRES_NOT_NULL(channelMapping);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(size >= GetHardwareOpusMultiStreamDecoderWorkBufferSize(sampleRate, channelCount, totalStreamCount, stereoStreamCount));
    NN_UNUSED(option);
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    OpusMultiStreamDecoderType* pDecoder = reinterpret_cast<OpusMultiStreamDecoderType*>(pOutDecoder);
    auto result = InitializeOpusMultiStreamDecoder(pDecoder, sampleRate, channelCount, totalStreamCount, stereoStreamCount, channelMapping, buffer, size);
    if (result != OpusResult_Success)
    {
        return result;
    }
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES(nn::util::is_aligned(reinterpret_cast<uintptr_t>(buffer), nn::os::MemoryPageSize));

    auto manager = CreateHardwareOpusDecoderManager();
    detail::HardwareOpusMultiStreamDecoderParameterInternal parameter =
    {
        sampleRate,
        channelCount,
        totalStreamCount,
        stereoStreamCount,
        { 0 },
    };
    for (int i = 0; i < channelCount; ++i)
    {
        parameter.channelMapping[i] = channelMapping[i];
    }

    nn::os::TransferMemory transferMemory(buffer, size, nn::os::MemoryPermission_None);
    nn::sf::SharedPointer<detail::IHardwareOpusDecoder> decoder;
    auto result = manager->OpenHardwareOpusDecoderForMultiStream(&decoder, parameter, nn::sf::NativeHandle(transferMemory.Detach(), true), size);

    // 適切なステータスを返す必要がある。
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultCpuOutOfResource)
        {
            return OpusResult_OutOfResource;
        }
        // 上記と Success 以外の result は、ここまで返ってこずに abort するはず。
        // NN_RESULT_CATCH_ALL
        // {
        // }
    NN_RESULT_END_TRY

    pOutDecoder->_handle = decoder.Detach();
    pOutDecoder->_buffer = buffer;
    pOutDecoder->_size = size;
    pOutDecoder->_sampleRate = sampleRate;
    pOutDecoder->_channelCount = channelCount;
    pOutDecoder->_totalStreamCount = totalStreamCount;
    pOutDecoder->_stereoStreamCount = stereoStreamCount;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)

    pOutDecoder->_perf = 0;
    pOutDecoder->_isResetRequested = false;

    return OpusResult_Success;
}

void FinalizeHardwareOpusMultiStreamDecoder(HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    FinalizeOpusMultiStreamDecoder(reinterpret_cast<OpusMultiStreamDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    nn::sf::ReleaseSharedObject(static_cast<detail::IHardwareOpusDecoder*>(pDecoder->_handle));
    pDecoder->_handle = nullptr;
    pDecoder->_buffer = nullptr;
    pDecoder->_size = 0;
    pDecoder->_sampleRate = 0;
    pDecoder->_channelCount = 0;
    pDecoder->_totalStreamCount = 0;
    pDecoder->_stereoStreamCount = 0;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

int GetHardwareOpusMultiStreamDecoderSampleRate(const HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusMultiStreamDecoderSampleRate(reinterpret_cast<const OpusMultiStreamDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_sampleRate;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

int GetHardwareOpusMultiStreamDecoderChannelCount(const HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusMultiStreamDecoderChannelCount(reinterpret_cast<const OpusMultiStreamDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_channelCount;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

int GetHardwareOpusMultiStreamDecoderTotalStreamCount(const HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusMultiStreamDecoderTotalStreamCount(reinterpret_cast<const OpusMultiStreamDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_totalStreamCount;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

int GetHardwareOpusMultiStreamDecoderStereoStreamCount(const HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return GetOpusMultiStreamDecoderStereoStreamCount(reinterpret_cast<const OpusMultiStreamDecoderType*>(pDecoder));
#else  // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return pDecoder->_stereoStreamCount;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

OpusResult DecodeOpusMultiStreamInterleavedWithHardware(HardwareOpusMultiStreamDecoderType* pDecoder, size_t* pOutConsumed, int* pOutSampleCount, int16_t* outputBuffer, size_t outputSize, const void* inputBuffer, size_t inputSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    NN_SDK_REQUIRES_NOT_NULL(pOutConsumed);
    NN_SDK_REQUIRES_NOT_NULL(pOutSampleCount);
    NN_SDK_REQUIRES_NOT_NULL(outputBuffer);
    NN_SDK_REQUIRES_NOT_NULL(inputBuffer);

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    return DecodeOpusMultiStreamInterleaved(reinterpret_cast<OpusMultiStreamDecoderType*>(pDecoder), pOutConsumed, pOutSampleCount, outputBuffer, outputSize, inputBuffer, inputSize);
#else // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    nn::codec::detail::OpusPacketInternal::Header header;
    const size_t frameHeaderSize = sizeof(header);
    if (inputSize < frameHeaderSize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    auto packet = static_cast<const uint8_t*>(inputBuffer);
    nn::codec::detail::OpusPacketInternal::GetHeaderFromPacket(&header, packet);
    const auto frameBodySize = header.packetSize;
    if (inputSize < frameHeaderSize + frameBodySize)
    {
        return OpusResult_InsufficientOpusBuffer;
    }
    int sampleCount = 0;
    const auto opusResult = GetOpusPacketSampleCountInPacket(&sampleCount, packet, inputSize, GetHardwareOpusMultiStreamDecoderSampleRate(pDecoder));
    if (OpusResult_Success != opusResult)
    {
        return opusResult;
    }
    if (outputSize < sampleCount * GetHardwareOpusMultiStreamDecoderChannelCount(pDecoder) * sizeof(int16_t))
    {
        return OpusResult_InsufficientPcmBuffer;
    }

    nn::sf::InBuffer inBuffer(static_cast<const char*>(inputBuffer), frameBodySize + frameHeaderSize);
    nn::sf::OutBuffer outBuffer(reinterpret_cast<char*>(outputBuffer), outputSize);

    const auto isResetRequested = pDecoder->_isResetRequested;
    pDecoder->_isResetRequested = false;

    int consumed = 0;
    int outputSampleCount = 0;
    int64_t perf = 0;
    auto result = static_cast<detail::IHardwareOpusDecoder*>(pDecoder->_handle)->DecodeInterleavedForMultiStream(&consumed, &perf, &outputSampleCount, outBuffer, inBuffer, isResetRequested);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(ResultOpusInvalidPacket)
        {
            return OpusResult_BrokenData;
        }
#if 0
        // Result が定義されたら有効にする (TBD)
        NN_RESULT_CATCH(ResultOpusInsufficientPcmBuffer)
        {
            return OpusResult_InsufficientPcmBuffer;
        }
#endif
        // ユーザの管理外なので、ユーザはこのステートを受け取っても何もできない。
        // NN_RESULT_CATCH(ResultOpusInvalidState)
        // {
        //     return OpusResult_InvalidWorkBuffer;
        // }
        // 上記と Success 以外の result は、ここまで返ってこずに abort するはず。
        // NN_RESULT_CATCH_ALL
        // {
        // }
    NN_RESULT_END_TRY

    *pOutConsumed = consumed;
    *pOutSampleCount = outputSampleCount;
    pDecoder->_perf = perf;

    return OpusResult_Success;
#endif // !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
}

nn::TimeSpan GetHardwareOpusMultiStreamDecoderProcessingTime(const HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);
    return nn::TimeSpan::FromMicroSeconds(pDecoder->_perf);
}

void ResetHardwareOpusMultiStreamDecoder(HardwareOpusMultiStreamDecoderType* pDecoder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDecoder);
    NN_SDK_REQUIRES_NOT_NULL(pDecoder->_handle);

#if !defined(NN_CODEC_HAVE_HARDWARE_ACCELERATOR)
    ResetOpusMultiStreamDecoder(reinterpret_cast<OpusMultiStreamDecoderType*>(pDecoder));
#else
    pDecoder->_isResetRequested = true;
#endif
}

}}  // nn::codec
