﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/codec/codec_Result.private.h>
#include "codec_HardwareOpusDecoderManagerImpl.h"
#include "codec_HardwareOpusDecoderImpl.h"
#include "detail/codec_HardwareOpusDecoderManagerImpl-spec.NX.h"

namespace nn { namespace codec {

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::Initialize(
    const detail::HardwareOpusDecoderParameterInternal& parameter,
    nn::sf::NativeHandle& sfHandle, int workBufferSize) NN_NOEXCEPT
{
    // TODO:
    //   - sampleRate と channelCount の値確認を追加する (TODO)
    //   - workBuffer とそのサイズが有効であるかどうかの確認を追加する (TODO)
    //   - 初期化の確認をおこない、適切な result を返す。
    void* workBufferAddress = nullptr;
    size_t size = workBufferSize & 0xffffffff;
    m_TransferMemory.Attach(size, sfHandle.GetOsHandle(), sfHandle.IsManaged());
    sfHandle.Detach();
    NN_RESULT_DO(m_TransferMemory.Map(&workBufferAddress, nn::os::MemoryPermission_None));
    // 元の所有者がおこなったメモリアクセスを真っ新にする。
    // ワーク領域は CPU 側から操作しないため (RW 問わず)、この措置だけで十分
    nn::dd::InvalidateDataCache(workBufferAddress, workBufferSize);
    m_IsTransferMemoryMapped = true;
    auto result = m_DecoderImpl.Initialize(
            parameter.sampleRate,
            parameter.channelCount,
            workBufferAddress,
            workBufferSize);

    // sampleRate、channelCount、workBuffer の確認が実装されたら、
    // その他のあらゆる失敗で Abort する。
#if 0
    // ResultUserError は返す、そうでなければ Abort する、という I/F にする予定
    NN_RESULT_THROW_UNLESS( !ResultUserError::Includes(result), result );
#endif
    return result;
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::DecodeInterleavedOld(
    nn::sf::Out<int> pOutConsumed,
    nn::sf::Out<int> pOutSampleCount,
    const nn::sf::OutBuffer& outputBuffer,
    const nn::sf::InBuffer& inputBuffer) NN_NOEXCEPT
{
    return DecodeInterleaved(pOutConsumed, nullptr, pOutSampleCount, outputBuffer, inputBuffer, false);
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::DecodeInterleavedWithPerfOld(
    nn::sf::Out<int> pOutConsumed,
    nn::sf::Out<int64_t> pOutPerf,
    nn::sf::Out<int> pOutSampleCount,
    const nn::sf::OutBuffer& outputBuffer,
    const nn::sf::InBuffer& inputBuffer) NN_NOEXCEPT
{
    return DecodeInterleaved(pOutConsumed, pOutPerf, pOutSampleCount, outputBuffer, inputBuffer, false);
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::DecodeInterleaved(
    nn::sf::Out<int> pOutConsumed,
    nn::sf::Out<int64_t> pOutPerf,
    nn::sf::Out<int> pOutSampleCount,
    const nn::sf::OutBuffer& outputBuffer,
    const nn::sf::InBuffer& inputBuffer,
    bool isResetRequested) NN_NOEXCEPT
{
    // TODO:
    //   - 初期化の確認をおこない、適切な result を返す。
    //   - ResultUserError を定義し、使うようにする。
    int consumed;
    int outputSampleCount;
    nn::TimeSpan perf;
    auto result = m_DecoderImpl.DecodeInterleaved(
            &consumed,
            &outputSampleCount,
            reinterpret_cast<int16_t*>(outputBuffer.GetPointerUnsafe()),
            outputBuffer.GetSize(),
            reinterpret_cast<const uint8_t*>(inputBuffer.GetPointerUnsafe()),
            inputBuffer.GetSize(),
            &perf,
            isResetRequested);

    // 失敗した場合、
    //   1. result が ResultUserError に含まれるなら返す。
    //   2. 1 以外なら Abort する。
    // NN_RESULT_THROW_UNLESS( !ResultUserError::Includes(result), result );
    // ResultUserError を定義していないので、暫定的に ResultOpusError を使用する。
    NN_RESULT_THROW_UNLESS( !ResultOpusError::Includes(result), result );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *pOutConsumed = consumed;
    *pOutSampleCount = outputSampleCount;
    if (pOutPerf.GetPointer() != nullptr)
    {
        *pOutPerf = perf.GetMicroSeconds();
    }
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::SetContext(const nn::sf::InBuffer& context) NN_NOEXCEPT
{
    // TODO:
    //   - 初期化の確認をおこない、適切な result を返す。
    //   - ResultUserError を定義し、使うようにする。
    auto result = m_DecoderImpl.SetContext(
        reinterpret_cast<const uint8_t*>(context.GetPointerUnsafe()),
        context.GetSize());
#if 0
    // ResultUserError は返す、そうでなければ Abort する、という I/F にする予定
    NN_RESULT_THROW_UNLESS( !ResultUserError::Includes(result), result );
#endif
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::InitializeForMultiStream(
    const detail::HardwareOpusMultiStreamDecoderParameterInternal& parameter,
    nn::sf::NativeHandle& sfHandle, int workBufferSize) NN_NOEXCEPT
{
    // TODO:
    //   - sampleRate と channelCount の値確認を追加する (TODO)
    //   - workBuffer とそのサイズが有効であるかどうかの確認を追加する (TODO)
    //   - 初期化の確認をおこない、適切な result を返す。
    void* workBufferAddress = nullptr;
    size_t size = workBufferSize & 0xffffffff;
    m_TransferMemory.Attach(size, sfHandle.GetOsHandle(), sfHandle.IsManaged());
    sfHandle.Detach();
    NN_RESULT_DO(m_TransferMemory.Map(&workBufferAddress, nn::os::MemoryPermission_None));
    // 元の所有者がおこなったメモリアクセスを真っ新にする。
    // ワーク領域は CPU 側から操作しないため (RW 問わず)、この措置だけで十分
    nn::dd::InvalidateDataCache(workBufferAddress, workBufferSize);
    m_IsTransferMemoryMapped = true;
    auto result = m_DecoderImpl.InitializeForMultiStream(
            parameter.sampleRate,
            parameter.channelCount,
            parameter.totalStreamCount,
            parameter.stereoStreamCount,
            parameter.channelMapping,
            workBufferAddress,
            workBufferSize);

    // sampleRate、channelCount、workBuffer の確認が実装されたら、
    // その他のあらゆる失敗で Abort する。
#if 0
    // ResultUserError は返す、そうでなければ Abort する、という I/F にする予定
    NN_RESULT_THROW_UNLESS( !ResultUserError::Includes(result), result );
#endif
    return result;
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::DecodeInterleavedForMultiStreamOld(
    nn::sf::Out<int> pOutConsumed,
    nn::sf::Out<int> pOutSampleCount,
    const nn::sf::OutBuffer& outputBuffer,
    const nn::sf::InBuffer& inputBuffer) NN_NOEXCEPT
{
    return DecodeInterleavedForMultiStream(pOutConsumed, nullptr, pOutSampleCount, outputBuffer, inputBuffer, false);
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::DecodeInterleavedForMultiStreamWithPerfOld(
    nn::sf::Out<int> pOutConsumed,
    nn::sf::Out<int64_t> pOutPerf,
    nn::sf::Out<int> pOutSampleCount,
    const nn::sf::OutBuffer& outputBuffer,
    const nn::sf::InBuffer& inputBuffer) NN_NOEXCEPT
{
    return DecodeInterleavedForMultiStream(pOutConsumed, pOutPerf, pOutSampleCount, outputBuffer, inputBuffer, false);
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::DecodeInterleavedForMultiStream(
    nn::sf::Out<int> pOutConsumed,
    nn::sf::Out<int64_t> pOutPerf,
    nn::sf::Out<int> pOutSampleCount,
    const nn::sf::OutBuffer& outputBuffer,
    const nn::sf::InBuffer& inputBuffer,
    bool isResetRequested) NN_NOEXCEPT
{
    // TODO:
    //   - 初期化の確認をおこない、適切な result を返す。
    //   - ResultUserError を定義し、使うようにする。
    int consumed;
    int outputSampleCount;
    nn::TimeSpan perf;
    auto result = m_DecoderImpl.DecodeInterleavedForMultiStream(
            &consumed,
            &outputSampleCount,
            reinterpret_cast<int16_t*>(outputBuffer.GetPointerUnsafe()),
            outputBuffer.GetSize(),
            reinterpret_cast<const uint8_t*>(inputBuffer.GetPointerUnsafe()),
            inputBuffer.GetSize(),
            &perf,
            isResetRequested);

    // 失敗した場合、
    //   1. result が ResultUserError に含まれるなら返す。
    //   2. 1 以外なら Abort する。
    // NN_RESULT_THROW_UNLESS( !ResultUserError::Includes(result), result );
    // ResultUserError を定義していないので、暫定的に ResultOpusError を使用する。
    NN_RESULT_THROW_UNLESS( !ResultOpusError::Includes(result), result );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *pOutConsumed = consumed;
    *pOutSampleCount = outputSampleCount;
    if (pOutPerf.GetPointer() != nullptr)
    {
        *pOutPerf = perf.GetMicroSeconds();
    }
    NN_RESULT_SUCCESS;
}

nn::Result HardwareOpusDecoderManagerImpl::HardwareOpusDecoderImpl::SetContextForMultiStream(const nn::sf::InBuffer& context) NN_NOEXCEPT
{
    // TODO:
    //   - 初期化の確認をおこない、適切な result を返す。
    //   - ResultUserError を定義し、使うようにする。
    auto result = m_DecoderImpl.SetContext(
        reinterpret_cast<const uint8_t*>(context.GetPointerUnsafe()),
        context.GetSize());
#if 0
    // ResultUserError は返す、そうでなければ Abort する、という I/F にする予定
    NN_RESULT_THROW_UNLESS( !ResultUserError::Includes(result), result );
#endif
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

}}  // namespace nn::codec
