﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdint>
#include <cmath>

namespace nn { namespace codec {

namespace {

const int OrderMax = 16;  // TODO: 2 is enough for now
const double Epsilon = 1e-30;

}

int ExecuteDurbin(double* ac, int order, double* ref, double* taps, double* e2)
{
    // Levinson-Durbin method for solving for the
    // Reflection coefficients from the autocorrelation coefficients.
    // Stability is checked. Return value is >0 if solution
    // is unstable. Tap values are also provided.

    double e, m;
    int stable = 0;

    e = ac[0];

    taps[0] = 1.0;

    for (auto i = 1; i <= order; ++i)
    {
        m = 0;

        for (auto j = 1; j < i; ++j)
        {
            m += taps[j] * ac[i - j];
        }

        // Calculate reflection coefficient value - which is taps[i] in
        // this case.

        if (e > 0)
        {
            taps[i] = -(ac[i] + m) / e;
        }
        else
        {
            taps[i] = 0;
        }

        ref[i] = taps[i];

        if (fabs(ref[i]) > 1.0)
        {
            stable++;
        }

        // Recalculate the tap values
        for (auto j = 1; j < i; ++j)
        {
            taps[j] = taps[j] + taps[i]*taps[i - j];
        }

        // Calculate forward error after predictor
        e = (1 - taps[i] * taps[i]) * e;
    }

    *e2 = e;

    return stable;
}

void CalculateTapFromReflection(double* ref, double* taps, int order)
{
    // Get the tap values from reflecion coefficients.

    taps[0] = 1.0;

    for (auto i = 1; i <= order; ++i)
    {
        taps[i] = ref[i];

        // Recalculate the tap values
        for (auto j = 1; j < i; ++j)
        {
            taps[j] = taps[j] + taps[i] * taps[i - j];
        }
    }
}

int CalculateReflectionFromTap(double* taps, double* ref, int order)
{
//     NN_SDK_REQUIRES(order <= OrderMax);

    // Get the reflecion coefficients from the tap values.
    double  tmp[OrderMax];
    int     stable = 0;

    ref[order] = taps[order];

    for (auto i = order - 1; i > 0; --i)
    {
        for (auto j = 0; j <= i; ++j)
        {
            double den = 1 - ref[i + 1] * ref[i + 1];

            // Check here for unstable filter
            if (den == 0)
            {
                return 1;
            }

            tmp[j] = (taps[j] - ref[i + 1] * taps[i + 1 - j]) / den;
        }

        for (auto j = 0; j <= i; ++j)
        {
            taps[j] = tmp[j];
        }

        ref[i] = tmp[i];

        if (fabs(ref[i]) > 1.0)
        {
            stable++;
        }
    }

    return stable;
}

void CalculateAutoCorrelationFromTapOrder2(double* a, int n, double* r)
{
//     NN_SDK_REQUIRES(n == 2);

    // Calculate the signal autocorrelation from the
    // model tap values. See Atal and Hanauer, Journal
    // of the Acoustical Society of America, Vol 50, 1971,
    // pp. 637-655.
    double  aa[3][3];

    aa[2][0] = 1.0;
    aa[2][1] = -a[1];
    aa[2][2] = -a[2];

    // コンパイラの最適化の影響を受けないようにするため括弧が必要
    aa[1][1] = (aa[2][1] + (aa[2][2] * aa[2][1])) / (1.0 - (aa[2][2] * aa[2][2]));

    // Now calculate the autocorrelation coefficients
    r[0] = 1.0;
    r[1] = aa[1][1] * r[0];
    r[2] = aa[2][1] * r[1] + aa[2][2] * r[0];
}

void CalculateAutoCorrelationFromTap(double* a, int n, double* r)
{
//     NN_SDK_REQUIRES(n <= OrderMax);

    // Calculate the signal autocorrelation from the
    // model tap values. See Atal and Hanauer, Journal
    // of the Acoustical Society of America, Vol 50, 1971,
    // pp. 637-655.
    double  ref;
    double  aa[OrderMax][OrderMax];

    aa[n][0] = 1.0;

    for (auto i = 1; i <= n; ++i)
    {
        aa[n][i] = -a[i];
    }

    for (auto i = n; i >= 1; --i)
    {
        ref = (1 - aa[i][i] * aa[i][i]);

        // Calculate the tap values for this i
        for (auto j = 1; j <= i - 1; ++j)
        {
            aa[i - 1][j] = (aa[i][j] + aa[i][i] * aa[i][i - j]) / ref;
        }
    }

    // Now calculate the autocorrelation coefficients
    r[0] = 1.0;

    for (auto i = 1; i <= n; ++i)
    {
        r[i] = 0;

        for (auto j = 1; j <= i; ++j)
        {
            r[i] += aa[i][j] * r[i - j];
        }
    }
}

double CalculateModelDistanceOrder2(double* ta, double* sa, int order)
{
//     NN_SDK_ASSERT(order == 2);

    // Calculate the normalized model distance
    // See, Gray, et al IEEE Transacations, ASSP-28, No. 4
    // Aug. 1980, pp. 367-376.
    double  r[3], ra[3], d;

    // Calculate signal autocorrelation coefficients from
    // model sa
    CalculateAutoCorrelationFromTapOrder2(sa, order, r);

    // Calculate the coefficient autocorrelations from
    // model ta
    ra[0] = ta[0] * ta[0] + ta[1] * ta[1] + ta[2] * ta[2];
    ra[1] = ta[0] * ta[1] + ta[1] * ta[2];
    ra[2] = ta[0] * ta[2];


    // Model distance
    d = r[0] * ra[0] + 2.0 * r[1] * ra[1] + 2.0 * r[2] * ra[2];

    return d;
}

double CalculateModelDistance(double* ta, double* sa, int order)
{
//     NN_SDK_ASSERT(order <= OrderMax);

    // Calculate the normalized model distance
    // See, Gray, et al IEEE Transacations, ASSP-28, No. 4
    // Aug. 1980, pp. 367-376.
    double  r[OrderMax], ra[OrderMax], d;

    // Calculate signal autocorrelation coefficients from
    // model sa
    CalculateAutoCorrelationFromTap(sa, order, r);

    // Calculate the coefficient autocorrelations from
    // model ta
    for (auto i = 0; i <= order; ++i)
    {
        ra[i] = 0;

        for (auto j = 0; j <= order - i; ++j)
        {
            ra[i] += ta[j] * ta[j + i];
        }
    }

    // Model distance
    d = r[0] * ra[0];

    for (auto i = 1; i <= order; ++i)
    {
        d += 2 * r[i] * ra[i];
    }
    return d;
}

void CalculateAutoCorrelationMatrix(int16_t* in_buffer, int order, int length, double ** a)
{
    for (auto i = 1; i <= order; ++i)
    {
        for (auto j = 1; j <= order; ++j)
        {
            a[i][j] = 0;
            for (auto n = 0; n < length; ++n)
            {
                a[i][j] += in_buffer[n - i] * in_buffer[n - j];
            }
        }
    }
}

void CalculateAutoCorrelationVector(int16_t* in_buffer, int order, int length, double* a)
{
    for (auto i = 0; i <= order; ++i)
    {
        a[i] = 0;
        for (auto n = 0; n < length; ++n)
        {
            a[i] -= in_buffer[n - i] * in_buffer[n];
        }
    }
}

int ExecuteLuComposition(double** a, int n, int32_t* index)
{
//     NN_SDK_REQUIRES(n <= OrderMax);

    int     imax;
    double  scale[OrderMax + 1];

    for (auto i = 1; i <= n; ++i)
    {
        double max = 0;
        for (auto j = 1; j <= n; ++j)
        {
            max = std::max(max, fabs(a[i][j]));
        }

        if (max == 0)
        {
            return 1;
        }

        scale[i] = 1 / max;
    }

    for (auto j = 1; j <= n; ++j)
    {
        for (auto i = 1; i < j; ++i)
        {
            double sum = 0;
            for (auto k = 1; k < i; ++k)
            {
                sum += a[i][k] * a[k][j];
            }
            a[i][j] -= sum;
        }

        for (auto i = j; i <= n; ++i)
        {
            double sum = 0;
            for (auto k = 1; k < j; ++k)
            {
                sum += a[i][k] * a[k][j];
            }
            a[i][j] -= sum;
        }

        double max = 0;
        for (auto i = j; i <= n; ++i)
        {
            auto tmp = scale[i] * fabs(a[i][j]);
            if (tmp >= max)
            {
                max = tmp;
                imax = i;
            }
        }

        if (j != imax)
        {
            for (auto k = 1; k <= n; ++k)
            {
                std::swap(a[imax][k], a[j][k]);
            }
            scale[imax] = scale[j];
        }

        index[j] = imax;

        if (a[j][j] == 0)  // Matrix is singular
        {
            return 1;
        }

        if (j != n)
        {
            auto tmp = 1 / a[j][j];
            for (auto i = j + 1; i <= n; ++i)
            {
                a[i][j] *= tmp;
            }
        }
    }

    // Search for maximum and minimum magnitude pivots
    double pmax = 0;
    double pmin = 1 / Epsilon;

    for (auto j = 1; j <= n; ++j)
    {
        auto mag = fabs(a[j][j]);
        pmin = std::min(pmin, mag);
        pmax = std::max(pmax, mag);
    }

    // Ad hoc singularity check. If the ratio
    // of pmin/pmax is less than Epsilon call it singular
    if (pmin / pmax < Epsilon)
    {
        return 1;
    }

    return 0;
}

void SolveEquationWithLuDecomposition(double** a, int n, int32_t* index, double* b)
{
    int     ii = 0, ip;
    double  sum;

    for (auto i = 1; i <= n; ++i)
    {
        ip = index[i];
        sum = b[ip];
        b[ip] = b[i];

        if (ii)
        {
            for (auto j = ii; j <= i - 1; ++j)
            {
                sum -= a[i][j] * b[j];
            }
        }
        else if (sum)
        {
            ii = i;
        }

        b[i] = sum;
    }

    for (auto i = n; i >= 1; --i)
    {
        sum = b[i];

        for (auto j = i + 1; j <= n; ++j)
        {
            sum -= a[i][j] * b[j];
        }

        b[i] = sum / a[i][i];
    }
}

}}  // namespace nn::codec
