﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdint>
#include <cstdlib>
#include <limits>

#include <nn/codec/codec_AdpcmCommon.h>
#include "codec_AdpcmCommonInternal.h"

namespace nn { namespace codec {

namespace {

const int Order = AdpcmPredictionOrder;

template <typename T, typename U>
T Clamp(U x)
{
    const U max = static_cast<U>(std::numeric_limits<T>::max());
    const U min = static_cast<U>(std::numeric_limits<T>::min());
    x = std::min(x, max);
    x = std::max(x, min);
    return static_cast<T>(x);
}

}

uint16_t EncodeAdpcm(int16_t *inbuffer, uint8_t *outbuffer, int16_t *coeftable, int step)
{
    int opt, scale, optscale;
    double err, e, min;
    int32_t prediction[AdpcmCoefficientsTableCount][AdpcmFrameSampleCount + Order];
    int32_t diff[AdpcmFrameSampleCount];
    int32_t quantum[AdpcmCoefficientsTableCount][AdpcmFrameSampleCount];
    int32_t maxdiff, maxclip;

    min       = 1e30;
    opt       = 0;
    optscale  = 0;

    for (auto i = 0; i < AdpcmCoefficientsTableCount; ++i)
    {
        for (auto k = 0; k < Order; ++k)
        {
            prediction[i][k] = static_cast<int32_t>(inbuffer[k * step]);
        }

        // Find the prediction error from predictor.
        for (auto j = 0; j < AdpcmFrameSampleCount; ++j)
        {
            prediction[i][j + Order] = 0;

            for (auto k = 0; k < Order; ++k)
            {
                prediction[i][j + Order] += static_cast<int32_t>(inbuffer[(j + Order - k - 1) * step]) * static_cast<int32_t>(coeftable[i * Order + k]);
            }

            prediction[i][j + Order] /= AdpcmCoefficientScaling;

            diff[j] = static_cast<int32_t>(inbuffer[(j + Order) * step]) - prediction[i][j + Order];
            diff[j] = Clamp<int16_t>(diff[j]);
        }

        // Find initial range value
        maxdiff = 0;
        for (auto j = 0; j < AdpcmFrameSampleCount; ++j)
        {
            if (abs(diff[j]) > abs(maxdiff))
            {
                maxdiff = diff[j];
            }
        }

        for (scale = 0; scale <= AdpcmMaxScale; ++scale)
        {
            if ((maxdiff <= 7) && (maxdiff >= -8))
            {
                break;
            }
            maxdiff /= 2;
        }

        // Prediction error with a quantizer in the loop
        scale -= 2;
        scale = std::max(scale, -1);

        do
        {
            ++scale;
            maxclip = 0;
            err = 0;

            for (auto j = 0; j < AdpcmFrameSampleCount; ++j)
            {
                prediction[i][j + Order] = 0;

                for (auto k = 0; k < Order; ++k)
                {
                    prediction[i][j + Order] += static_cast<int32_t>(prediction[i][j + Order - k - 1]) * static_cast<int32_t>(coeftable[i * Order + k]);
                }
                diff[j] = static_cast<int32_t>(inbuffer[(j + Order) * step]) * AdpcmCoefficientScaling - prediction[i][j + Order];

                // quantize
                auto qf = static_cast<float>(diff[j]) / static_cast<float>((1 << scale) * AdpcmCoefficientScaling);
                if (diff[j] > 0)
                {
                    quantum[i][j] = static_cast<int32_t>(qf + 0.4999999);
                }
                else
                {
                    quantum[i][j] = static_cast<int32_t>(qf - 0.4999999);
                }

                // Clipping check
                if (quantum[i][j] < -8)
                {
                    maxclip = std::max(maxclip, - 8 - quantum[i][j]);
                    quantum[i][j] = -8;
                }
                if (quantum[i][j] > 7)
                {
                    maxclip = std::max(maxclip, quantum[i][j] - 7);
                    quantum[i][j] = 7;
                }

                prediction[i][j + Order] += quantum[i][j] * (1 << scale) * AdpcmCoefficientScaling;

                // rounding and cut off
                prediction[i][j + Order] += ( AdpcmCoefficientScaling >> 1 );
                prediction[i][j + Order] >>= AdpcmCoefficientScalingBitCount;
                prediction[i][j + Order] = Clamp<int16_t>(prediction[i][j + Order]);

                e = (double)(static_cast<int32_t>(inbuffer[(j + Order) * step]) - prediction[i][j + Order]);
                err += e * e;
//                 NN_SDK_LOG("0x%08X[0x%08X] ", diff[j], prediction[i][j+Order]);  // Debug
            }

            for (auto k = maxclip + 8; k > 256; k >>= 1)
            {
                if (++scale > AdpcmMaxScale - 1)
                {
                    scale = AdpcmMaxScale - 1;
                    break;
                }
            }
//             NN_SDK_LOG("\n");  // Debug
        }
        while((scale < AdpcmMaxScale) && (maxclip > AdpcmMaxClip));

        if (err < min)
        {
            opt      = i;
            min      = err;
            optscale = scale;
        }
//         NN_SDK_LOG("Index %d : error = %e\n", i, err);  // Debug
    }

//     NN_SDK_LOG("Opt. Index = %d  error = %e  scale = %d\n", opt, min, scale); // Debug

    // Write back calculated PCM data to input buffer
    for (auto j = 0; j < AdpcmFrameSampleCount; ++j)
    {
        inbuffer[(j + Order) * step] = static_cast<int16_t>(prediction[opt][j + Order]);
    }

    // Write header and bitstream into output buffer
    uint8_t ps = (opt << 4) | (optscale & 0x0F);
    *outbuffer++ = ps;
//     NN_SDK_LOG("%02X ", ps);  // Debug

    for (auto i = 0; i < AdpcmFrameSampleCount; i += 2)
    {
        auto value = static_cast<uint8_t>(((quantum[opt][i] & 0x0F) << 4) | (quantum[opt][i + 1] & 0x0F));
        *outbuffer++ = value;
//         printf("%02X ", value);  // Debug
    }
//     NN_SDK_LOG("\n");  // Debug

    return ps;
}  // NOLINT(impl/function_size)

}}  // namespace nn::codec
