﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <algorithm>
#include <limits>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

#include <nn/codec/codec_AdpcmCommon.h>
#include "codec_AdpcmCommonInternal.h"

namespace nn {
namespace codec {

void DecodeAdpcm(int16_t* output, AdpcmContext* pContext, const AdpcmParameter* pParameter, const void* input, std::size_t inputSize, int sampleCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParameter);
    NN_SDK_REQUIRES_NOT_NULL(pContext);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_UNUSED(inputSize);
    NN_SDK_REQUIRES(inputSize == AdpcmFrameSize);
    NN_SDK_REQUIRES(sampleCount >= 0 && sampleCount <= nn::codec::AdpcmFrameSampleCount);
    if (sampleCount == 0)
    {
        return;
    }

    auto p = static_cast<const unsigned char*>(input);
    int16_t h0 = pContext->_history[0];
    int16_t h1 = pContext->_history[1];

    int16_t nibble[AdpcmFrameSampleCount];

    // frame header
    unsigned char predScale = *p++;

    int16_t gain = 1 << (predScale & 0x0F);
    int16_t ci = predScale >> 4;

    for (int i = 0; i < AdpcmFrameSampleCount; i += 2)
    {
        auto tmp = *p++;
        nibble[i + 0] = static_cast<int16_t>(tmp >> 4);
        nibble[i + 1] = static_cast<int16_t>(tmp & 0x0F);
    }

    // fix negative value
    for (int i = 0; i < AdpcmFrameSampleCount; i++)
    {
        if (nibble[i] >= AdpcmMaxLevel)
        {
            nibble[i] = nibble[i] - AdpcmMaxLevel * 2;
        }
    }

    // decode the samples
    auto c0 = pParameter->_coefficients[ci * AdpcmPredictionOrder + 0];
    auto c1 = pParameter->_coefficients[ci * AdpcmPredictionOrder + 1];
    for (int i = 0 ; i < sampleCount; i++)
    {
        int32_t mac = static_cast<int32_t>(nibble[i]) * static_cast<int32_t>(gain) * AdpcmCoefficientScaling;

        mac += static_cast<int32_t>(h0) * static_cast<int32_t>(c0);
        mac += static_cast<int32_t>(h1) * static_cast<int32_t>(c1);

        // rounding and cut off
        mac += (AdpcmCoefficientScaling >> 1);
        mac >>= AdpcmCoefficientScalingBitCount;

        // clamping
        int32_t max = std::numeric_limits<int16_t>::max();
        int32_t min = std::numeric_limits<int16_t>::min();
        mac = std::min(max, std::max(min, mac));

        if (output != nullptr)
        {
            *output++ = static_cast<int16_t>(mac);
        }

        h1 = h0;
        h0 = static_cast<int16_t>(mac);
    }

    pContext->_predScale = predScale;
    pContext->_history[0] = h0;
    pContext->_history[1] = h1;
}

void GetAdpcmLoopContext(AdpcmContext* pContext, const AdpcmParameter* pParameter, const void* input, std::size_t inputSize, int sampleCount) NN_NOEXCEPT
{
    return DecodeAdpcm(nullptr, pContext, pParameter, input, inputSize, sampleCount);
}

}  // namespace codec
}  // namespace nn
