﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_SdkLog.h>

#include "cec_Error.h"
#include "cec_FiniteStateMachine.h"
#include "cec_Receiver.h"
#include "cec_TransmitterFiniteStateMachine.h"
#include "cec_LogicalAddressFiniteStateMachine.h"

namespace nn { namespace cec { namespace detail {

typedef struct {
    uint8_t                 index;
    uint8_t                 baseLogicalAddress;
    bool                    isDone;
    int32_t                 lastError;
    nn::os::MutexType       mutex;
    TransmitReturnValueType transmitReturnValue;
    FiniteStateMachineType  fsmState;
} LogicalAddressStateType;

namespace {

/* this should be an enum */
const FiniteStateMachineStateType LogicalAddressStateInit = 0;
const FiniteStateMachineStateType LogicalAddressStateToken1 = 1;
const FiniteStateMachineStateType LogicalAddressStateToken2 = 2;
const FiniteStateMachineStateType LogicalAddressStateToken3 = 3;
const FiniteStateMachineStateType LogicalAddressStateDone = 4;
const FiniteStateMachineStateType LogicalAddressStateNumber = 5;

/* this should be an enum */
const FiniteStateMachineInputType LogicalAddressInputError = 0;
const FiniteStateMachineInputType LogicalAddressInputAck = 1;
const FiniteStateMachineInputType LogicalAddressInputNak = 2;
const FiniteStateMachineInputType LogicalAddressInputStart = 3;
const FiniteStateMachineInputType LogicalAddressInputReset = 4;
const FiniteStateMachineInputType LogicalAddressInputNumber = 5;

const FiniteStateMachineActionValueType LogicalAddressActionResetAddress = 0;
const FiniteStateMachineActionValueType LogicalAddressActionHandleError = 1;
const FiniteStateMachineActionValueType LogicalAddressActionIncAddress = 2;
const FiniteStateMachineActionValueType LogicalAddressActionHandleSuccess = 3;
const FiniteStateMachineActionValueType LogicalAddressActionFinish = 4;
const FiniteStateMachineActionValueType LogicalAddressActionReset = 5;
const FiniteStateMachineActionValueType LogicalAddressActionNumber = 6;

const FiniteStateMachineStateType s_LogicalAddressStateMatrix[LogicalAddressStateNumber * LogicalAddressInputNumber] =
{
    LogicalAddressStateInit, LogicalAddressStateInit, LogicalAddressStateInit, LogicalAddressStateToken1, LogicalAddressStateInit,
    LogicalAddressStateDone, LogicalAddressStateToken2, LogicalAddressStateDone, LogicalAddressStateToken1, LogicalAddressStateInit,
    LogicalAddressStateDone, LogicalAddressStateToken3, LogicalAddressStateDone, LogicalAddressStateToken2, LogicalAddressStateInit,
    LogicalAddressStateDone, LogicalAddressStateDone, LogicalAddressStateDone, LogicalAddressStateToken3, LogicalAddressStateInit,
    LogicalAddressStateDone, LogicalAddressStateDone, LogicalAddressStateDone, LogicalAddressStateDone, LogicalAddressStateInit
};

const FiniteStateMachineActionType g_CecLAddrActionMatrix[LogicalAddressStateNumber * LogicalAddressInputNumber] =
{
    /* LogicalAddressStateInit */
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionResetAddress, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    /* LogicalAddressStateToken1 */
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionIncAddress, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleSuccess, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionReset, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    /* LogicalAddressStateToken2 */
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionIncAddress, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleSuccess, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionReset, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    /* LogicalAddressStateToken3 */
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleSuccess, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionReset, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}},
    /* LogicalAddressStateDone */
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleError, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionHandleSuccess, LogicalAddressActionFinish, FsmParamNotAnAction, FsmParamNotAnAction}},
    {{LogicalAddressActionReset, FsmParamNotAnAction, FsmParamNotAnAction, FsmParamNotAnAction}}
};

LogicalAddressStateType s_LogicalAddressState;
uint32_t                s_LogicalAddressStatus = 0;

const uint32_t LogicalAddressStatusBitEnabled = 1 << 0;

const uint8_t s_LogicalAddressTable[4] =
{
    0x04, 0x08, 0x0B, 0x0F
};
const uint32_t s_LogicalAddressNumberOfTableEntries = (sizeof(s_LogicalAddressTable) / sizeof(uint8_t));

void LogicalAddressActionHandlerResetAddress(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    LogicalAddressStateType*    pState;

    pState = static_cast<LogicalAddressStateType*>(pContext);
    pState->index = 0;
    pState->baseLogicalAddress = s_LogicalAddressTable[pState->index];
}

void LogicalAddressActionHandlerHandleError(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    LogicalAddressStateType*    pState;

    pState = static_cast<LogicalAddressStateType*>(pContext);
    pState->lastError = pState->transmitReturnValue.error;
    pState->baseLogicalAddress = CecConstantNotALogicalAddress;
    if(pState->lastError == CecNoError)
    {
        pState->lastError = CecLogAddrInvalidLogAddrError;
    }
}

void LogicalAddressActionHandlerIncAddress(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    LogicalAddressStateType*    pState;

    pState = static_cast<LogicalAddressStateType*>(pContext);
    pState->index = pState->index + 1;
    if(pState->index < s_LogicalAddressNumberOfTableEntries)
    {
        pState->baseLogicalAddress = s_LogicalAddressTable[pState->index];
    }
    else
    {
        pState->baseLogicalAddress = CecConstantNotALogicalAddress;
    }
}

void LogicalAddressActionHandlerHandleSuccess(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    LogicalAddressStateType*    pState;

    pState = static_cast<LogicalAddressStateType*>(pContext);
    pState->lastError = CecNoError;
}

void LogicalAddressActionHandlerFinish(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    LogicalAddressStateType*    pState;
    uint8_t*         pReturn;

    pState = static_cast<LogicalAddressStateType*>(pContext);
    pReturn = static_cast<uint8_t*>(pExtra);
    pState->isDone = true;
    if(pReturn != nullptr)
    {
        *pReturn = pState->baseLogicalAddress;
    }
}

void LogicalAddressActionHandlerReset(FiniteStateMachineActionValueType actCode, void* pContext, void* pExtra) NN_NOEXCEPT
{
    LogicalAddressStateType*    pState;

    pState = static_cast<LogicalAddressStateType*>(pContext);
    pState->index = 0;
    pState->baseLogicalAddress = s_LogicalAddressTable[pState->index];
    pState->isDone = false;
    pState->lastError = CecNoError;
}

const FiniteStateMachineActionHandlerFunctionPointerType s_LogicalAddressActionHandlers[LogicalAddressActionNumber] =
{
    LogicalAddressActionHandlerResetAddress,
    LogicalAddressActionHandlerHandleError,
    LogicalAddressActionHandlerIncAddress,
    LogicalAddressActionHandlerHandleSuccess,
    LogicalAddressActionHandlerFinish,
    LogicalAddressActionHandlerReset
};

}

void CecLogicalAddressInit() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!(s_LogicalAddressStatus & LogicalAddressStatusBitEnabled));
    nn::os::InitializeMutex(&s_LogicalAddressState.mutex, false, 0);
    s_LogicalAddressState.index = 0;
    s_LogicalAddressState.baseLogicalAddress = s_LogicalAddressTable[s_LogicalAddressState.index];
    s_LogicalAddressState.lastError = CecNoError;
    s_LogicalAddressState.isDone = false;
    CecFiniteStateMachineInit(&s_LogicalAddressState.fsmState,
                              LogicalAddressStateNumber,
                              LogicalAddressActionNumber,
                              LogicalAddressInputNumber,
                              s_LogicalAddressStateMatrix,
                              g_CecLAddrActionMatrix,
                              s_LogicalAddressActionHandlers,
                              static_cast<void*>(&s_LogicalAddressState));
    s_LogicalAddressStatus |= LogicalAddressStatusBitEnabled;
}

void    CecLogicalAddressShutdown() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_LogicalAddressStatus & LogicalAddressStatusBitEnabled);
    CecLogicalAddressReset();
    nn::os::FinalizeMutex(&s_LogicalAddressState.mutex);
    s_LogicalAddressStatus &= ~LogicalAddressStatusBitEnabled;
}

int32_t CecLogicalAddressGetLogicalAddress(uint8_t* pLogicalAddress) NN_NOEXCEPT
{
    int32_t                     rval;
    uint8_t                     pollValue;
    FiniteStateMachineInputType fsmInput;

    NN_SDK_ASSERT(s_LogicalAddressStatus & LogicalAddressStatusBitEnabled);
    nn::os::LockMutex(&s_LogicalAddressState.mutex);
    *pLogicalAddress = CecConstantNotALogicalAddress;
    CecFiniteStateMachineStep(&s_LogicalAddressState.fsmState,
                              LogicalAddressInputStart,
                              static_cast<void*>(pLogicalAddress));
    rval = s_LogicalAddressState.lastError;
    while((rval == CecNoError) && (s_LogicalAddressState.isDone == false))
    {
        /*
         * Create poll value.  It consists of the polled logical address and is used
         * as sender and receiver.  Sender: upper 4 bits; Receiver: lower 4 bits
         */
        pollValue = static_cast<uint8_t>(s_LogicalAddressState.baseLogicalAddress);
        pollValue |= pollValue << 4;

        rval = CecTransmitPollingRequest(&s_LogicalAddressState.transmitReturnValue, pollValue);
        if(rval == CecManagerImplCancelledError)
        {
            CecFiniteStateMachineStep(&s_LogicalAddressState.fsmState, LogicalAddressInputReset, nullptr);
            break;
        }
        else
        {
            rval = CecNoError;
            switch(s_LogicalAddressState.transmitReturnValue.ackNak)
            {
                case TransmitAckNakValueAck:
                    fsmInput = LogicalAddressInputAck;
                    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(CecReceiveParamFrameToFrameDelayNanoSeconds));
                    break;
                case TransmitAckNakValueNak:
                    fsmInput = LogicalAddressInputNak;
                    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(CecReceiveParamErrorToNextDelayNanoSeconds));
                    break;
                case TransmitAckNakValueOther:
                    NN_FALL_THROUGH;
                default:
                    fsmInput = LogicalAddressInputError;
                    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(CecReceiveParamErrorToNextDelayNanoSeconds));
                    break;
            }
            CecFiniteStateMachineStep(&s_LogicalAddressState.fsmState, fsmInput, static_cast<void*>(pLogicalAddress));
        }
    }
    rval = s_LogicalAddressState.lastError;
    nn::os::UnlockMutex(&s_LogicalAddressState.mutex);
    return rval;
}

void CecLogicalAddressReset() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_LogicalAddressStatus & LogicalAddressStatusBitEnabled);
    nn::os::LockMutex(&s_LogicalAddressState.mutex);
    CecFiniteStateMachineStep(&s_LogicalAddressState.fsmState, LogicalAddressInputReset, nullptr);
    nn::os::UnlockMutex(&s_LogicalAddressState.mutex);
}

}
}   // namespace cec
}   // namespace nn
