﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file  cdmsc_Utils.h
 * @brief Helper utilities
 *
 * @details
 */

#pragma once

#include <list>
#include <memory>
#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>

#include <nn/dd/dd_DeviceAddressSpaceCommon.h>

#include <nn/cdmsc/cdmsc_Types.h>

#if defined(NN_BUILD_CONFIG_COMPILER_GCC)   || \
    defined(NN_BUILD_CONFIG_COMPILER_CLANG) || \
    defined(NN_BUILD_CONFIG_COMPILER_GHS)   || \
    defined(NN_BUILD_CONFIG_COMPILER_OASIS_CAFE_CLANG)

    #define NN_CDMSC_PACKED_ALIGNED(alignment)              \
        __attribute__((__packed__,__aligned__(alignment)))
    #define NN_CDMSC_PACKED                                 \
        __attribute__((__packed__))

#else // VC

    #define NN_CDMSC_PACKED_ALIGNED(alignment)
    #define NN_CDMSC_PACKED()

#endif

#define NN_CDMSC_DMA_ALIGN NN_ALIGNAS(nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

#define NN_CDMSC_ABORT(format,...)              \
    NN_ABORT("abort: " format, ##__VA_ARGS__)
#define NN_CDMSC_ABORT_UNLESS_SUCCESS(result)                                            \
    NN_ABORT_UNLESS((result).IsSuccess())
#define NN_CDMSC_ABORT_IF_NULL(pointerVal)                                               \
    NN_ABORT_UNLESS_NOT_NULL(pointerVal)

// Error handling macros
#define NN_CDMSC_ASSERT_IF_ZERO(uintval)                                                 \
    if(!(uintval))                                                                       \
    {                                                                                    \
        NN_CDMSC_ABORT("%s is zero in func-%s @ line-%d.\n",                             \
                        #uintval,__FUNCTION__,__LINE__);                                 \
    }

#define NN_CDMSC_BREAK_UPON_ERROR(attemptedMethodCall)                                   \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                                   \
    {                                                                                    \
        NN_CDMSC_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                   \
                           #attemptedMethodCall,result.GetModule(),                      \
                           result.GetDescription(),__FUNCTION__,__LINE__);               \
        break;                                                                           \
    }

#define NN_CDMSC_BREAK_UNLESS(condition, failureResult)                                  \
    if (!(condition))                                                                    \
    {                                                                                    \
        NN_CDMSC_WARN("condition %s false in func-%s @ line-%d.\n",                      \
                         #condition,__FUNCTION__,__LINE__);                              \
        result=failureResult;                                                            \
        break;                                                                           \
    }

#define NN_CDMSC_BREAK_UPON_MEM_ALLOC_FAIL(attemptedAlloc)                               \
    if (!((attemptedAlloc)!=NULL))                                                       \
    {                                                                                    \
        result = ResultMemAllocFailure();                                                \
        NN_CDMSC_WARN("%s unsuccessful alloc in func-%s @ line-%d.\n",                   \
                        #attemptedAlloc,__FUNCTION__,__LINE__);                          \
        break;                                                                           \
    }

#define NN_CDMSC_BREAK_UPON_NULL_WITH_ERROR(pointer, resultCode)                         \
    if (!((pointer)!=NULL))                                                              \
    {                                                                                    \
        result = resultCode;                                                             \
        NN_CDMSC_WARN("%s is NULL in func-%s @ line-%d, error %d:%d.\n",                 \
                        #pointer,__FUNCTION__,__LINE__,result.GetModule(),               \
                        result.GetDescription());                                        \
        break;                                                                           \
    }

#define NN_CDMSC_ABORT_UNLESS(condition)                                                 \
    if (!(condition))                                                                    \
    {                                                                                    \
        NN_CDMSC_ABORT("condition %s false in func-%s @ line-%d.\n",                     \
                     #condition,__FUNCTION__,__LINE__);                                  \
    }


#define NN_CDMSC_RETURN_UPON_ERROR(attemptedMethodCall)                                  \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                                   \
    {                                                                                    \
        NN_CDMSC_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                   \
                        #attemptedMethodCall,result.GetModule(),                         \
                        result.GetDescription(),__FUNCTION__,__LINE__);                  \
        return result;                                                                   \
    }

#define NN_CDMSC_ABORT_UPON_ERROR(attemptedMethodCall)                                   \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                                   \
    {                                                                                    \
        NN_CDMSC_ABORT("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                  \
                     #attemptedMethodCall,result.GetModule(),                            \
                     result.GetDescription(),__FUNCTION__,__LINE__);                     \
    }

#define NN_CDMSC_EXPECT(expectation, result)                                             \
    do {                                                                                 \
        if (!(expectation))                                                              \
        {                                                                                \
            NN_CDMSC_WARN("Excpecting %s %d:%d in func-%s @ line-%d.\n",                 \
                          #expectation, result.GetModule(),                              \
                          result.GetDescription(),__FUNCTION__,__LINE__);                \
            return (result);                                                             \
        }                                                                                \
    } while (0)

#define NN_CDMSC_DO(action)                                                              \
    do {                                                                                 \
        result = action;                                                                 \
        if (result.IsFailure())                                                          \
        {                                                                                \
            NN_CDMSC_WARN("%s failed with %d:%d in func-%s @ line-%d.\n",                \
                          #action, result.GetModule(), result.GetDescription(),          \
                          __FUNCTION__,__LINE__);                                        \
        }                                                                                \
    } while(0)

#define NN_CDMSC_LOG_UPON_ERROR(attemptedMethodCall)                                     \
    if (!((result=(attemptedMethodCall)).IsSuccess()))                                   \
    {                                                                                    \
        NN_CDMSC_WARN("%s unsuccessful %d:%d in func-%s @ line-%d.\n",                   \
                        #attemptedMethodCall,result.GetModule(),                         \
                        result.GetDescription(),__FUNCTION__,__LINE__);                  \
    }

#define NN_CDMSC_ROUNDUP_SIZE(inputSize, powerOfTwoRoundedSize)                          \
    (((inputSize) + (powerOfTwoRoundedSize) - 1) & ~size_t((powerOfTwoRoundedSize) - 1))

#define NN_CDMSC_ROUNDUP_DMA_SIZE(size)                                                  \
    NN_CDMSC_ROUNDUP_SIZE(size,nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

#define BmRequestType(type, recip, dir)                                                  \
    (nn::usb::UsbCtrlXferReqTypeMask_Type##type   |                                      \
     nn::usb::UsbCtrlXferReqTypeMask_Recip##recip |                                      \
     nn::usb::UsbCtrlXferReqTypeMask_Dir##dir     )

#define NN_CDMSC_ARRAY_SIZE(array) (sizeof(array) / sizeof((array)[0]))

namespace nn {
namespace cdmsc {
namespace detail {

void  DumpBuffer(void *buffer, int size) NN_NOEXCEPT;
void  SetAllocator(AllocateFunction alloc, DeallocateFunction dealloc) NN_NOEXCEPT;
void* Allocate(size_t alignment, size_t size) NN_NOEXCEPT;
void  Deallocate(void* p, size_t size) NN_NOEXCEPT;

static inline bool IsLittleEndian()
{
    const union {
        long a;
        char c[sizeof(long)];
    } u = { 1 };

    return (u.c[0] == 1);
}

static inline uint16_t CpuToBe16(uint16_t var)
{
    if (IsLittleEndian())
    {
        return
            (var & 0xff00) >> 8 |
            (var & 0x00ff) << 8 ;
    }
    else
    {
        return var;
    }
}

static inline uint16_t BeToCpu16(uint16_t var)
{
    return CpuToBe16(var);
}

static inline uint32_t CpuToBe32(uint32_t var)
{
    if (IsLittleEndian())
    {
        return
            (var & 0xff000000) >> 24 |
            (var & 0x00ff0000) >>  8 |
            (var & 0x0000ff00) <<  8 |
            (var & 0x000000ff) << 24 ;
    }
    else
    {
        return var;
    }
}

static inline uint32_t BeToCpu32(uint32_t var)
{
    return CpuToBe32(var);
}

static inline uint64_t CpuToBe64(uint64_t var)
{
    if (IsLittleEndian())
    {
        return
            (var & 0xff00000000000000) >> 56 |
            (var & 0x00ff000000000000) >> 40 |
            (var & 0x0000ff0000000000) >> 24 |
            (var & 0x000000ff00000000) >>  8 |
            (var & 0x00000000ff000000) <<  8 |
            (var & 0x0000000000ff0000) << 24 |
            (var & 0x000000000000ff00) << 40 |
            (var & 0x00000000000000ff) << 56 ;
    }
    else
    {
        return var;
    }
}

static inline uint64_t BeToCpu64(uint64_t var)
{
    return CpuToBe64(var);
}

static inline uint16_t CpuToLe16(uint16_t var)
{
    if (IsLittleEndian())
    {
        return var;
    }
    else
    {
        return
            (var & 0xff00) >> 8 |
            (var & 0x00ff) << 8 ;
    }
}

static inline uint16_t LeToCpu16(uint16_t var)
{
    return CpuToLe16(var);
}

static inline uint32_t CpuToLe32(uint32_t var)
{
    if (IsLittleEndian())
    {
        return var;
    }
    else
    {
        return
            (var & 0xff000000) >> 24 |
            (var & 0x00ff0000) >>  8 |
            (var & 0x0000ff00) <<  8 |
            (var & 0x000000ff) << 24 ;
    }
}

static inline uint32_t LeToCpu32(uint32_t var)
{
    return CpuToLe32(var);
}

static inline uint64_t CpuToLe64(uint64_t var)
{
    if (IsLittleEndian())
    {
        return var;
    }
    else
    {
        return
            (var & 0xff00000000000000) >> 56 |
            (var & 0x00ff000000000000) >> 40 |
            (var & 0x0000ff0000000000) >> 24 |
            (var & 0x000000ff00000000) >>  8 |
            (var & 0x00000000ff000000) <<  8 |
            (var & 0x0000000000ff0000) << 24 |
            (var & 0x000000000000ff00) << 40 |
            (var & 0x00000000000000ff) << 56 ;
    }
}

static inline uint64_t LeToCpu64(uint64_t var)
{
    return CpuToLe64(var);
}

} // end of namespace detail
} // end of namespace cdmsc
} // end of namespace nn


