﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidWin32Keyboard_Private.h"

namespace nn {
namespace cdhid {
namespace win32 {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardCodeBook::Initialize() NN_NOEXCEPT
{
    memset(m_CodeBook, 0, sizeof(m_CodeBook));

    uint8_t *pDst = m_CodeBook;

    // Keyboard usually have an output report to set LEDs but in this
    // case Windows keyboard driver handle this. Since we are not setting
    // LEDs, the LED states are stuffed in the input data [1] for this
    // implementation such that proper output can be generated by keyboard
    // driver by looking at the LED states in input data.

    pDst += PrintCodeBookHeader(pDst, 0x06, 0x01, 4, 0, 0, WIN32KEYBOARD_INPUT_ITEMS, WIN32KEYBOARD_OUTPUT_ITEMS, WIN32KEYBOARD_FEATURE_ITEMS);

    // special keys
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe0, 0xe0, 0, 0x01, 0);
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe1, 0xe1, 0, 0x02, 1);
/* Windows keyboard API cannot support L/R alt, L/R GUI
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe2, 0xe2, 0, 0x04, 2);
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe3, 0xe3, 0, 0x08, 3);
 */
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe4, 0xe4, 0, 0x10, 4);
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe5, 0xe5, 0, 0x20, 5);
/* Windows keyboard API cannot support L/R alt, L/R GUI
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe6, 0xe6, 0, 0x40, 6);
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x07, 0xe7, 0xe7, 0, 0x80, 7);
 */

    // LEDs
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x08, 0x01, 0x01, 1, 0x01, 0);
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x08, 0x02, 0x02, 1, 0x02, 1);
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x08, 0x03, 0x03, 1, 0x04, 2);
/* Windows does not support compose
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x08, 0x04, 0x04, 1, 0x08, 3);
 */
    pDst += PrintItemBitmask8(pDst, 0, 0, 0x08, 0x05, 0x05, 1, 0x10, 4);

    // scan codes
    pDst += PrintItemLogical8(pDst, 0, 0, 0x07, 0x00, 0x65, 2, 0x00, 0x65);
    pDst += PrintItemLogical8(pDst, 0, 1, 0x07, 0x00, 0x65, 3, 0x00, 0x65);
    pDst += PrintItemLogical8(pDst, 0, 2, 0x07, 0x00, 0x65, 4, 0x00, 0x65);
    pDst += PrintItemLogical8(pDst, 0, 3, 0x07, 0x00, 0x65, 5, 0x00, 0x65);
    pDst += PrintItemLogical8(pDst, 0, 4, 0x07, 0x00, 0x65, 6, 0x00, 0x65);
    pDst += PrintItemLogical8(pDst, 0, 5, 0x07, 0x00, 0x65, 7, 0x00, 0x65);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardCodeBook::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32KeyboardCodeBook::Get(uint8_t* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    if (pBuffer && (bufferSize >= sizeof(m_CodeBook)))
    {
        memcpy(pBuffer, m_CodeBook, sizeof(m_CodeBook));

        return ResultSuccess();
    }

    return nn::ahid::ResultBufferSize();
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
uint32_t Win32KeyboardCodeBook::PrintCodeBookHeader(
    uint8_t*    pDst,
    uint8_t     usage,
    uint8_t     usagePage,
    uint32_t    inputSize,
    uint32_t    outputSize,
    uint32_t    featureSize,
    uint32_t    inputItems,
    uint32_t    outputItems,
    uint32_t    featureItems
    )
{
    nn::ahid::CodeBookHeader *pCodeBookHeader = reinterpret_cast<nn::ahid::CodeBookHeader*>(pDst);

    pCodeBookHeader->reportCount    = 1;
    pCodeBookHeader->reportId[0]    = 0;
    pCodeBookHeader->usage[0]       = usage;
    pCodeBookHeader->usagePage[0]   = usagePage;
    pCodeBookHeader->inputSize      = inputSize;
    pCodeBookHeader->outputSize     = outputSize;
    pCodeBookHeader->featureSize    = featureSize;
    pCodeBookHeader->inputItems     = inputItems;
    pCodeBookHeader->outputItems    = outputItems;
    pCodeBookHeader->featureItems   = featureItems;

    return sizeof(nn::ahid::CodeBookHeader);
}


//////////////////////////////////////////////////////////////////////////////
size_t Win32KeyboardCodeBook::PrintItemBitmask8(
    uint8_t*    pDst,
    uint8_t     reportId,
    uint8_t     index,
    uint16_t    usagePage,
    uint16_t    usageMin,
    uint16_t    usageMax,
    uint16_t    dataOffset,
    uint8_t     dataBits,
    uint8_t     dataShift
    )
{
    nn::ahid::ItemBitmask8 *pItemBitmask8     = reinterpret_cast<nn::ahid::ItemBitmask8*>(pDst);

    pItemBitmask8->item.type        = nn::ahid::DataType::DataType_Bitmask8;
    pItemBitmask8->item.length      = sizeof(nn::ahid::ItemBitmask8);
    pItemBitmask8->item.index       = index;
    pItemBitmask8->item.reportId    = reportId;
    pItemBitmask8->item.usagePage   = usagePage;
    pItemBitmask8->item.usageMin    = usageMin;
    pItemBitmask8->item.usageMax    = usageMax;

    pItemBitmask8->dataOffset       = dataOffset;
    pItemBitmask8->dataBits         = dataBits;
    pItemBitmask8->dataShift        = dataShift;

    return sizeof(nn::ahid::ItemBitmask8);
}


//////////////////////////////////////////////////////////////////////////////
size_t Win32KeyboardCodeBook::PrintItemLogical8(
    uint8_t*    pDst,
    uint8_t     reportId,
    uint8_t     index,
    uint16_t    usagePage,
    uint16_t    usageMin,
    uint16_t    usageMax,
    uint16_t    dataOffset,
    uint8_t     dataMin,
    uint8_t     dataMax
    )
{
    nn::ahid::ItemLogical8 *pItemLogical8 = reinterpret_cast<nn::ahid::ItemLogical8*>(pDst);

    pItemLogical8->item.type        = nn::ahid::DataType::DataType_Logical8;
    pItemLogical8->item.length      = sizeof(nn::ahid::ItemLogical8);
    pItemLogical8->item.reportId    = reportId;
    pItemLogical8->item.index       = index;
    pItemLogical8->item.usagePage   = usagePage;
    pItemLogical8->item.usageMin    = usageMin;
    pItemLogical8->item.usageMax    = usageMax;

    pItemLogical8->dataOffset       = dataOffset;
    pItemLogical8->logicalMinimum   = dataMin;
    pItemLogical8->logicalMaximum   = dataMax;

    return sizeof(nn::ahid::ItemLogical8);
}



}   // namespace win32
}   // namespace cdhid
}   // namespace nn
