﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ahid/hdr/hdr.h>

namespace nn {
namespace cdhid {
namespace usb {

const uint8_t SkipBytes[4] = {1, 2, 3, 5};
const uint8_t DataBits8[9] = {0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff};
const uint16_t DataBits16[17] = {0x0000, 0x0001, 0x0003, 0x0007, 0x000f, 0x001f, 0x003f, 0x007f, 0x00ff,
                                 0x01ff, 0x03ff, 0x07ff, 0x0fff, 0x1fff, 0x3fff, 0x7fff, 0xffff};

enum ItemType
{
    Main                = 0x0,
    Global              = 0x1,
    Local               = 0x2,
};


enum ItemSize
{
    Zero                = 0x0,
    One                 = 0x1,
    Two                 = 0x2,
    Four                = 0x3,
};

enum MainItemTag
{
    Input               = 0x8,
    Output              = 0x9,
    Feature             = 0xb,
    Collection          = 0xa,
    EndCollection       = 0xc,
};

enum GlobalItemTag
{
    UsagePage           = 0x0,
    LogicalMinimum      = 0x1,
    LogicalMaximum      = 0x2,
    PhysicalMinimum     = 0x3,
    PhysicalMaximum     = 0x4,
    UnitExponent        = 0x5,
    Unit                = 0x6,
    ReportSize          = 0x7,
    ReportId            = 0x8,
    ReportCount         = 0x9,
    Push                = 0xa,
    Pop                 = 0xb,
};

enum LocalItemTag
{
    Usage               = 0x0,
    UsageMinimum        = 0x1,
    UsageMaximum        = 0x2,
    DesignatorIndex     = 0x3,
    DesignatorMinimum   = 0x4,
    DesignatorMaximum   = 0x5,
    StringIndex         = 0x7,
    StringMinimum       = 0x8,
    StringMaximum       = 0x9,
    Delimiter           = 0xa,
};


enum GlobalItems
{
    ParamsCount         = 10,
    StackDepth          = 16,
};


class UsbHidCodeBook
{
    NN_DISALLOW_COPY(UsbHidCodeBook);
    NN_DISALLOW_MOVE(UsbHidCodeBook);

public:

    UsbHidCodeBook() NN_NOEXCEPT{};
    ~UsbHidCodeBook() NN_NOEXCEPT{};

    Result Initialize(uint8_t* pHidReportDescriptor, uint16_t length) NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;
    Result GetCodeBook(uint8_t* pBuffer, uint32_t bufferSize) NN_NOEXCEPT;

private:

    // current Codebook size in bytes
    int32_t     m_CodeBookBytes;

    // HID parser "Global Items" states
    int32_t     m_GlobalItemsStackDepth;
    int32_t     m_GlobalItems[StackDepth][ParamsCount];

    // HID parser "Local Items" states
    uint16_t    m_UsageReport;
    uint16_t    m_Usage[256];
    int32_t     m_UsageMinimum;
    int32_t     m_UsageMaximum;
    int32_t     m_DesignatorIndex;
    int32_t     m_DesignatorMinimum;
    int32_t     m_DesignatorMaximum;
    int32_t     m_StringIndex;
    int32_t     m_StringMinimum;
    int32_t     m_StringMaximum;

    int         m_UsageIndex;

    // HID parsor runtime
    int         m_StartingBit[256];

    // CodeBook encoding runtime
    bool        m_GenerateReportIds;
    uint8_t*    m_pCodeBookWrite;
    uint8_t     m_CodeBook[nn::ahid::CodeBook::CodeBook_Size];

    void ParseHidReportDescriptor(uint8_t* pHidReportDescriptor, uint16_t length, uint8_t tag) NN_NOEXCEPT;
    void ParseMainItemShort(uint8_t* pDataParams, uint8_t dataParams, uint8_t bSize, uint8_t tag) NN_NOEXCEPT;
    void ParseGlobalItemShort(uint8_t* pData, uint8_t dataParams, uint8_t bSize) NN_NOEXCEPT;
    void ParseLocalItemShort(uint8_t* pData, uint8_t dataParams, uint8_t bSize) NN_NOEXCEPT;
    int32_t GetData(uint8_t* pData, uint8_t bSize) NN_NOEXCEPT;

    int32_t GlobalItem(int index) NN_NOEXCEPT;
    int32_t StartingBit() NN_NOEXCEPT;
    void ClearGlobalItems() NN_NOEXCEPT;
    void ClearLocalItems() NN_NOEXCEPT;

    void CodeBookEntry(uint8_t bTag, uint8_t dataParams) NN_NOEXCEPT;
    void AddItemDataBit(uint16_t usageMin, uint16_t usageMax, uint8_t bTag, uint8_t dataParams, uint8_t index);
    void AddItemData8(uint16_t usageMin, uint16_t usageMax, uint8_t bTag, uint8_t dataParams, uint8_t index);
    void AddItemData16(uint16_t usageMin, uint16_t usageMax, uint8_t bTag, uint8_t dataParams, uint8_t index);
    void AddEntryToHeader(uint8_t bTag);
    void AddEntry(void* pData, uint8_t bytes, uint8_t bTag);

    void PrintGlobalItems() NN_NOEXCEPT;
    void PrintLocalItems() NN_NOEXCEPT;
};

} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
